package com.twentyfouri.tvlauncher.setupwizard.postcoresetup

import android.app.Activity
import android.content.Context
import android.view.View
import android.view.inputmethod.EditorInfo
import android.widget.Button
import android.widget.Toast
import androidx.annotation.IntRange
import androidx.lifecycle.MutableLiveData
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.error.GeneralApiException
import com.twentyfouri.smartmodel.model.error.InvalidCredentialsException
import com.twentyfouri.smartmodel.model.error.SuspendedException
import com.twentyfouri.smartmodel.model.error.WrongInputException
import com.twentyfouri.smartmodel.KtSmartApi
import com.twentyfouri.smartmodel.model.error.*
import com.twentyfouri.smartmodel.model.user.*
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.analytics.AnalyticsConstants
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.ui.TvLauncherToast
import com.twentyfouri.tvlauncher.setupwizard.Flavor
import com.twentyfouri.tvlauncher.setupwizard.LoginCredentials
import com.twentyfouri.tvlauncher.setupwizard.R
import com.twentyfouri.tvlauncher.setupwizard.basesetup.BaseSetupModel
import com.twentyfouri.tvlauncher.setupwizard.basesetup.BaseSetupView
import com.twentyfouri.tvlauncher.setupwizard.basesetup.BaseSetupViewModel
import com.twentyfouri.tvlauncher.setupwizard.postcoresetup.models.*
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.first
import java.util.*

class PostCoreSetupViewModel(
    private val model: PostCoreSetupModel,
    private val smartApi: FlowSmartApi?,
    private val resourceRepository: ResourceRepository,
    private val context: Context,
    private val analytics: YouboraAnalytics
) : BaseSetupViewModel() {

    var defaultText: String = ""
        set(value) {
            field = value.toUpperCase(Locale.ROOT)
            text.value = field
        }

    private var screens: List<Screens> = Flavor().usernameScreenList
    private var currentScreenPosition = 0
    private var currentScreen = screens[currentScreenPosition]


    // region Binded

    val titleRes = MutableLiveData<Int>()
    val descriptionRes = MutableLiveData<Int>()
    val text = MutableLiveData<String>()
    val buttonsVisibility = MutableLiveData<Int>()
    val editTextVisibility = MutableLiveData<Int>()
    val editTextEnabled = MutableLiveData<Boolean>().apply { value = true }
    val loginButtonsVisibility = MutableLiveData<Int>()
    val imeOptions = MutableLiveData<Int>()
    val inputType = MutableLiveData<Int>()
    val editLetterSpacing = MutableLiveData<Float>()
    val captchaVisibility = MutableLiveData<Int>()

    companion object{
        private const val UPDATE_PROFILE_FAILED = 1
        private const val AUTHENTICATE_FAILED = 2
        private const val INVALID_CREDENTIALS = 3
        private const val GENERAL_ERROR = 4
        private const val SUSPENDED_USER = 5
        private const val CONNECTION_ERROR = 6

        private const val NUMBER_OF_CAPTCHA_ATTEMPTS = 3
    }

    @IntRange(from = 1, to = NUMBER_OF_CAPTCHA_ATTEMPTS.toLong()) private var attempt: Int = 1

    val onActionDoneListener = object : OnActionDoneListener {
        override fun onActionDone() {
            when (currentScreen) {
                Screens.PasswordScreen -> {
                    if (!text.value.isNullOrEmpty()) {
                    model.password = text.value
                    val credentials = Flavor().getLoginCredentialsType(LoginCredentials(
                        null,
                        model.userName ?: "",
                        model.password ?:"")
                    )
                    login(credentials)
                    }
                }
                Screens.CaptchaScreen -> {
                    if (view?.getCaptchaResult().toString() == text.value ?: "") {
                        val credentials = Flavor().getLoginCredentialsType(LoginCredentials(
                            null,
                            model.userName ?: "",
                            model.password ?: "")
                        )
                        login(credentials)
                    }
                    else {
                        if (attempt == NUMBER_OF_CAPTCHA_ATTEMPTS) {
                            TvLauncherToast.makeText(
                                context,
                                R.string.toast_regeneration_captcha,
                                Toast.LENGTH_SHORT
                            )?.show()
                            view?.regenerateCaptcha()
                            text.value = ""
                            disableInput()
                            attempt = 1
                        }
                        else {
                            TvLauncherToast.makeText(
                                context,
                                context.getString(R.string.toast_remaining_attempts_captcha, (NUMBER_OF_CAPTCHA_ATTEMPTS - attempt).toString()),
                                Toast.LENGTH_SHORT
                            )?.show()
                            attempt++
                        }
                    }
                }
                else -> {
                    if (!text.value.isNullOrEmpty()) {
                        model.loginPin = text.value
                        val credentials = Flavor().getLoginCredentialsType(LoginCredentials(
                            model.loginPin,
                            model.userName,
                            null)
                        )
                        login(credentials)
                    }
                }
            }
        }

        override fun onActionNext() {
            when (currentScreen) {
                Screens.UsernameScreen -> {
                    if (!text.value.isNullOrEmpty()) {
                        model.userName = text.value
                        text.value = ""
                    }
                }
                Screens.PasswordScreen -> {
                    if (!text.value.isNullOrEmpty()) {
                        model.password = text.value
                        val credentials = Flavor().getLoginCredentialsType(LoginCredentials(
                            null,
                            model.userName ?: "",
                            model.password ?:"")
                        )
                    }
                }
                else -> {
                    if (!text.value.isNullOrEmpty()) {
                        model.loginPin = text.value
                    }
                    else {
                        return
                    }
                }
            }
            switchToNextScreen()
        }
    }

    @Suppress("UNUSED_PARAMETER")
    fun onYesButtonClick(v: View) {
        model.recommendations = true
        finishLogin()
    }

    @Suppress("UNUSED_PARAMETER")
    fun onNoButtonClick(v: View) {
        model.recommendations = false
        finishLogin()
    }

    @Suppress("UNUSED_PARAMETER")
    fun onPinButtonClick(v: View) {
        text.value = ""
        screens = Flavor().pinScreenList
        switchToNextScreen()
    }

    @Suppress("UNUSED_PARAMETER")
    fun onUsernameButtonClick(v: View) {
        screens = Flavor().usernameScreenList
        switchToNextScreen()
    }


    fun onFocusChange(v: View, hasFocus: Boolean) {
        when (v) {
            is Button -> view?.onFocusChange(v, hasFocus)
        }
    }
    // endregion

    init {
        setModel(Flavor().getScreenModel(currentScreen))
    }

    fun switchToPreviousScreen(): Boolean = if (currentScreenPosition > 0) {
        currentScreenPosition--
        currentScreen = screens[currentScreenPosition]
        setModel(Flavor().getScreenModel(currentScreen))
        view?.onScreenSwitched(currentScreen)
        true
    } else false

    private fun resetModel(){
        buttonsVisibility.value = View.GONE
        editTextVisibility.value = View.GONE
        loginButtonsVisibility.value = View.GONE

    }

    private fun setModel(model: BaseSetupModel) {
        resetModel()
        titleRes.value = model.titleRes
        descriptionRes.value = model.descriptionRes
        captchaVisibility.value = if (model.showCaptcha) View.VISIBLE else View.INVISIBLE

        when (model) {
            is InitialScreenModel -> { loginButtonsVisibility.value = View.VISIBLE }
            is PasswordScreenModel -> { editTextVisibility.value = View.VISIBLE }
            is PinScreenModel -> { editTextVisibility.value = View.VISIBLE }
            is RecommendationsScreenModel -> { buttonsVisibility.value = View.VISIBLE }
            is UsernameScreenModel -> { editTextVisibility.value = View.VISIBLE }
            is CaptchaScreenModel -> { editTextVisibility.value = View.VISIBLE }
        }

        if (titleRes.value == R.string.title_enter_username) {
            inputType.value = EditorInfo.TYPE_TEXT_FLAG_NO_SUGGESTIONS or EditorInfo.TYPE_TEXT_VARIATION_VISIBLE_PASSWORD
            text.value = this.model.userName ?: resourceRepository.getString(R.string.input_you) //"smart63@24i.com" "nikostest1@24i.com"
            editLetterSpacing.value = 0.0f
        }
        if (titleRes.value == R.string.title_enter_password) {
            inputType.value = EditorInfo.TYPE_CLASS_TEXT or EditorInfo.TYPE_TEXT_VARIATION_PASSWORD
            text.value = this.model.password ?: "" //"1234" "123456"
            editLetterSpacing.value = resourceRepository.getFloat(R.dimen.baseSetup_letterSpacing)
        }
        if (titleRes.value == R.string.title_enter_pin) {
            inputType.value = EditorInfo.TYPE_CLASS_NUMBER or EditorInfo.TYPE_NUMBER_VARIATION_PASSWORD
            editLetterSpacing.value = resourceRepository.getFloat(R.dimen.baseSetup_letterSpacing)
        }
        if (model is CaptchaScreenModel) {
            inputType.value = EditorInfo.TYPE_CLASS_NUMBER or EditorInfo.TYPE_NUMBER_VARIATION_NORMAL
            text.value = this.model.captchaResult?.toString() ?: ""
            editLetterSpacing.value = 0.0f
            regenerateCaptcha()
        }

        imeOptions.value = Flavor().getImeAction(currentScreen, screens)
    }

    private fun regenerateCaptcha() {
        attempt = 1
        view?.regenerateCaptcha()
    }

    private fun disableInput() {
        MainScope().launch {
            text.value = ""
            editTextEnabled.value = false
            delay(5000)
            text.value = ""
            editTextEnabled.value = true
        }
    }

    private fun isLastScreenShown() = currentScreen == screens.last()

    private fun switchToNextScreen() {
        if (!isLastScreenShown()) {
            currentScreenPosition++
            currentScreen = screens[currentScreenPosition]
            setModel(Flavor().getScreenModel(currentScreen))
            view?.onScreenSwitched(currentScreen)
        } else {
//            val credentials = Flavor().getLoginCredentialsType(LoginCredentials(
//                model.loginPin,
//                model.userName,
//                null)
//            )
//            login(credentials)
            finishLogin()
        }
    }

    private fun login(credentials: SmartLoginCredentials) {
        CoroutineScope(Dispatchers.Main).launch {
            try {
                val login = smartApi?.login(credentials)?.first()
                Flavor().processLogin(credentials)
                login.let {
                    if (it?.sessionState == SmartSessionState.LOGGED_IN) {
                        switchToNextScreen()
                    } else {
                        view?.showMessageDialog(null, getErrorMessage(AUTHENTICATE_FAILED), resourceRepository.getString(R.string.button_back))
                    }
                }
            } catch (e: Exception) {
                when (e) {
                    is SuspendedException -> view?.showMessageDialog(resourceRepository.getString(R.string.user_suspended_error_title), getErrorMessage(SUSPENDED_USER, e, e.code), resourceRepository.getString(R.string.dialog_error_retry)) {login(credentials)}
                    is GeneralApiException, // Keep these here as we may later need different handling for more general errors
                    is WrongInputException, // Keep these here as we may later need different handling for more general errors
                    is InvalidCredentialsException -> view?.showInvalidCredentialsDialog(e as GeneralApiException)
                    is InternetConnectionException -> view?.showMessageDialog(resourceRepository.getString(R.string.authenticate_failed), getErrorMessage(CONNECTION_ERROR, e),  resourceRepository.getString(R.string.button_back))
                    else -> view?.showMessageDialog(null, getErrorMessage(GENERAL_ERROR, e), resourceRepository.getString(R.string.button_back))
                }
            }
        }
    }

    private fun BaseSetupView.showInvalidCredentialsDialog(e: GeneralApiException) {
        analytics.reportApiException(e, e.code)
        showMessageDialog(null, getErrorMessage(INVALID_CREDENTIALS, e, e.code), resourceRepository.getString(R.string.button_back))
    }

    private fun finishLogin() {
        CoroutineScope(Dispatchers.IO).launch {
            try {
                val updateProfileResult = smartApi?.updateProfile(
                    SmartUserProfileChange(Flavor().getUserProfileReference(model.userName ?: ""))
                        .apply { extras = Flavor().gdprProfileChangeObject(model.recommendations)
                        }
                )?.first()
                if (updateProfileResult != null && Flavor().isGdprAcceptedProfile(updateProfileResult.extras, model.recommendations))  {
                    view?.onFinish()
                }
            } catch (e: Exception) {
                view?.onFinish()
            }
        }
    }

    private fun getErrorMessage(type: Int, e: Exception? = null, code: String = ""): String {
        val s = resourceRepository.getString(R.string.dialog_message_wrong_credentials)
        return when (type) {
            UPDATE_PROFILE_FAILED -> resourceRepository.getString(R.string.update_profile_failed)
            AUTHENTICATE_FAILED -> resourceRepository.getString(R.string.authenticate_failed)
            INVALID_CREDENTIALS -> Flavor().processErrorMessage(code,resourceRepository)
            GENERAL_ERROR -> "${resourceRepository.getString(R.string.unknown_error)}${getErrorMessage(e)}"
            SUSPENDED_USER -> resourceRepository.getString(R.string.user_suspended_error, code)
            CONNECTION_ERROR -> context.getString(R.string.connection_exception)
            else -> s
        }
    }

    private fun getErrorMessage(e: Exception?): String {
        if (e == null)
            return ""
        if (e.cause == null)
            return ":\n${e.message}"
        return ":\n${e.message}\n${e.cause.toString()}"
    }

    fun onCleanStart(activity: Activity?){
        analytics.setActivity(activity)
        analytics.beginSession()
        analytics.reportOpenScreen(AnalyticsConstants.SCREEN_LOGIN)
    }
}
