package com.twentyfouri.searchproxy.resourcegen

import java.io.File

/**
 * Abstraction for src-directories, allowing reading and writing of
 * string resources inside flavor-specific directories.
 *
 * "Language" in here refers to alternative "values" files.
 * The term "language" is not entirely accurate, but it is for the
 * purpose of this tool.
 */
class ResourceDirs(
    val dir: File
) {
    private val stringResourceCache = HashMap<String, StringResources>()

    /**
     * Read all the strings for a given variant/language combination.
     */
    fun readStrings(variant: String = "main", lang: String = ""): StringResources {
        val variantDir = getVariantDir(variant, lang)
        return stringResourceCache.computeIfAbsent(variantDir.absolutePath) {
            readStringsDirectory(File(it))
        }
    }

    /**
     * Get a list of all the languages for a given variant/language combination.
     */
    fun getLanguages(variant: String = "main"): Set<String> {
        val resDir = dir.toPath()
            .resolve(variant)
            .resolve("res")
            .toFile()

        val list = resDir.listFiles()?.filter { file ->
            file.isDirectory && file.name.startsWith("values-")
        }?.map { file ->
            file.name.removePrefix("values-")
        } ?: emptyList()
        return list.toSet()
    }

    /**
     * Write a strings-generated.xml file for the given variant/language
     * combination.
     *
     * If the variant/language combination directory does not contain a
     * strings.xml file, an empty dummy file will be added.
     * This is needed due to a quirk in the build process, where only having a
     * strings-generated.xml in a values directory will cause duplicate-resource
     * errors.
     */
    fun writeStrings(variant: String, strings: StringResources, lang: String = "") {
        val generatedDest = getVariantDir(variant, lang).toPath()
            .resolve("strings-generated.xml").toFile()
        writeStringsToFile(
            strings = strings,
            dest = generatedDest,
            comment = "Generated automatically using searchproxy-resource-generator"
        )

        val dummyDest = getVariantDir(variant, lang).toPath()
            .resolve("strings.xml").toFile()
        if (!dummyDest.exists()) {
            writeStringsToFile(
                strings = StringResources(),
                dest = dummyDest,
                comment = "Empty resources file to prevent false-positive duplicate resource errors during build"
            )
        }
    }

    private fun getVariantDir(variant: String, lang: String = ""): File {
        var valuesDir = "values"
        if (lang.isNotEmpty()) {
           valuesDir += "-${lang}"
        }

        return dir.toPath()
            .resolve(variant)
            .resolve("res")
            .resolve(valuesDir)
            .toFile()
    }

    private fun writeStringsToFile(strings: StringResources, dest: File, comment: String? = null) {
        val doc = createNewXmlDocument()
        if (comment != null) {
            doc.appendChild(doc.createComment(comment))
        }
        val root = doc.appendChild(doc.createElement("resources"))

        strings.map { name, value, translatable ->
            val element = doc.createElement("string")
            element.setAttribute("name", name)
            element.textContent = value
            if (translatable != null) {
                element.setAttribute("translatable", translatable.toString())
            }
            element
        }.forEach { root.appendChild(it) }

        dest.parentFile.mkdirs()
        doc.writeToFile(dest)
    }

    /**
     * Read all the xml files in a given directory and combine their
     * string resoures into one StringResources.
     */
    private fun readStringsDirectory(valuesDir: File):StringResources {
        val result = StringResources()

        val files = valuesDir.listFiles() ?: emptyArray()

        files.filter { file ->
            file.isFile && file.extension == "xml"
        }.forEach { file ->
            result.add(readStringsFile(file))
        }

        return result
    }

    /**
     * Read all the string resources in an xml file into a StringResources
     */
    private fun readStringsFile(file: File): StringResources {
        val result = StringResources()
        val document = file.readXmlDocument()

        val stringNodes = document.getElementsByTagName("resources")
            .toElementsList().flatMap { node ->
                node.getElementsByTagName("string").toElementsList()
            }
        stringNodes.forEach { node ->
            result.add(
                name = node.getAttribute("name"),
                value = node.textContent,
                translatable = node.getAttributeOrNull("translatable")?.toBoolean()
            )
        }

        return result
    }
}