package com.twentyfouri.tvlauncher.common

import android.content.ContentValues
import android.content.Context
import android.content.pm.PackageInfo
import android.net.Uri
import android.os.Build
import com.google.gson.Gson
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.caching.LowMemoryHandler
import com.twentyfouri.smartmodel.epg.EpgDatabase
import com.twentyfouri.smartmodel.model.error.GeneralApiException
import com.twentyfouri.smartmodel.model.user.SmartSavedSession
import com.twentyfouri.smartmodel.model.user.SmartSessionPersistence
import com.twentyfouri.smartmodel.model.user.SmartUserProfileReference
import com.twentyfouri.smartmodel.serialization.SmartDataObject
import com.twentyfouri.smartmodel.smartvideo.SmartVideoFlowSmartApi
import com.twentyfouri.smartmodel.smartvideo.SmartVideoKtSmartApi
import com.twentyfouri.smartmodel.smartvideo.caching.SmartVideoSavedSession
import com.twentyfouri.smartmodel.smartvideo.configuration.KabelnoordConfiguration
import com.twentyfouri.smartmodel.smartvideo.configuration.SmartVideoConfiguration
import com.twentyfouri.smartmodel.smartvideo.configuration.YoufoneConfiguration
import com.twentyfouri.smartmodel.smartvideo.reference.SmartVideoUserProfileReference
import com.twentyfouri.smartvideo.data.SmartVideoException
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.data.SetupDataRepository
import com.twentyfouri.tvlauncher.common.extensions.ifNotNull
import com.twentyfouri.tvlauncher.common.utils.DeviceId
import com.twentyfouri.tvlauncher.common.utils.DeviceInfoConfig
import com.twentyfouri.tvlauncher.common.utils.OselToggableLogger
import com.twentyfouri.tvlauncher.common.utils.DeviceProperties
import com.twentyfouri.tvlauncher.common.utils.ReflectionUtils
import kotlinx.coroutines.*
import okhttp3.logging.HttpLoggingInterceptor
import timber.log.Timber
import javax.net.ssl.HttpsURLConnection

class Flavor : FlavorBase() {

    override fun getUsername(sessionData: String?): String? {
        val data = Gson().fromJson(sessionData, SmartVideoSavedSession::class.java)
        return data?.username
    }

    override fun getSmartApi(
        applicationContext: Context?,
        setupDataRepo: SetupDataRepository,
        oselLogger: OselToggableLogger?
    ): FlowSmartApi {
        applicationContext ?: throw IllegalStateException("Context null when creating SmartApi")
        return buildSmartVideoSmartApi(
            getYoufoneConfiguration(applicationContext),
            applicationContext,
            setupDataRepo)
    }

    private fun buildSmartVideoSmartApi(
        configuration: SmartVideoConfiguration,
        context:Context,
        setupDataRepo: SetupDataRepository
    ) = SmartVideoFlowSmartApi (
            SmartVideoKtSmartApi(
        smartVideoConfiguration = configuration,
        sessionPersistence = object : SmartSessionPersistence {
            override fun loadSession(): SmartSavedSession? = runBlocking {
                Timber.tag("SessionStorage").d("SmartSessionPersistence loadSession")
                val setupData = setupDataRepo.getSetupDataObject()
                Timber.tag("SessionStorage").d("SmartSessionPersistence loadSession json: ${setupData?.sessionData}")
                val sessionFromProvider = getSavedSession(setupData?.sessionData)
                sessionFromProvider.ifNotNull { return@runBlocking sessionFromProvider }

                // try shared preferences. This was used for storage by previous versions, so this can be
                // deleted when we are way over build 81 and we are sure we will never get installed over
                // version older than 81
                //TODO delete some day
                val sp = context.getSharedPreferences(SHARED_PREFERENCES_FILE, Context.MODE_PRIVATE)
                val json = sp.getString(SHARED_PREFERENCES_JSON, null)
                Timber.tag("SessionStorage").d("SmartSessionPersistence loadSession json: $json")
                val sessionFromSharedPreferences = getSavedSession(json)
                sessionFromSharedPreferences.ifNotNull { return@runBlocking sessionFromSharedPreferences }

                return@runBlocking null
            }
            override fun saveSession(session: SmartSavedSession?) {
                val json: String? = session?.let { Gson().toJson(it) }
                val values = ContentValues()
                val uri: Uri = Uri.parse(context.getString(R.string.credentials_uri))
                values.put("sessionData",json)
                context.contentResolver.insert(uri, values)
            }
        },
        lowMemoryHandler = LowMemoryHandler(context),
        epgDao = EpgDatabase.getInstance(context).epgDao
    )
    )

    @Suppress("SpellCheckingInspection", "unused")
    private fun getKabelnoordConfiguration(applicationContext: Context?) = KabelnoordConfiguration(
        endpoint = ENDPOINT_KABELNOORD,
        deviceId = applicationContext?.let { DeviceId.getDeviceId(it) } ?: DEFAULT_DEVICE_ID,
        serviceName = SERVICE,
        servicePassword = SERVICE_PASSWORD,
        serviceToken = SERVICE_TOKEN,
        platform = PLATFORM,
        osVersion = Build.VERSION.RELEASE,
        appVersion = "BuildConfig.VERSION_NAME",
        epgExpirationInDays = epgExpirationDays
        )

    @Suppress("SpellCheckingInspection")
    private fun getYoufoneConfiguration(applicationContext: Context?) = YoufoneConfiguration(
            endpoint = ENDPOINT_YOUFONE,
            deviceId = applicationContext?.let { DeviceId.getDeviceId(it) } ?: DEFAULT_DEVICE_ID,
            platform = PLATFORM,
            osVersion = Build.VERSION.RELEASE,
            appVersion = getVersionName(applicationContext),
            epgExpirationInDays = epgExpirationDays,
            onNowProgramsPlusMinusHours = 2,
            hoursToCoverEdgesOfEpgWindow = 4,
            epgWindowMinusDays = EPG_DAYS_INTO_PAST,
            epgWindowPlusDays = EPG_DAYS_INTO_FUTURE,
            seriesRecordingsEnabled = this@Flavor.seriesRecordingsEnabled,
            httpLoggingInterceptor = HttpLoggingInterceptor().apply {
                level = if (BuildConfig.IS_DEBUG) HttpLoggingInterceptor.Level.HEADERS else HttpLoggingInterceptor.Level.BASIC
            },
            senderId = ReflectionUtils.getPackageInfo()?.let { "${it.versionName}(${it.versionCode})" }
    )

//    override val epgExpirationDays = 1

    private fun getVersionName(applicationContext: Context?): String {
        applicationContext ?: return "BuildConfig.VERSION_NAME"
        val pInfo: PackageInfo = applicationContext.packageManager.getPackageInfo(applicationContext.packageName, 0);
        return pInfo.versionName
    }

    override suspend fun performPing(ktSmartApi: FlowSmartApi) {
        require(ktSmartApi is SmartVideoFlowSmartApi) { "Wrong type of KtSmartApi" }
        ktSmartApi.ktSmartApi.loadServerTime(true)
    }

    override fun clearSessionData(context: Context) {}

    override suspend fun clearEpgData(context: Context?, flowSmartApi: FlowSmartApi?) {
        context?.let {
            EpgDatabase.getInstance(it).epgDao.apply {
                deleteAllEpgEvents()
                deleteAllEpgHoles()
                deleteAllPlaylistItems()
                deleteAllEpgChannels()
            }
        }
    }

    override fun getSavedSession(setupDataString: String?): SmartSavedSession? {
        val extras = Gson().fromJson(setupDataString, SmartVideoSavedSession::class.java)
        extras ?: return null
        return extras
    }

    override fun getUserProfileReference(userName: String): SmartUserProfileReference {
        return SmartVideoUserProfileReference(userName)
    }

    override fun gdprProfileChangeObject(isGdrpAccepted: Boolean?): SmartDataObject {
        //TODO
        return SmartDataObject()
    }

    override fun parentalControlProfileChangeObject(isParentalControlActivated: Boolean?): SmartDataObject {
        return SmartDataObject(PARENTAL_CONTROL_EXTRA_KEY to isParentalControlActivated?.not())
    }

    override fun isGdprAcceptedProfile(extras: SmartDataObject): Boolean? {
        //TODO
        return false
    }

    override fun isParentalControlActivatedProfile(extras: SmartDataObject): Boolean {
        return !extras[PARENTAL_CONTROL_EXTRA_KEY].getBoolean(false)
    }

    override fun showDialogs() = false

    override fun handleProjectSpecificException(exception: Exception): Boolean {
        when {
            exception is SmartVideoException && exception.httpCode == HttpsURLConnection.HTTP_NOT_FOUND -> {
                //We should ignore 404 errors
                Log.i("Exception", "Ignored exception ${exception.httpCode}, ${exception.fieldMessages}")
                return true
            }
            exception is ClassCastException
                    && exception.message?.contains("HardcodedPlaylistReference") == true
                    && exception.message?.contains("SmartVideoPlaylistReference") == true -> {
                //TODO solve issue with java.lang.ClassCastException: com.twentyfouri.tvlauncher.data.HardcodedPlaylistReference cannot be cast to com.twentyfouri.smartmodel.smartvideo.reference.SmartVideoPlaylistReference
                Log.i("Exception", "Ignored exception ${exception.message}")
                return true
            }
        }
        return false
    }

    override fun getErrorMessage(e:GeneralApiException, applicationContext: Context): String {
        val resourceRepository = ResourceRepository(applicationContext)
        return "${resourceRepository.getString(R.string.backend_error)}: ${e.code} ${e.serverMessage}"
    }

    override suspend fun verifyPurchasePin(smartApi: FlowSmartApi, purchasePin: String) = Unit

    override val seriesRecordingsEnabled = true

    override val deviceInfoConfig = DeviceInfoConfig("com.twentyfouri.youfonelauncher", "com.twentyfouri.setupwizard.youfone")

    override val firmwareVersion = GlobalScope.async(Dispatchers.IO, CoroutineStart.LAZY) {
        DeviceProperties().getProperty("ro.amino.build.version") ?: ""
    }

    companion object {
        private const val SHARED_PREFERENCES_FILE = "savedSession"
        private const val SHARED_PREFERENCES_TYPE = "type"
        private const val SHARED_PREFERENCES_JSON = "data"
        @Suppress("SpellCheckingInspection")
        private const val ENDPOINT_KABELNOORD = "https://media-api-1.kabelnoord.net/"
        private const val ENDPOINT_YOUFONE = "https://tv.youfone.services/"
        @Suppress("SpellCheckingInspection")
        private const val SERVICE = "kabelnoord_production"
        private const val SERVICE_PASSWORD = "JeHdauFmWv74dKn"
        private const val SERVICE_TOKEN = "561aa29a09517db775b12db9dba925f2"
        private const val DEFAULT_DEVICE_ID = "b1cc65240617e8ac99a3ef083a4b9213"
        private const val PLATFORM = "android_tv"
        private const val PARENTAL_CONTROL_EXTRA_KEY = "parentalControl"

        const val EPG_DAYS_INTO_PAST = 7
        const val EPG_DAYS_INTO_FUTURE = 6
    }
}