package com.twentyfouri.tvlauncher.common.widgets

import android.content.Context
import android.graphics.Canvas
import android.graphics.Paint
import android.text.InputType
import android.util.AttributeSet
import android.view.*
import android.view.inputmethod.InputMethodManager
import android.widget.EditText
import androidx.core.content.ContextCompat.getColor
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.R

class PinEntryView : androidx.appcompat.widget.AppCompatEditText {

    private var mSpace = 10f
    private var mCharSize = 0f
    private var mNumChars = 4f
    private var mLineSpacing = 5f
    private val XML_NAMESPACE_ANDROID = "http://schemas.android.com/apk/res/android"
    private var mLineStroke = 1f
    private lateinit var mLinesPaint: Paint
    private var mClickListener: View.OnClickListener? = null
    private var mOnPinEnteredListener: OnPinEnteredListener? = null

    private val focusedBlockTopOffset = -1.5f
    private val focusedBlockBottomOffset = 1.5f
    private val focusedBlockRightOffset = 9f


    private fun updateColorForLines(next: Boolean) {
        if (isFocused) {
            mLinesPaint.color = getColor(context, android.R.color.transparent)
            if (next) {
                mLinesPaint.color = getColor(context, R.color.edit_text_background)
            }
        } else {
            mLinesPaint.color = getColor(context, android.R.color.transparent)
        }
    }

    constructor(context: Context) : super(context)

    constructor(context: Context, attrs: AttributeSet) : super(context, attrs) {
        init(context, attrs)
    }

    constructor(
        context: Context, attrs: AttributeSet,
        defStyleAttr: Int
    ) : super(context, attrs, defStyleAttr) {
        init(context, attrs)
    }

    private fun init(context: Context, attrs: AttributeSet) = apply {
        val multi = context.resources.displayMetrics.density
        inputType = InputType.TYPE_CLASS_NUMBER
        mLineStroke *= multi
        mLinesPaint = Paint(paint)
        mLinesPaint.strokeWidth = mLineStroke
        mSpace *= multi
        mLineSpacing *= multi

        val mMaxLength = context.resources.getInteger(R.integer.pin_length)

        paint.color = getColor(context, android.R.color.white)

        mNumChars = mMaxLength.toFloat()

        super.setCustomSelectionActionModeCallback(
            object : ActionMode.Callback {
                override fun onPrepareActionMode(
                    mode: ActionMode,
                    menu: Menu
                ) = false

                override fun onDestroyActionMode(mode: ActionMode) {}

                override fun onCreateActionMode(
                    mode: ActionMode,
                    menu: Menu
                ) = false

                override fun onActionItemClicked(
                    mode: ActionMode,
                    item: MenuItem
                ) = false
            })

        super.setOnClickListener { v ->
            if (Flavor().pinDialogWithSoftInput()) {
                val imm = context.getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
                imm.toggleSoftInputFromWindow(windowToken, 0, 0)
            }
            setSelection(text?.length ?: 0)
            if (mClickListener != null) {
                mClickListener?.onClick(v)
            }
        }
    }

    override fun setOnClickListener(l: OnClickListener?) {
        mClickListener = l
    }

    override fun onDraw(canvas: Canvas) {
        val availableWidth = width - paddingRight - paddingLeft
        mCharSize = if (mSpace < 0) {
            (availableWidth / (mNumChars * 2 - 1))
        } else {
            (availableWidth - mSpace * (mNumChars - 1)) / mNumChars
        }

        var startX = paddingLeft.toFloat()
        var startX2 = paddingLeft.toFloat()
        val bottom = height.toFloat() - paddingBottom.toFloat()

        val text = text
        val textLength = text?.length ?: 0
        val textWidths = FloatArray(textLength)
        paint.getTextWidths(getText(), 0, textLength, textWidths)


        for (j in 0 until mNumChars.toInt()) { //drawing dots
            if (text != null && j < text.length)
                paint.color = getColor(context, android.R.color.transparent)
            else
                paint.color = getColor(context, android.R.color.white)

            val middle = startX2 + mCharSize / 2
            canvas.drawText(
                "•",
                0,
                1,
                middle - (mNumChars.toInt()-j) / 2,
                bottom - mLineSpacing - 1f,
                paint
            )
            startX2 += if (mSpace < 0) {
                mCharSize * 2
            } else {
                mCharSize + mSpace
            }
        }

        paint.color = getColor(context, android.R.color.white)
        for (i in 0 until mNumChars.toInt()) {
            updateColorForLines(i == textLength)
            canvas.drawRoundRect( //drawing selection background
                startX,
                focusedBlockTopOffset,
                startX + mCharSize + focusedBlockRightOffset,
                height.toFloat() + focusedBlockBottomOffset,
                5f,
                5f,
                mLinesPaint
            )

            if (text != null && text.length > i) {
                val middle = startX + mCharSize / 2
                canvas.drawText( //drawing pin mask
                    "*",
                    0,
                    1,
                    middle - textWidths[0] / 2,
                    bottom - mLineSpacing,
                    paint
                )
            }
            startX += if (mSpace < 0) {
                mCharSize * 2
            } else {
                mCharSize + mSpace
            }
        }
    }

    override fun onTextChanged(text: CharSequence, start: Int, lengthBefore: Int, lengthAfter: Int) {
        if (mOnPinEnteredListener != null && text.length.toFloat() == mNumChars) {
                mOnPinEnteredListener?.onPinEntered(this, text)
        }
    }

    override fun onKeyDown(keyCode: Int, event: KeyEvent?): Boolean {
        return if (event != null && (
                                event.keyCode == KeyEvent.KEYCODE_CHANNEL_UP ||
                                event.keyCode == KeyEvent.KEYCODE_CHANNEL_DOWN ||
                                event.scanCode == 91 ||
                                event.scanCode == 90 ||
                                event.keyCode == KeyEvent.KEYCODE_BACK)) {
            var code = event.keyCode
            if (event.scanCode == 91) code = KeyEvent.KEYCODE_CHANNEL_DOWN
            if (event.scanCode == 90) code = KeyEvent.KEYCODE_CHANNEL_UP
            mOnPinEnteredListener?.onChannelChangedBehind(code)
            true
        } else {
            super.onKeyDown(keyCode, event)
        }
    }

    fun setOnPinEnteredListener(l: OnPinEnteredListener) {
        mOnPinEnteredListener = l
    }

    interface OnPinEnteredListener {
        fun onPinEntered(v: View, str: CharSequence) {
            v.findViewById<PinEntryView>(R.id.message_dialog_pin_entry).text = null
        }
        fun onChannelChangedBehind(keyCode: Int)
    }
}