package com.twentyfouri.tvlauncher.common.utils.logging

import android.Manifest
import android.app.Activity
import android.app.Application
import android.content.Context
import android.content.pm.PackageManager
import android.os.Bundle
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.ContextCompat
import androidx.lifecycle.Lifecycle
import com.twentyfouri.tvlauncher.common.extensions.addEventObserver

/**
 * Utility for automatically requesting storage permissions when attempting to
 * activate remote logging through [OselToggleableLogger].
 *
 * This registers an automatic permission requester on each new activity opened in the app.
 * Whenever logging is enabled in [OselToggleableLogger], the currently-resumed activity may
 * be used to request storage permissions.
 */
class OselToggleableLoggerPermissions(
    private val context: Context
) {
    private val openActivities = arrayListOf<ActivityEntry>()

    var onPermissionRequestComplete: ((PermissionRequestResult) -> Unit)? = null

    /**
     * Whether or not permissions are requested
     */
    val hasPermissions: Boolean
        get() = REQUIRED_PERMISSIONS.all {
            context.hasPermission(it)
        }

    init {
        val application = context.applicationContext as Application
        application.registerActivityLifecycleCallbacks(object : Application.ActivityLifecycleCallbacks {
            override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
                (activity as? AppCompatActivity)?.let(::onNewActivity)
            }
            override fun onActivityStarted(activity: Activity) {}
            override fun onActivityResumed(activity: Activity) {}
            override fun onActivityPaused(activity: Activity) {}
            override fun onActivityStopped(activity: Activity) {}
            override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {}
            override fun onActivityDestroyed(activity: Activity) {}
        })
    }

    fun requestPermissionFromUser(): Boolean {
        val entry = openActivities.lastOrNull()
        return if (entry != null) {
            val ungranted = REQUIRED_PERMISSIONS.filterNot {
                context.hasPermission(it)
            }
            if (ungranted.isEmpty()) {
                onPermissionRequestComplete?.invoke(PermissionRequestResult.GRANTED)
            } else {
                entry.isRequestOpen = true
                entry.launcher.launch(ungranted.toTypedArray())
            }
            true
        } else {
            return false
        }
    }

    private fun onNewActivity(activity: AppCompatActivity) {
        val activityEntry = ActivityEntry()

        activity.activityResultRegistry
        activityEntry.launcher = activity.registerForActivityResult(
            ActivityResultContracts.RequestMultiplePermissions()
        ) {
            activityEntry.isRequestOpen = false

            if (hasPermissions) {
                onPermissionRequestComplete?.invoke(PermissionRequestResult.GRANTED)
            } else {
                onPermissionRequestComplete?.invoke(PermissionRequestResult.DENIED)
            }
        }

        activity.lifecycle.addEventObserver { _, event ->
            when (event) {
                Lifecycle.Event.ON_RESUME -> {
                    openActivities.add(activityEntry)
                }
                Lifecycle.Event.ON_PAUSE -> {
                    openActivities.remove(activityEntry)
                }
                Lifecycle.Event.ON_DESTROY -> {
                    // With some luck, this should catch most request-never-answered cases
                    if (activityEntry.isRequestOpen && !hasPermissions) {
                        onPermissionRequestComplete?.invoke(PermissionRequestResult.ACTIVITY_CLOSED)
                    }
                }
                else -> {}
            }
        }
    }

    private fun Context.hasPermission(permission: String): Boolean =
        ContextCompat.checkSelfPermission(this, permission) == PackageManager.PERMISSION_GRANTED

    enum class PermissionRequestResult {
        GRANTED, DENIED, ACTIVITY_CLOSED
    }

    private class ActivityEntry(
        var isRequestOpen: Boolean = false
    ) {
        lateinit var launcher: ActivityResultLauncher<Array<String>>
    }

    companion object {
        private val REQUIRED_PERMISSIONS = listOf(
            Manifest.permission.READ_EXTERNAL_STORAGE,
            Manifest.permission.WRITE_EXTERNAL_STORAGE
        )
    }
}