package com.twentyfouri.tvlauncher.common.utils

import android.content.Context
import android.net.ConnectivityManager
import android.net.LinkProperties
import android.net.Network
import android.net.NetworkInfo
import android.net.RouteInfo
import android.net.wifi.WifiConfiguration
import android.net.wifi.WifiInfo
import android.net.wifi.WifiManager
import androidx.core.content.getSystemService
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.R
import com.twentyfouri.tvlauncher.common.ui.SemaphoreState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.joda.time.DateTime
import timber.log.Timber
import java.io.BufferedInputStream
import java.io.BufferedReader
import java.io.FileNotFoundException
import java.io.InputStream
import java.io.InputStreamReader
import java.net.Inet4Address
import java.net.Inet6Address
import java.net.InetAddress
import java.net.NetworkInterface
import java.net.URL
import java.util.Collections
import java.util.Timer
import java.util.*
import kotlin.concurrent.fixedRateTimer

class NetworkData {
    var privateIPs: ArrayList<String> = arrayListOf()
    var privateIPsAssignment = Assignment.UNASSIGNED
    var dnsIPs: ArrayList<String> = arrayListOf()
    var gateway: ArrayList<String> = arrayListOf()
    var proxy: String = ""
    var publicIP: String = ""
    var displayName: String = ""
    var interfaceName: String = ""
    var default: Boolean = false
    var state = NetworkInfo.DetailedState.DISCONNECTED
    var mac = ""
    var localSpeed = ""
    var ssid: String = "" //empty for ethernet
    var signalStrength: String = "" //empty for ethernet
    var signalFrequency: String = "" //empty for ethernet

    fun toString(context: Context): String {
        val result = StringBuilder()
        result.append("${context.getString(R.string.di_network)} $displayName ($interfaceName)")
        if (default) result.append(" ${context.getString(R.string.di_default)}")
        result.appendln()
        val dsResource = DetailedState.values().find { state.name == it.name }?.resId
        val dsString = if (dsResource != null) context.getString(dsResource) else "?"
        result.appendln("${context.getString(R.string.di_state)} $dsString")
        if (mac.isNotEmpty()) result.appendln("${context.getString(R.string.di_mac)} $mac")
        if (ssid.isNotEmpty()) result.appendln("${context.getString(R.string.di_ssid)} $ssid")
        if (signalStrength.isNotEmpty()) result.appendln("${context.getString(R.string.di_sig_str)} $signalStrength")
        if (signalFrequency.isNotEmpty()) result.appendln("${context.getString(R.string.di_sig_freq)} $signalFrequency")
        result.appendln(privateIPs.joinToString(separator = "\n"))
        if (privateIPsAssignment != Assignment.UNASSIGNED) result.appendln("${context.getString(R.string.di_assignment)} ${context.getString(privateIPsAssignment.resId)}")
        result.appendln(dnsIPs.joinToString(separator = "\n") { "${context.getString(R.string.di_dns)}${dnsIPs.indexOf(it) + 1}: $it"})
        if (gateway.isNotEmpty()) result.appendln("${context.getString(R.string.di_gateway)}" +
                " ${gateway[0] + if (gateway.size > 1) " (+${gateway.size - 1} ${context.getString(R.string.di_more)})" else ""}")
        if (proxy.isNotEmpty()) result.appendln("${context.getString(R.string.di_proxy)} $proxy")
        if (publicIP.isNotEmpty()) result.appendln("${context.getString(R.string.di_public_ip)} $publicIP")
        if (localSpeed.isNotEmpty()) result.appendln("${context.getString(R.string.di_intf_speed)} $localSpeed")
        return result.toString()
    }
}

class NetworkInfo(context: Context, val infoUpdated: InfoUpdated) {

    var ethernet: NetworkData? = null
    var wifi: NetworkData? = null
    var connToGateway = ""
    var connToGoogle = ""
    var connToBackend = ""

    private var repeatTimer: Timer? = null


    init {
        CoroutineScope(Dispatchers.Default).launch { update(context) }
    }

    fun startAutoUpdate(context: Context) {
        repeatTimer?.cancel()
        repeatTimer = null
        repeatTimer = fixedRateTimer("DeviceInfoTimer", true, 15000, 15000) {
            CoroutineScope(Dispatchers.Default).launch {
                update(context)
            }
        }
    }

    fun stopAutoUpdate() {
        repeatTimer?.cancel()
        repeatTimer = null
    }

    fun toString(context: Context): String {
        val result = StringBuilder()
        result.appendln("${context.getString(R.string.di_conn_gateway)} ${connToGateway}")
        result.appendln("${context.getString(R.string.di_conn_internet)} ${connToGoogle}")
        result.append("${context.getString(R.string.di_conn_backend)} ${connToBackend}")
        return result.toString()
    }

    suspend fun update(context: Context) {
        val connectivityManager = context.getSystemService<ConnectivityManager>()!!
        connToGateway = context.getString(R.string.di_conn_pending)
        connToGoogle = context.getString(R.string.di_conn_pending)
        connToBackend = context.getString(R.string.di_conn_pending)

        ethernet = NetworkData().apply {
            displayName = context.getString(R.string.di_ethernet)
            interfaceName = INTERFACE_ETHERNET
            mac = getMacAddress(INTERFACE_ETHERNET)
            privateIPs = getIPsArrayFromNetworkInterface(INTERFACE_ETHERNET)
        }

        wifi = NetworkData().apply {
            displayName = context.getString(R.string.di_wifi)
            interfaceName = INTERFACE_WIFI
            mac = getMacAddress(INTERFACE_WIFI)
            privateIPs = getIPsArrayFromNetworkInterface(INTERFACE_WIFI)
        }

        val defaultNetwork = connectivityManager.activeNetwork
        var defaultNetworkHaveProxy = false
        val networks = connectivityManager.allNetworks
        for (network in networks) {
            val networkInfo = connectivityManager.getNetworkInfo(network)
            val networkCapabilities = connectivityManager.getNetworkCapabilities(network)
            val props = connectivityManager.getLinkProperties(network)
            //ETHERNET
            if (props?.interfaceName == INTERFACE_ETHERNET) {
                ethernet?.proxy = props.httpProxy?.let { it.host + ":" + it.port } ?: ""
                ethernet?.privateIPs = getIPsArray(props.linkAddresses.map { it.address })
                ethernet?.dnsIPs = getIPsArray(props.dnsServers, false)
                ethernet?.gateway = getGatewaysArrayFromRoutes(props.routes, context)
                ethernet?.default = defaultNetwork == network
                if (defaultNetwork == network && ethernet?.proxy?.isNotEmpty() == true) defaultNetworkHaveProxy = true
                networkInfo?.detailedState?.let { ethernet?.state = it }
                networkCapabilities?.linkDownstreamBandwidthKbps?.let { ethernet?.localSpeed = "${it}Kbps" }
                ethernet?.publicIP = context.getString(R.string.di_public_ip_pending)
                CoroutineScope(Dispatchers.IO).launch {
                    ethernet?.publicIP = getPublicIpFromNetwork(network, context)
                    CoroutineScope(Dispatchers.Main).launch {
                        infoUpdated.onInfoUpdated()
                    }
                }
            }

            //WIFI
            if (props?.interfaceName == INTERFACE_WIFI) {
                val wifiManager = context.getSystemService<WifiManager>()!!
                val connectionInfo = wifiManager.connectionInfo
                val wifiConfiguration = wifiManager.configuredNetworks.find { it.networkId == connectionInfo.networkId }
                wifi?.proxy = props.httpProxy?.let { it.host + ":" + it.port } ?: ""
                wifi?.privateIPs = getIPsArray(props.linkAddresses.map { it.address })
                wifi?.dnsIPs = getIPsArray(props.dnsServers, false)
                wifi?.gateway = getGatewaysArrayFromRoutes(props.routes, context)
                wifi?.default = defaultNetwork == network
                if (defaultNetwork == network && wifi?.proxy?.isNotEmpty() == true) defaultNetworkHaveProxy = true
                networkInfo?.detailedState?.let { wifi?.state = it }
                wifi?.localSpeed = "${connectionInfo.linkSpeed}${WifiInfo.LINK_SPEED_UNITS}"
                wifi?.signalStrength = "${WifiManager.calculateSignalLevel(connectionInfo.rssi, 100)}% (RSSI: ${connectionInfo.rssi})"
                wifi?.signalFrequency = "${getFrequencyClass(connectionInfo.frequency)} (${connectionInfo.frequency}${WifiInfo.FREQUENCY_UNITS})"
                wifi?.ssid = getSsidFromWifiConfiguration(wifiConfiguration, context)
                if (wifi?.ssid.isNullOrBlank()) wifi?.ssid = ""
                wifi?.privateIPsAssignment = getIpAssignmentFromWifiConfiguration(wifiConfiguration)
                wifi?.publicIP = context.getString(R.string.di_public_ip_pending)
                wifi?.publicIP = getPublicIpFromNetwork(network, context)
                withContext(Dispatchers.Main) {
                    infoUpdated.onInfoUpdated()
                }
            }
        }

        val defaultGateway = when {
            ethernet?.default == true -> ethernet?.gateway
            wifi?.default == true -> wifi?.gateway
            else -> arrayListOf()
        }

        if (!defaultGateway.isNullOrEmpty()) {
            Probe(
                host = defaultGateway[0],
                timeoutAll = NetworkConnectionState.PROBE_TIMEOUT_ALL * 2,
                timeoutSingle = NetworkConnectionState.PROBE_TIMEOUT_SINGLE * 2,
                resultInterface = { ip1success, ip1time ->
                    if (ip1success) {
                        connToGateway = "${context.getString(R.string.di_conn_ok)} (${ip1time}ms)"
                    } else {
                        if (defaultGateway.size > 1) {
                            Probe(
                                host = defaultGateway[1],
                                timeoutAll = NetworkConnectionState.PROBE_TIMEOUT_ALL * 2,
                                timeoutSingle = NetworkConnectionState.PROBE_TIMEOUT_SINGLE * 2,
                                resultInterface = { ip2success, ip2time ->
                                    connToGateway = if (ip2success) {
                                        "${context.getString(R.string.di_conn_ok)} (${ip2time}ms)"
                                    } else {
                                        "${context.getString(R.string.di_conn_failed)} (${ip2time}ms)"
                                    }
                                })
                        } else {
                            connToGateway = context.getString(R.string.di_conn_failed)
                        }
                    }
                    CoroutineScope(Dispatchers.Main).launch {
                        infoUpdated.onInfoUpdated()
                    }
                })
        } else {
            connToGateway = context.getString(R.string.di_conn_failed)
        }

        Probe(NetworkConnectionState.PROBE_HOST, NetworkConnectionState.PROBE_TIMEOUT_ALL*2, NetworkConnectionState.PROBE_TIMEOUT_SINGLE*2) { success, time ->
            connToGoogle = if (success) "${context.getString(R.string.di_conn_ok)} (${time}ms)" else "${context.getString(R.string.di_conn_failed)} (${time}ms)"
            CoroutineScope(Dispatchers.Main).launch {
                infoUpdated.onInfoUpdated()
            }
        }

        val overProxy = if (defaultNetworkHaveProxy) " (${context.getString(R.string.di_conn_over_proxy)})" else ""
        val start = DateTime.now().millis
        try {
            val conn = defaultNetwork?.openConnection(URL(BACKEND_TEST_URL)) ?: throw IllegalStateException()
            conn.connectTimeout = 3000
            val content: InputStream = BufferedInputStream(conn.getInputStream())
            val bufferedReader = BufferedReader(InputStreamReader(content))
            val res = bufferedReader.readText() //result is unused but need to be read
            content.close()
            //any response is fine because it means server is reachable
            connToBackend = "${context.getString(R.string.di_conn_ok)}$overProxy (${DateTime.now().millis - start}ms)"
        } catch (e: FileNotFoundException) {
            // in fact this is 404 response from server, it is also ok fur us because server replied
            connToBackend = "${context.getString(R.string.di_conn_ok)}$overProxy (${DateTime.now().millis - start}ms)"
        } catch (e: Exception) {
            Timber.tag("BACKENDCONN").d("failed. ${e.message}")
            connToBackend = "${context.getString(R.string.di_conn_failed)}$overProxy (${DateTime.now().millis - start}ms)"
        }
        withContext(Dispatchers.Main) {
            infoUpdated.onInfoUpdated()
        }
        infoUpdated.onInfoUpdated()
    }

    private fun getPublicIpFromNetwork(network: Network, context: Context): String {
        try {
            val conn = network.openConnection(URL(PUBLIC_IP_SERVICE))
            conn.connectTimeout = 2000
            val content: InputStream = BufferedInputStream(conn.getInputStream())
            val bufferedReader = BufferedReader(InputStreamReader(content))
            val res = bufferedReader.readText()
            content.close()
            Timber.tag("PUBLICIP").d(res)
            return res
        } catch (e: Exception) {
            Timber.tag("PUBLICIP").d("Getting public IP failed. ${e.message}")
            return context.getString(R.string.di_public_ip_not_available)
        }
    }

    private fun getIPsArray(adresses: List<InetAddress>, prefixVersion: Boolean = true): ArrayList<String> {
        val result = arrayListOf<String>()
        val v4Prefix = if (prefixVersion) "IPv4: " else ""
        val v6Prefix = if (prefixVersion) "IPv6: " else ""
        adresses.filterIsInstance<Inet4Address>().forEach {
            result.add(v4Prefix + it.hostAddress)
        }
        adresses.filterIsInstance<Inet6Address>().forEach {
            result.add(v6Prefix + it.hostAddress)
        }
        return result
    }

    private fun getGatewaysArrayFromRoutes(routes: List<RouteInfo>, context: Context): ArrayList<String> {
        val gateway = routes.filter { it.gateway?.isAnyLocalAddress != true }
        return if (gateway.isNotEmpty()) {
            ArrayList(gateway.map { it.gateway?.hostAddress ?: ""})
        } else {
            arrayListOf()
        }
    }

    private fun getIPsArrayFromNetworkInterface(interfaceName: String): ArrayList<String> {
        val nif = NetworkInterface.getByName(interfaceName)
        nif ?: return ArrayList()
        return getIPsArray(nif.inetAddresses?.toList() ?: emptyList())
    }

    private fun getFrequencyClass(frequency: Int): String {
        return when (frequency) {
            in WIFI_24GHZ_RANGE -> "2.4GHz"
            in WIFI_5GHZ_RANGE -> "5GHz"
            else -> ""
        }
    }

    private fun getGatewayFromRoutes(routes: List<RouteInfo>, context: Context): String {
        val gateways = routes.filter { it.gateway?.isAnyLocalAddress == false }
        return if (gateways.isNotEmpty()) {
            //gateway should be only one, we will get the first one and if there is more than one indicate it in brackets
            gateways[0].gateway!!.hostAddress + if (gateways.size > 1) " (+${gateways.size - 1} ${context.getString(R.string.di_more)})" else ""
        } else {
            ""
        }
    }

    private fun getSsidFromWifiConfiguration(conf: WifiConfiguration?, context: Context): String {
        conf ?: return ""
        return conf.SSID + if (conf.hiddenSSID) " ${context.getString(R.string.di_hidden)}" else ""
    }

    private fun getIpAssignmentFromWifiConfiguration(conf: WifiConfiguration?): Assignment {
        conf ?: return Assignment.UNASSIGNED
        val assignment = ReflectionUtils.getIpAssignment(ReflectionUtils.getIpConfiguration(conf)).toString()
        return Assignment.values().find { it.toString() == assignment } ?: Assignment.UNASSIGNED
    }

    private fun getMacAddress(interfaceName: String): String {
        val lines = Runtime.getRuntime().exec("ip address").inputStream.bufferedReader().readText().lines()
        val index = lines.indexOfFirst { it.contains("$interfaceName:", true) }
        if (index == -1 || index + 1 >= lines.size) return ""
        //mac address is written on the next line
        lines.getOrNull(index + 1)?.split(" ")?.let { words ->
            words.find { word -> word.count { char -> char == ':' } == 5 }?.let { mac -> //MAC have five colons
                //MAC found so return it
                return mac.uppercase()
            }
        }
        return ""
    }

    fun getNetworkInterfaces(): String {
        val all = Collections.list(NetworkInterface.getNetworkInterfaces())
        return all.joinToString(separator = ", ") { it.name }
    }

    fun logNetworkProperties(props: LinkProperties) {
        Timber.tag("NETDATA").d("DNS: " + props.dnsServers.joinToString(separator = ", ") { it.hostAddress })
        Timber.tag("NETDATA").d("DOMAIN: " + props.domains)
        Timber.tag("NETDATA").d("NAME: " + props.interfaceName)
        Timber.tag("NETDATA").d("PROXY: " + props.httpProxy?.host + ":" + props.httpProxy?.port)
        Timber.tag("NETDATA").d("LINK: " + props.linkAddresses.joinToString(separator = ", ") { it.address.toString() })
        Timber.tag("NETDATA").d("ROUTE: " + props.routes.joinToString(separator = ", ") { it?.gateway?.hostAddress.toString() })
    }

    companion object {
        private const val PUBLIC_IP_SERVICE = "http://api.ipify.org"
        private val BACKEND_TEST_URL = Flavor().backendConnectionTestUrl
        private const val INTERFACE_ETHERNET = "eth0"
        private const val INTERFACE_WIFI = "wlan0"
        private val WIFI_5GHZ_RANGE = 4900..5900
        private val WIFI_24GHZ_RANGE = 2400..2500

        fun getActiveConnectionInfo(context: Context?): ActiveConnectionInfo {
            if (context != null) {
                val connectivityManager = context.getSystemService<ConnectivityManager>()!!
                connectivityManager.activeNetwork?.let { network ->
                    connectivityManager.getLinkProperties(network)?.let { props ->
                        return when (props.interfaceName) {
                            INTERFACE_ETHERNET -> ActiveConnectionInfo(InterfaceType.LAN)
                            INTERFACE_WIFI -> {
                                val wifiInfo = context.getSystemService<WifiManager>()!!.connectionInfo
                                ActiveConnectionInfoWifi(
                                    wifiInfo.rssi,
                                    wifiInfo.frequency in WIFI_5GHZ_RANGE,
                                    wifiInfo.ssid,
                                    wifiInfo.frequency
                                )
                            }
                            else -> ActiveConnectionInfo(InterfaceType.OTHER)
                        }

                    }
                } ?: return ActiveConnectionInfo(InterfaceType.NONE)
            }
            return ActiveConnectionInfo(InterfaceType.NONE)
        }

        fun isConnectionOK(activeConnectionInfo: ActiveConnectionInfo): Boolean {
            return when {
                activeConnectionInfo.type == InterfaceType.LAN -> true //LAN is always OK
                activeConnectionInfo is ActiveConnectionInfoWifi && activeConnectionInfo.is5GHz -> { //this is WIFI 5GHz
                    when (activeConnectionInfo.rssi) {
                        in WiFiSignalQualityRange.GREEN.rssiRange -> true // -57 and more is OK
                        else -> false // less than -57 is BAD
                    }
                }
                activeConnectionInfo is ActiveConnectionInfoWifi && !activeConnectionInfo.is5GHz -> false //this is WIFI 2.4GHz which is always BAD
                else -> false //another type of connection like USB or mobile data which is always BAD
            }
        }

        fun getCurrentNetworkQualityState(activeConnectionInfo: ActiveConnectionInfo): SemaphoreState = when {
            activeConnectionInfo.type == InterfaceType.LAN -> SemaphoreState.GREEN_LAN
            activeConnectionInfo is ActiveConnectionInfoWifi -> {
                when (activeConnectionInfo.rssi) {
                    in WiFiSignalQualityRange.GREEN.rssiRange -> if (activeConnectionInfo.is5GHz) SemaphoreState.GREEN_WIFI else SemaphoreState.ORANGE_WIFI
                    in WiFiSignalQualityRange.ORANGE.rssiRange -> SemaphoreState.ORANGE_WIFI
                    else -> SemaphoreState.RED_WIFI
                }
            }
            else -> SemaphoreState.RED_WIFI
        }.also {
            Timber.tag(OSELConnectivity.TAG).d("$it")
        }

        fun getConnectionSemaphoreState(): SemaphoreState = SharedPreferencesUtils.getConnectionSemaphoreState()

        fun saveApprovedNetwork(context: Context?) {
            if (Flavor().useSafeListOfApprovedNetworks && context != null) {
                val connectivityManager = context.getSystemService<ConnectivityManager>()!!
                connectivityManager.activeNetwork?.let { network ->
                    connectivityManager.getLinkProperties(network)?.let { props ->
                         when (props.interfaceName) {
                            INTERFACE_ETHERNET -> {} // do nothing
                            INTERFACE_WIFI -> {
                                val wifiInfo = context.getSystemService<WifiManager>()!!.connectionInfo
                                val currentApprovedList = SharedPreferencesUtils.getApprovedNetworks()?.toMutableSet()
                                    ?: mutableSetOf()
                                if (wifiInfo.networkId.toString() !in currentApprovedList) {
                                    currentApprovedList.add(wifiInfo.networkId.toString())
                                    SharedPreferencesUtils.putApprovedNetworks(currentApprovedList)
                                }
                            }
                            else -> {} // do nothing
                        }

                    }
                }
            }
        }

    }
}

open class ActiveConnectionInfo(
        val type: InterfaceType
)

class ActiveConnectionInfoWifi(
        val rssi: Int,
        val is5GHz: Boolean,
        val ssid: String,
        val frequency: Int
): ActiveConnectionInfo(InterfaceType.WIFI)

enum class InterfaceType {
    WIFI,
    LAN,
    OTHER,
    NONE //used when no connected interface
}

enum class DetailedState(val resId: Int) {
    IDLE(R.string.ds_idle),
    SCANNING(R.string.ds_scanning),
    CONNECTING(R.string.ds_connecting),
    AUTHENTICATING(R.string.ds_authenticating),
    OBTAINING_IPADDR(R.string.ds_obtaining_ipaddr),
    CONNECTED(R.string.ds_connected),
    SUSPENDED(R.string.ds_suspended),
    DISCONNECTING(R.string.ds_disconnecting),
    DISCONNECTED(R.string.ds_disconnected),
    FAILED(R.string.ds_failed),
    BLOCKED(R.string.ds_blocked),
    VERIFYING_POOR_LINK(R.string.ds_verifying_poor_link),
    CAPTIVE_PORTAL_CHECK(R.string.ds_captive_portal_check)
}

enum class Assignment(val resId: Int) {
    DHCP(R.string.ipa_dhcp),
    STATIC(R.string.ipa_static),
    UNASSIGNED(R.string.ipa_unassigned)
}

enum class WiFiSignalQualityRange(val rssiRange: IntRange){
    GREEN(-57..Int.MAX_VALUE),
    ORANGE(-70..-58),
    RED(Int.MIN_VALUE..-71)
}