package com.twentyfouri.tvlauncher.common.utils

import android.annotation.SuppressLint
import android.content.Context
import android.os.Build
import android.util.Log
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.receiver.SerialNumberReceiver
import timber.log.Timber
import java.net.NetworkInterface
import java.util.*

object DeviceId {

    @Suppress("NULLABILITY_MISMATCH_BASED_ON_JAVA_ANNOTATIONS")
    fun getDeviceId(context: Context): String {
//        return "deviceid709"
        return if (Flavor().getSerialNumberFromBroadcast()) {
            getSerialNumberFromSharedPreferences(context).let {
                if(it.isNullOrEmpty()) {
                    Timber.e("Serial number should be stored from broadcast but it is NOT!")
                    getBackupId()
                } else {
                    it
                }
            }
        } else {
            getBackupId()
        }
    }

    private fun getSerialNumberFromSharedPreferences(context: Context): String? {
        return context.getSharedPreferences(SerialNumberReceiver.SP_FILE, Context.MODE_PRIVATE).getString(SerialNumberReceiver.SP_ID, "")
    }

    fun isSerialNumberStored(context: Context): Boolean {
        return !getSerialNumberFromSharedPreferences(context).isNullOrEmpty()
    }

    // getting serial number via SystemProperties, recommended and functional for amino STBs
    @SuppressLint("PrivateApi")
    fun getDeviceIdViaProperties(): String {
        var deviceId = ""
        try {
            val systemProperties = Class.forName("android.os.SystemProperties")
            val getMethod = systemProperties.getMethod("get", String::class.java)
            deviceId = getMethod.invoke(systemProperties, "persist.sys.hwblk.sn").toString()
        } catch (e: java.lang.Exception) {
            e.printStackTrace()
        }
        return deviceId
    }

    @SuppressLint("MissingPermission", "HardwareIds")
    @Suppress("DEPRECATION")
    private fun getBackupId(): String {
        val ethernetMac = getMacAddress()
        if (ethernetMac.isNotEmpty()) return ethernetMac.replace(":", "")
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            Build.getSerial()
        } else {
            Build.SERIAL
        }
    }

    private fun getMacAddress(): String {
        try {
            val all = Collections.list(NetworkInterface.getNetworkInterfaces())
            for (nif in all) {
                if (!nif.name.equals("eth0", ignoreCase = true)) continue

                val macBytes = nif.hardwareAddress ?: return ""

                val res1 = StringBuilder()
                for (b in macBytes) {
                    res1.append(String.format("%02X:", b))
                }

                if (res1.isNotEmpty()) res1.deleteCharAt(res1.length - 1)
                return res1.toString()
            }
        } catch (ex: Exception) {
            //handle exception
        }

        return ""
    }

}