package com.twentyfouri.tvlauncher.common.ui.fading

import android.util.ArraySet
import android.view.View
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import kotlin.concurrent.fixedRateTimer

/*
 *  Controller is responsible for animating alpha value and set it to all views. By this all views are animated together.
 *
 *  Views can be grouped by adding FadingControllerView into layout and settings same id in "fadingController". (see FadingControllerView)
 *  In case view do not have set "fadingController" it will create its own controller and will be animated separately.
 *
 *  When any animated view need to update its state it should call "update()" on its controller. Controller then start animate alpha towards 0 (fade OUT).
 *  Even if currently fading IN direction of animation is switched to fade OUT. So multiple updates from different views just causing fade OUT.
 *
 *  Possible enhancements
 *  - configurable animation step
 *  - configurable timer tick period
 */
class FadingController {

    private val animatedViews = ArraySet<FadingViewInterface>()
    private var shouldFadeOut: Boolean = false
    private var shouldFadeIn: Boolean = false
    private var detached: Boolean = false
    private var animatedAlpha = 1f
    private var animationStep = 0.1f

    fun update() {
        if(detached) return
        if(!shouldFadeOut) animateOut()
    }

    //detach stop animations and no more action will be performed, ready to be destroyed
    fun detach() {
        animatedViews.clear()
        detached = true
    }

    fun addView(v: FadingViewInterface) {
        animatedViews.add(v)
        detached = false
    }

    fun removeView(v: FadingViewInterface) {
        animatedViews.remove(v)
        if (animatedViews.isEmpty()) detached = true
    }

    fun removeAndAddAllViews(list: ArrayList<FadingViewInterface>) {
        animatedViews.clear()
        animatedViews.addAll(list)
    }

    private fun informOnFadedOut() {
        GlobalScope.launch(Dispatchers.Main) {
            animatedViews.forEach {
                it.onFadedOut()
            }
        }
    }

    private fun informOnFadeOutStarted() {
        GlobalScope.launch(Dispatchers.Main) {
            animatedViews.forEach {
                it.onFadeOutStarted()
            }
        }
    }

    private fun informOnFadeInFinished() {
        GlobalScope.launch(Dispatchers.Main) {
            animatedViews.forEach {
                it.onFadeInFinished()
            }
        }
    }

    private fun updateAlpha() {
        GlobalScope.launch(Dispatchers.Main) {
            animatedViews.forEach {
                (it as? View)?.alpha = animatedAlpha
            }
        }
    }

    private fun animateOut() {
        shouldFadeIn = false
        shouldFadeOut = true
        informOnFadeOutStarted()
        fixedRateTimer("animateOut", true, 0, 20) {
            if(detached) {
                cancel()
                return@fixedRateTimer
            }
            if(animatedAlpha <= 0 || !shouldFadeOut) {
                cancel()
                informOnFadedOut()
                animateIn()
            }
            animatedAlpha -= animationStep
            updateAlpha()
        }
    }

    private fun animateIn() {
        shouldFadeIn = true
        shouldFadeOut = false
        fixedRateTimer("animateIn", true, 0, 20) {
            if(detached) {
                cancel()
                return@fixedRateTimer
            }
            if(animatedAlpha >= 1 || !shouldFadeIn) {
                cancel()
                informOnFadeInFinished()
            }
            animatedAlpha += animationStep
            updateAlpha()
        }
    }

}