package com.twentyfouri.tvlauncher.common.data.apihandler

import android.os.Handler
import android.os.Looper
import android.util.Log
import android.view.KeyEvent
import android.widget.Toast
import com.twentyfouri.smartmodel.model.error.GeneralApiException
import com.twentyfouri.smartmodel.model.error.NotFoundException
import com.twentyfouri.smartmodel.model.error.SuspendedException
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.R
import com.twentyfouri.tvlauncher.common.data.apihandler.ApiHandler.Companion.TAG
import com.twentyfouri.tvlauncher.common.ui.TvLauncherToast
import com.twentyfouri.tvlauncher.common.ui.messagedialog.*
import com.twentyfouri.tvlauncher.common.utils.PurchaseChecker

internal fun ApiHandler.catchGeneralApiExceptionWithDialog(e: GeneralApiException) {
    if (isForegroundPlaybackActive) {
        Log.i(TAG,"Ignored exception: ${e.message}")
        return
    }
    val messageDialogModel = MessageDialogModel(
        Flavor().getErrorMessage(e,appContext),
        null,
        arrayOf(resourceRepository.getString(R.string.error_retry)),
        resourceRepository.getString(R.string.button_back),
        "${MessageDialogCodes.generalApiException}; S ${e.code}"
    )
    val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
    dialogFragment.mListener = MessageDialogFragmentListener.from(this::onDefaultErrorDialogResult)
    navigatorCommon.navigateDialog(dialogFragment)
}

private val ApiHandler.isForegroundPlaybackActive
    get() = mainActivityAction?.isForegroundPlaybackActive() == true

internal fun ApiHandler.catchSuspendedExceptionWithDialog(e: SuspendedException) {
    val messageDialogModel = MessageDialogModel(
        resourceRepository.getString(R.string.user_suspended_error_title),
        resourceRepository.getString(R.string.user_suspended_error, e.code),
        arrayOf(resourceRepository.getString(R.string.dialog_error_retry)),
        if (Flavor().allowCancelableDialog(e)) resourceRepository.getString(R.string.button_back) else null,
        MessageDialogCodes.userSuspended
    )
    val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
    dialogFragment.isCancelable = Flavor().allowCancelableDialog(e)
    dialogFragment.mListener = MessageDialogFragmentListener.from(this::onDefaultErrorDialogResult)
    navigatorCommon.navigateDialog(dialogFragment)
}

internal fun ApiHandler.catchOtherApiExceptionWithDialog(messageResId: Int, code: String ="") {
    val messageDialogModel = MessageDialogModel(
        resourceRepository.getString(messageResId,code),
        null,
        resourceRepository.getString(R.string.button_back),
        MessageDialogCodes.unprocessedApiException
    )
    val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
    dialogFragment.mListener = MessageDialogFragmentListener.from(this::onDefaultErrorDialogResult)
    navigatorCommon.navigateDialog(dialogFragment)
}

private fun MessageDialogAction.isOptionA() = this is MessageDialogAction.Result && type == OPTION_A

private fun MessageDialogAction.isOptionB() = this is MessageDialogAction.Result && type == OPTION_B

private fun MessageDialogAction.isOptionCancel() = this is MessageDialogAction.Result && type == CANCEL

private fun MessageDialogAction.isOptionBack() = this is MessageDialogAction.Event && keyEvent.keyCode == KeyEvent.KEYCODE_BACK

internal fun ApiHandler.catchUnsubscribedExceptionWithDialog(messageResId: Int, code: String ="") {
    PurchaseChecker.dismissPinDialog()
    val messageDialogModel = MessageDialogModel(
        resourceRepository.getString(messageResId,code),
        null,
        resourceRepository.getString(R.string.button_back),
        MessageDialogCodes.missingSubscription
    )
    val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
    dialogFragment.mDismissListener = MessageDialogDismissListener.from{ dialogFragment.backFromSubscriptionDialog() }
    navigatorCommon.navigateDialog(dialogFragment)
}

internal fun ApiHandler.catchLoginFailureExceptionWithDialog(messageResId: Int) {
    val messageDialogModel = MessageDialogModel(
        resourceRepository.getString(messageResId),
        null,
        arrayOf(resourceRepository.getString(R.string.authentication_failure_dialog_login)),
        resourceRepository.getString(R.string.authentication_failure_dialog_restart),
        MessageDialogCodes.loginFailure
    )
    val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
    val reboot = mainActivityAction?.isLauncher() ?: false
    if (reboot)
        dialogFragment.mListener = MessageDialogFragmentListener.from(this::onLoginErrorDialogResultWithReboot)
    else
        dialogFragment.mListener = MessageDialogFragmentListener.from(this::onLoginErrorDialogResultWithRestart)
    dialogFragment.isCancelable = false
    navigatorCommon.navigateDialog(dialogFragment)
}

internal fun ApiHandler.catchAppCannotRunExceptionWithDialog(messageResId: Int) {
    val messageDialogModel = MessageDialogModel(
        resourceRepository.getString(messageResId),
        null,
        null,
        resourceRepository.getString(R.string.authentication_failure_dialog_restart),
        MessageDialogCodes.appCannotRun
    )
    val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
    dialogFragment.mListener = MessageDialogFragmentListener.from(this::onLoginErrorDialogResultWithReboot)
    dialogFragment.isCancelable = false
    navigatorCommon.navigateDialog(dialogFragment)
}

internal fun ApiHandler.catchNotFoundExceptionWithDialog(e: NotFoundException) {
    if (Flavor().shouldIgnoreNotFoundException) {
        Log.i(TAG,"Ignored exception: ${e.message}")
        return
    }
    if (isForegroundPlaybackActive) {
        Log.i(TAG,"Ignored exception: ${e.message}")
        return
    }
    val messageDialogModel = MessageDialogModel(
        Flavor().getErrorMessage(e,appContext),
        null,
        arrayOf(resourceRepository.getString(R.string.authentication_failure_dialog_login)),
        resourceRepository.getString(R.string.authentication_failure_dialog_restart),
        MessageDialogCodes.notFound
    )
    val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
    dialogFragment.mListener = MessageDialogFragmentListener.from(this::onLoginErrorDialogResultWithReboot)
    dialogFragment.isCancelable = false
    navigatorCommon.navigateDialog(dialogFragment)
}

internal fun ApiHandler.catchInvalidSessionExceptionWithDialog() {
    val dialogOptionButtons = mutableListOf(resourceRepository.getString(R.string.error_retry))
    if (Flavor().offerLoginIfInvalidSession) dialogOptionButtons.add(
        resourceRepository.getString(R.string.authentication_failure_dialog_login)
    )
    val messageDialogModel = MessageDialogModel(
        resourceRepository.getString(R.string.authentication_failure_invalid_session),
        null,
        dialogOptionButtons.toTypedArray(),
        resourceRepository.getString(R.string.authentication_failure_dialog_offline),
        MessageDialogCodes.loginFailure
    )
    val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
    dialogFragment.mListener = MessageDialogFragmentListener.from(this::onInvalidSessionException)
    dialogFragment.isCancelable = true
    navigatorCommon.navigateDialog(dialogFragment)
}

internal fun ApiHandler.catchMissingSessionExceptionWithDialog() {
    val messageDialogModel = if (Flavor().offerOfflineIfMissingSession) {
        MessageDialogModel(
            resourceRepository.getString(R.string.authentication_failure_missing_session),
            null,
            arrayOf(resourceRepository.getString(R.string.authentication_failure_dialog_login)),
            resourceRepository.getString(R.string.authentication_failure_dialog_offline),
            MessageDialogCodes.loginFailure
        )
    } else {
        MessageDialogModel(
            resourceRepository.getString(R.string.authentication_failure_missing_session),
            null,
            resourceRepository.getString(R.string.authentication_failure_dialog_login),
            MessageDialogCodes.loginFailure
        )
    }
    val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
    dialogFragment.mListener = MessageDialogFragmentListener.from(this::onMissingSessionException)
    dialogFragment.mDismissListener = MessageDialogDismissListener.from {  }
    dialogFragment.isCancelable = false
    navigatorCommon.navigateDialog(dialogFragment)
}

internal fun ApiHandler.catchServiceForbiddenWithDialog(messageResId: Int) {
    val messageDialogModel = MessageDialogModel(
        resourceRepository.getString(messageResId),
        null,
        arrayOf(resourceRepository.getString(R.string.authentication_failure_dialog_restart)),
        null,
        MessageDialogCodes.serviceForbidden
    )
    val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
    dialogFragment.mListener = MessageDialogFragmentListener.from{
        restart()
        true
    }
    dialogFragment.isCancelable = false
    navigatorCommon.navigateDialog(dialogFragment)
}

internal fun ApiHandler.catchExceptionWithToast(message: String){
    if (isForegroundPlaybackActive) {
        Log.i(TAG,"Ignored exception: $message")
        return
    }
    Handler(Looper.getMainLooper()).post{
        TvLauncherToast.makeText(
            appContext,
            message,
            Toast.LENGTH_LONG
        )?.show()
    }
    Log.d(TAG, message)
}

private fun ApiHandler.onDefaultErrorDialogResult(answer: MessageDialogAction): Boolean {
    when {
        answer.isOptionA() -> retry()
        answer.isOptionCancel() -> {} //do nothing
        answer.isOptionBack() -> {} //do nothing
        else -> {} //do nothing
    }
    return true //key will be consumed
}

private fun ApiHandler.onLoginErrorDialogResultWithReboot(answer: MessageDialogAction): Boolean {
    when {
        answer.isOptionA() -> openLogin()
        answer.isOptionCancel() -> reboot()
        answer.isOptionBack() -> reboot()
        else -> reboot()
    }
    return true //key will be consumed
}

private fun ApiHandler.onLoginErrorDialogResultWithRestart(answer: MessageDialogAction): Boolean {
    when {
        answer.isOptionA() -> openLogin()
        answer.isOptionCancel() -> restart()
        answer.isOptionBack() -> restart()
        else -> restart()
    }
    return true //key will be consumed
}

private fun ApiHandler.onInvalidSessionException(answer: MessageDialogAction): Boolean {
    when {
        answer.isOptionA() -> restartToOnline()
        answer.isOptionB() -> openLogin()
        answer.isOptionCancel() -> restartToOffline()
        answer.isOptionBack() -> {} //do nothing
        else -> {} //do nothing
    }
    return true //key will be consumed
}

private fun ApiHandler.onMissingSessionException(answer: MessageDialogAction): Boolean {
    when {
        answer.isOptionA() -> openLogin()
        answer.isOptionCancel() -> restartToOffline()
        answer.isOptionBack() -> {} //do nothing
        else -> {} //do nothing
    }
    return true //key will be consumed
}