package com.twentyfouri.tvlauncher.common.analytics

import com.google.android.exoplayer2.ExoPlaybackException
import com.google.android.exoplayer2.Player
import com.google.android.exoplayer2.SimpleExoPlayer
import com.google.android.exoplayer2.trackselection.MappingTrackSelector
import com.google.android.exoplayer2.upstream.HttpDataSource
import com.npaw.youbora.lib6.YouboraUtil
import com.npaw.youbora.lib6.exoplayer2.CustomEventLogger
import com.npaw.youbora.lib6.exoplayer2.Exoplayer2Adapter
import com.twentyfouri.smartexoplayer.RealSmartPlayer

class YouboraAdapter(player: Player) : Exoplayer2Adapter(player) {

    private var customEventLogger: CustomEventLogger? = null
    private var customEventLoggerEnabled = false

    private fun allowReportStopPlayback() = (plugin as? YouboraPlugin)?.allowReportStopPlayback == true

    // Handled internally
    override fun onPlayerError(error: ExoPlaybackException) {}
    
    fun onPlayerErrorInternal(error: ExoPlaybackException, isFatal: Boolean){
        val classError = error.cause?.javaClass?.name
        when (error.type) {
            ExoPlaybackException.TYPE_SOURCE -> {
                val errorMessage = "${error.message}, ${error.sourceException.cause ?: error.cause}"
                when (error.sourceException) {
                    is HttpDataSource.InvalidResponseCodeException -> {
                        val invalidResponseCodeException = error
                            .sourceException as HttpDataSource.InvalidResponseCodeException
                        fireErrorInternal(
                            code = classError,
                            msg = errorMessage,
                            errorMetadata = "Response message: ${invalidResponseCodeException.responseMessage}\n${error.stackTraceToString()}",
                            isFatal = isFatal
                        )
                    }

                    is HttpDataSource.HttpDataSourceException -> {
                        val httpDataSourceException = error
                            .sourceException as HttpDataSource.HttpDataSourceException

                        when (httpDataSourceException.type) {
                            HttpDataSource.HttpDataSourceException.TYPE_OPEN -> fireErrorInternal(
                                code = classError,
                                msg = "OPEN - $errorMessage",
                                errorMetadata = error.stackTraceToString(),
                                isFatal = isFatal
                            )
                            HttpDataSource.HttpDataSourceException.TYPE_READ -> fireErrorInternal(
                                code = classError,
                                msg = "READ - $errorMessage",
                                errorMetadata = error.stackTraceToString(),
                                isFatal = isFatal
                            )
                            HttpDataSource.HttpDataSourceException.TYPE_CLOSE -> fireErrorInternal(
                                code = classError,
                                msg = "CLOSE - $errorMessage",
                                errorMetadata = error.stackTraceToString(),
                                isFatal = isFatal
                            )
                        }
                    }
                    else -> fireErrorInternal(
                        code = classError,
                        msg = errorMessage,
                        errorMetadata = error.stackTraceToString(),
                        isFatal = isFatal
                    )
                }
            }
            ExoPlaybackException.TYPE_RENDERER -> {
                val errorMessage = "${error.message}, ${error.rendererException.cause ?: error.cause}, ${error.rendererName}, ${error.rendererFormat}"
                fireErrorInternal(
                    code = classError,
                    msg = errorMessage,
                    errorMetadata = error.stackTraceToString(),
                    isFatal = isFatal
                )
            }
            ExoPlaybackException.TYPE_REMOTE -> {
                val errorMessage = "${error.message}, ${error.cause}"
                fireErrorInternal(
                    code = classError,
                    msg = errorMessage,
                    errorMetadata = error.stackTraceToString(),
                    isFatal = isFatal
                )
            }
            ExoPlaybackException.TYPE_UNEXPECTED -> {
                val errorMessage = "${error.message}, ${error.unexpectedException.cause ?: error.cause}"
                fireErrorInternal(
                    code = classError,
                    msg = errorMessage,
                    errorMetadata = error.stackTraceToString(),
                    isFatal = isFatal
                )
            }
        }
    }

    private fun fireErrorInternal(
        code: String? = null,
        msg: String? = null,
        errorMetadata: String? = null,
        exception: Exception? = null,
        isFatal: Boolean,
    ) {
        if (isFatal) {
            fireFatalError(code, "Fatal - $msg", errorMetadata, exception)
        } else {
            fireError(code, "Non-fatal -$msg", errorMetadata, exception)
            fireStop()
        }
    }

    override fun stateChangedEnded() {
        if (allowReportStopPlayback().not()) return
        super.stateChangedEnded()
    }

    override fun stateChangedIdle() {
        if (allowReportStopPlayback().not()) return
        super.stateChangedIdle()
    }

    override fun stateChangedReady() {
        if (allowReportStopPlayback().not()) return
        super.stateChangedReady()
    }

    override fun dispose() {
        if (allowReportStopPlayback().not()) {
            monitor?.stop()
            player = null
        } else super.dispose()
    }

    override fun setCustomEventLogger(trackSelector: MappingTrackSelector?) {
        val player = (player as? RealSmartPlayer)?.exoPlayer
        if (player != null) {
            customEventLogger = CustomEventLogger(trackSelector)
            customEventLogger?.let(player::addAnalyticsListener)
            customEventLoggerEnabled = true
        }
    }

    override fun getBitrate(): Long? {
        return if (customEventLoggerEnabled)
            customEventLogger?.bitrate
        else
            qualityProvider?.bitrate
    }

    override fun getTotalBytes(): Long? {
        return if (customEventLoggerEnabled)
            customEventLogger?.totalBytesAccumulated
        else null
    }

    override fun getDroppedFrames(): Int? {
        return customEventLogger?.droppedFrames
    }

    override fun getLatency(): Double? {
        return null
    }

    override fun buildQualityProvider(force: Boolean) {
        /*
         * Only can extract this info from a SimpleExoPlayer. If it is one, create the Provider.
         * Otherwise (ExoPlayer) return a default implementation.
         */
        if (force || qualityProvider == null) {

            val player = (player as? RealSmartPlayer)?.exoPlayer
            if (player is SimpleExoPlayer) {

                qualityProvider = object : PlayerQualityProvider<SimpleExoPlayer>(
                    player
                ) {

                    override val bitrate: Long?
                        get() {
                            return if (customEventLoggerEnabled)
                                customEventLogger?.bitrate
                            else
                                player.videoFormat?.bitrate?.toLong() ?: 0
                        }

                    override val rendition: String?
                        get() {
                            val format = player.videoFormat

                            var bitrate = 0
                            var width = 0
                            var height = 0

                            format?.let {
                                // We use getBitrate() instead of property bitrate just to not mix
                                // it with the bitrate on the scope
                                getBitrate()?.let { bitrate = it.toInt() }
                                width = format.width
                                height = format.height
                            }

                            return if ((width <= 0 || height <= 0) && bitrate <= 0)
                                super.rendition
                            else
                                YouboraUtil.buildRenditionString(width, height, bitrate.toDouble())
                        }

                    override val framerate: Double?
                        get() {
                            return player.videoFormat?.frameRate?.toDouble()
                        }
                }
            } else {
                qualityProvider = PlayerQualityProvider(player)
            }
        }
    }
}