package com.twentyfouri.tvlauncher.common

import android.content.ContentValues
import android.content.Context
import android.net.Uri
import android.text.format.DateFormat
import com.google.firebase.remoteconfig.FirebaseRemoteConfig
import com.google.gson.Gson
import com.google.gson.JsonParseException
import com.twentyfouri.phoenixapi.data.ErrorCodes
import com.twentyfouri.phoenixapi.data.ErrorCodes.APP_TOKEN_NOT_FOUND
import com.twentyfouri.phoenixapi.data.ErrorCodes.ASSET_NOT_FOUND
import com.twentyfouri.phoenixapi.data.ErrorCodes.INVALID_ACTION_PARAMETER
import com.twentyfouri.phoenixapi.data.ErrorCodes.INVALID_UDID
import com.twentyfouri.phoenixapi.data.ErrorCodes.MISSING_PARAMETER
import com.twentyfouri.phoenixapi.data.ErrorCodes.SERVICE_FORBIDDEN
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.caching.LowMemoryHandler
import com.twentyfouri.smartmodel.epg.EpgDatabase
import com.twentyfouri.smartmodel.model.error.AnonymousSessionException
import com.twentyfouri.smartmodel.model.error.GeneralApiException
import com.twentyfouri.smartmodel.model.error.SuspendedException
import com.twentyfouri.smartmodel.model.menu.SmartNavigationAction
import com.twentyfouri.smartmodel.model.user.*
import com.twentyfouri.smartmodel.persistence.SingleSourceOfTruth
import com.twentyfouri.smartmodel.phoenix.*
import com.twentyfouri.smartmodel.phoenix.PhoenixConfiguration
import com.twentyfouri.smartmodel.phoenix.PhoenixKtSmartApi
import com.twentyfouri.smartmodel.phoenix.PhoenixSavedSession
import com.twentyfouri.smartmodel.phoenix.PhoenixSavedSession.PhoenixSavedUser.Companion.DYNAMIC_DATA_KEY_FOR_CUSTOMER_PROFILE_TYPE
import com.twentyfouri.smartmodel.phoenix.configuration.Dms
import com.twentyfouri.smartmodel.phoenix.configuration.EditionFilter
import com.twentyfouri.smartmodel.phoenix.mapper.SmartMediaMapper
import com.twentyfouri.smartmodel.phoenix.referencemodels.PhoenixPlaylistReference
import com.twentyfouri.smartmodel.phoenix.referencemodels.PhoenixUserProfileReference
import com.twentyfouri.smartmodel.phoenix.util.getParentalRuleList
import com.twentyfouri.smartmodel.serialization.SmartDataObject
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.data.SetupDataRepository
import com.twentyfouri.tvlauncher.common.extensions.ifNotNull
import com.twentyfouri.tvlauncher.common.utils.*
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import kotlinx.coroutines.*
import okhttp3.logging.HttpLoggingInterceptor
import timber.log.Timber
import java.io.BufferedReader
import java.io.FileReader
import java.io.IOException
import java.lang.StringBuilder
import java.net.HttpURLConnection
import java.net.SocketTimeoutException
import java.net.URL
import java.net.UnknownHostException
import kotlin.collections.ArrayList

const val SHARED_PREFERENCES_FILE = "savedSession"
const val SHARED_PREFERENCES_JSON = "data"

const val STG_HOME_PAGE_ID = "5297"
const val STG_EPG_PAGE_ID = "5298"
const val STG_APPS_PAGE_ID = "5299"
const val PROD_HOME_PAGE_ID = "13535"
const val PROD_EPG_PAGE_ID = "13536"
const val PROD_APPS_PAGE_ID = "13537"
const val YOUBORA_DEV_CODE = "enteldev"
const val YOUBORA_PROD_CODE = "entel"

private const val fileFormatOnnetFTTH = "FULL_HD"
private const val fileFormatOnnetOTT = "LIGHT_HD"
private const val fileFormatOffnetFTTH = "FULL_HD_OFFNET"
private const val fileFormatOffnetOTT = "LIGHT_HD_OFFNET"

private const val EPG_DATABASE_MAXIMUM_SIZE_MB = 300

enum class OnNetState {
    ON_NET,
    OFF_NET,
    USER_TYPE_BASED
}

enum class ProfileType(val value: String) {
    FTTH("1"),
    OTT("2"),
    //DTH("3")
    ;

    companion object {
        fun fromString(value: String?): ProfileType {
            value ?: return FTTH
            return try {
                values().find { it.value == value } ?: FTTH
            } catch (e: Exception) {
                FTTH
            }
        }
    }
}

class Flavor : FlavorBase() {

    companion object {
        private const val PARTNER_ID = 313
        private const val CONFIGURATION_URL_STG = "https://313-api.frs1.ott.kaltura.com/api_v3/service/"
        private const val CONFIGURATION_URL_PROD = "http://rest-us.ott.kaltura.com/v5_2_2/api_v3/service/"
        private const val CONFIGURATION_URL_PREPROD = "https://313.ort2.ott.kaltura.com/api_v3/service/"
        private const val USER = "dms"
        private const val CREDENTIALS = "tvinci"
        private const val APP_NAME = "com.kaltura.entel.android.stb"
        private const val PLATFORM = "STB"
        private const val VERSION = "1.0.6"
        private const val VERSION_PROD = "1.0.0"

        private val rowMappingSTG = hashMapOf(
                Pair("type:LastChannels", PhoenixPlaylistReference.KIND_LAST_WATCHED),
                Pair("type:RecommendedApps", PhoenixPlaylistReference.KIND_RECOMMENDED_APPS),
                Pair("type:MostWatched", PhoenixPlaylistReference.KIND_EPG_CHANNELS_CHANNEL),
                Pair("type:AllChannels", PhoenixPlaylistReference.KIND_EPG_CHANNELS_CHANNEL),
                Pair("type:Editorial", PhoenixPlaylistReference.KIND_EPG_CHANNELS_CHANNEL)
        )

        private val rowMappingPROD = hashMapOf(
                Pair("13538", PhoenixPlaylistReference.KIND_ON_NOW_CHANNEL), //TODO remove once PROD is migrated to STG configuration
                Pair("13539", PhoenixPlaylistReference.KIND_RECOMMENDED_APPS), //TODO remove once PROD is migrated to STG configuration
                Pair("13540", PhoenixPlaylistReference.KIND_EPG_CHANNELS_CHANNEL), //TODO remove once PROD is migrated to STG configuration
                Pair("type:LastChannels", PhoenixPlaylistReference.KIND_LAST_WATCHED),
                Pair("type:RecommendedApps", PhoenixPlaylistReference.KIND_RECOMMENDED_APPS),
                Pair("type:MostWatched", PhoenixPlaylistReference.KIND_MOST_WATCHED),
                Pair("type:AllChannels", PhoenixPlaylistReference.KIND_EPG_CHANNELS_CHANNEL),
                Pair("type:Editorial", PhoenixPlaylistReference.KIND_EPG_CHANNELS_CHANNEL)
        )

        var activeProfile: PhoenixSavedSession.PhoenixSavedUser? = null

        private var _onnet = OnNetState.OFF_NET
        val onnet get() = _onnet

        private const val FFTH_MOBILE_MESSAGE_KEY = "ftth_mobile_block_message"
        private const val FFTH_WIFI_MESSAGE_KEY = "ftth_wifi_block_message"
        private const val FFTH_ETH_MESSAGE_KEY = "ftth_ethernet_block_message"
        private const val OTT_MOBILE_MESSAGE_KEY = "ott_mobile_block_message"
        private const val OTT_WIFI_MESSAGE_KEY = "ott_wifi_block_message"
        private const val OTT_ETH_MESSAGE_KEY = "ott_ethernet_block_message"
    }

    override val isYouboraEnabled: Boolean = true

    override val youboraAccountCode = if (BuildConfig.IS_DEBUG) YOUBORA_DEV_CODE else YOUBORA_PROD_CODE

    override fun getUsername(sessionData: String?): String? {
        val data = Gson().fromJson(sessionData, PhoenixSavedSession::class.java)
        return data?.activeProfile?.id
    }

    override fun getUserProfileId(userProfile: SmartUserProfile?): String? {
        return (userProfile?.reference as? PhoenixUserProfileReference)?.id
    }

    override fun getUserType(sessionData: String?): String? {
        val data = Gson().fromJson(sessionData, PhoenixSavedSession::class.java)
        return data?.activeProfile?.dynamicData?.get(DYNAMIC_DATA_KEY_FOR_CUSTOMER_PROFILE_TYPE)
    }

    override fun getSmartApi(
        applicationContext: Context?,
        setupDataRepo: SetupDataRepository,
        oselLogger: OselToggleableLogger?
    ): FlowSmartApi {
        applicationContext ?: throw IllegalStateException("Context null when creating SmartApi")
        return PhoenixFlowSmartApi(
            buildPhoenixSmartApi(
                getPhoenixConfiguration(),
                applicationContext,
                setupDataRepo,
                oselLogger
            )
        )
    }

    override fun overrideEditionMapping(smartApi: FlowSmartApi) {
        (smartApi as? PhoenixFlowSmartApi)?.also {
            it.ktSmartApi.setEditionMapping(getEditionFilters(it.ktSmartApi.getFilesFormat()))
        }
    }

    private fun getEditionFilters(filesFormat: SmartDataObject?): List<EditionFilter>? {
        // OnNet + FTTH/missing -> fileFormatOnnetFTTH
        // OnNet + OTT -> fileFormatOnnetOTT
        // OffNet + FTTH/missing -> fileFormatOffnetFTTH
        // OffNet + OTT -> fileFormatOffnetOTT
        // UserTypeBased + FTTH/missing -> fileFormatOnnetFTTH
        // UserTypeBased + OTT -> fileFormatOffnetOTT

        val fileFormatFTTH = when (onnet) {
            OnNetState.ON_NET -> fileFormatOnnetFTTH
            OnNetState.OFF_NET -> fileFormatOffnetFTTH
            OnNetState.USER_TYPE_BASED -> fileFormatOnnetFTTH
        }

        val fileFormatOTT = when (onnet) {
            OnNetState.ON_NET -> fileFormatOnnetOTT
            OnNetState.OFF_NET -> fileFormatOffnetOTT
            OnNetState.USER_TYPE_BASED -> fileFormatOffnetOTT
        }

        //lowercase needed because GetMediaStreamHelper works also with lowerCase
        val filterFTTH = filesFormat?.get(fileFormatFTTH)?.getString()?.toLowerCase()
        val filterOTT = filesFormat?.get(fileFormatOTT)?.getString()?.toLowerCase()

        return ArrayList<EditionFilter>().apply {

            filterFTTH?.let {
                add(EditionFilter().apply {
                    type = "primary"
                    values = listOf(it)
                    userFilterType = DYNAMIC_DATA_KEY_FOR_CUSTOMER_PROFILE_TYPE
                    userFilterValue = ProfileType.FTTH.value
                })

                add(EditionFilter().apply {
                    type = "primary"
                    values = listOf(it)
                })
            }

            filterOTT?.let {
                add(EditionFilter().apply {
                    type = "primary"
                    values = listOf(it)
                    userFilterType = DYNAMIC_DATA_KEY_FOR_CUSTOMER_PROFILE_TYPE
                    userFilterValue = ProfileType.OTT.value
                })

            }
        }
    }

    override suspend fun preStartActions(smartApi: FlowSmartApi, callback: () -> Unit, analyticsReport: (Exception) -> Unit) {
        checkOnNet(smartApi, { callback() }, analyticsReport)
    }

    private suspend fun checkOnNet(
        smartApi: FlowSmartApi,
        callback: () -> Unit,
        analyticsReport: (Exception) -> Unit,
    ) {
        Timber.tag("ONNET").d("onnet started")
        if (NetworkConnectionState.instance.state != NetworkConnectionState.State.ONLINE) {
            Timber.tag("ONNET").d("onnet checking skipped because device is not ONLINE")
            withContext(Dispatchers.Main) { callback() }
            return
        }
        val checkOnNetJob = CoroutineScope(Dispatchers.Default).launch {
            try {
                val url = URL("http://ws-ott.entel.cl/ott/onnet-header.php")
                val urlConnection = (url.openConnection() as HttpURLConnection).apply {
                    connectTimeout = 3000
                    readTimeout = 3000
                }
                val responseCode = urlConnection.responseCode
                val responseMessage = urlConnection.responseMessage
                urlConnection.disconnect()
                Timber.tag("ONNET").d("$responseCode - $responseMessage")

                _onnet = when (responseCode) {
                    403 -> OnNetState.OFF_NET
                    200 -> OnNetState.ON_NET
                    else -> OnNetState.USER_TYPE_BASED
                }
            } catch (e: Exception) {
                Timber.tag("ONNET").d(e)
                _onnet = OnNetState.USER_TYPE_BASED
                if (e is SocketTimeoutException || e is IOException || e is UnknownHostException) {
                    analyticsReport(e)
                }
            }

            if (SharedPreferencesUtils.getLastOnnet() != _onnet.ordinal) {
                SharedPreferencesUtils.putLastOnnet(_onnet.ordinal)
                //clear EpgDatabase to prevent using of wrong stream URL
                clearEpgData(ReflectionUtils.getApplicationContext(), smartApi)
            }
        }
        checkOnNetJob.join()
        Timber.tag("ONNET").d("OnNet set to: $onnet")
        withContext(Dispatchers.Main) {
            callback()
        }
    }

    private fun getPhoenixConfiguration() = PhoenixConfiguration(
            partnerId = PARTNER_ID,
            dms = Dms(
                    configurationsUrl = when {
                        (BuildConfig.IS_STAGING) -> CONFIGURATION_URL_STG
                        (BuildConfig.IS_PRODUCTION) -> CONFIGURATION_URL_PROD
                        else -> CONFIGURATION_URL_PREPROD
                    },
                    user = USER,
                    appName = APP_NAME,
                    credentials = CREDENTIALS,
                    platform = PLATFORM,
                    version = if (BuildConfig.IS_STAGING) VERSION else VERSION_PROD
            ),
            menuMapping = hashMapOf(
                    Pair("id:${getHomePageId()}", SmartNavigationAction.BROWSE_PAGE.name),
                    Pair("id:${getEpgPageId()}", SmartNavigationAction.LIVE_EPG_FULL.name),
                    Pair("id:${getAppsPageId()}", SmartNavigationAction.BROWSE_PAGE.name)
            ),
            categoryIdToPlaylistKindMapping = if (BuildConfig.IS_STAGING) rowMappingSTG else rowMappingPROD,
            advisorySystem = "PhoenixAdvisorySystem",
            recommendationsChannelId = 0, //most likely not relevant for Entel
            epgExpirationInHours = 8,
            epgRepositoryPolicies = listOf(SingleSourceOfTruth.Policy.DONT_EMIT_DATABASE_INCOMPLETE),
            senderId = ReflectionUtils.getPackageInfo()?.let { "${it.versionName}(${it.versionCode})" },
            shouldTryLoginWithDeviceId = true,
            sessionExpirationSeconds = if (BuildConfig.IS_DEBUG) 60 else 60 * 60 * 24,
            refreshTokenExpirationSeconds = if (BuildConfig.IS_DEBUG) 60*20 else 60 * 60 * 24 * 365 * 10,
            allowUseOfAssetRules = false
    )

    @Suppress("ConstantConditionIf")
    private fun getHomePageId() = if (BuildConfig.IS_STAGING) STG_HOME_PAGE_ID else PROD_HOME_PAGE_ID

    @Suppress("ConstantConditionIf")
    private fun getEpgPageId() = if (BuildConfig.IS_STAGING) STG_EPG_PAGE_ID else PROD_EPG_PAGE_ID

    @Suppress("ConstantConditionIf")
    private fun getAppsPageId() = if (BuildConfig.IS_STAGING) STG_APPS_PAGE_ID else PROD_APPS_PAGE_ID

    private fun buildPhoenixSmartApi(
            configuration: PhoenixConfiguration,
            context: Context,
            setupDataRepo: SetupDataRepository,
            oselLogger: OselToggleableLogger?
    ) = PhoenixKtSmartApi(
            configuration = configuration,
//            udid = "AOT1000003789", //prod 1
//            udid = "AOT1000003", //stg 1
            //udid = "262015701035373" //Jiri's Entel box
            //to enable custom udids (mac address) you need to modify method getSerialNumberFromBroadcast
            udid = DeviceId.getDeviceId(context),
            deviceBrandCode = PhoenixKtSmartApi.DEVICE_BRAND_CODE_STB,
            cacheCleaner = LowMemoryHandler(context),
            sessionPersistence = object : SmartSessionPersistence {
                override fun loadSession(): SmartSavedSession? = runBlocking {
                    Timber.tag("SessionStorage").d("SmartSessionPersistence loadSession")
                    val setupData = setupDataRepo.getSetupDataObject()
                    Timber.tag("SessionStorage")
                        .d("SmartSessionPersistence loadSession json: ${setupData?.sessionData}")
                    val sessionFromProvider = getSavedSession(setupData?.sessionData)
                    sessionFromProvider.ifNotNull { return@runBlocking sessionFromProvider }

                    // try shared preferences. This was used for storage by previous versions, so this can be
                    // deleted when we are way over build 121 and we are sure we will never get installed over
                    // version older than 121
                    //TODO delete some day
                    val sp = context.getSharedPreferences(SHARED_PREFERENCES_FILE, Context.MODE_PRIVATE)
                    val json = sp.getString(SHARED_PREFERENCES_JSON, null)
                    Timber.tag("SessionStorage").d("SmartSessionPersistence loadSession json: $json")
                    val sessionFromSharedPreferences = getSavedSession(json)
                    sessionFromSharedPreferences.ifNotNull { return@runBlocking sessionFromSharedPreferences }

                    return@runBlocking null
                }

                override fun saveSession(session: SmartSavedSession?) {
                    if (session == null) {
                        Timber.tag("SessionStorage").d("SmartSessionPersistence saveSession - session is NULL and will not be saved")
                        return
                    }
                    Timber.tag("SessionStorage").d("SmartSessionPersistence saveSession")
                    val json: String? = session.let { Gson().toJson(it) }
                    Timber.tag("SessionStorage").d("SmartSessionPersistence saveSession json: $json")
                    val values = ContentValues()
                    val uri: Uri = Uri.parse(context.getString(R.string.credentials_uri))
                    values.put("sessionData", json)
                    context.contentResolver.insert(uri, values)
                }
            },
//            sessionPersistence = object : SmartSessionPersistence {
//                override fun loadSession(): SmartSavedSession? {
//                    Log.d("SessionStorage", "SmartSessionPersistence loadSession")
//                    return PhoenixSavedSession(SmartSessionState.LOGGED_IN).apply {
////                        debugUserName = "24000000-5" //stg 1
////                        debugUserName = "11401312-2" //prod 1
////                        debugUserName = "11401312-2" //Jiri's Entel box
//                        debugPassword = "123456"
//                    }
//                }
//
//                override fun saveSession(session: SmartSavedSession?) {
//                    Log.d("SessionStorage", "SmartSessionPersistence saveSession")
//                }
//            },
            buildType = BuildConfig.BUILD_TYPE,
//        buildType ="debug",
            epgDao = EpgDatabase.getInstance(context).epgDao,
            httpLoggingInterceptor = oselLogger?.let { HttpLoggingInterceptor(it).setLevel(HttpLoggingInterceptor.Level.BODY) }
    )

    override fun clearSessionData(context: Context) {
        val sp = context.getSharedPreferences(SHARED_PREFERENCES_FILE, Context.MODE_PRIVATE)
        sp.edit().clear().apply()
    }

    override suspend fun clearEpgData(context: Context?, flowSmartApi: FlowSmartApi?) {
        (flowSmartApi as? PhoenixFlowSmartApi)?.ktSmartApi?.invalidateAllCaches()
        context?.let {
            EpgDatabase.getInstance(it).epgDao.apply {
                deleteAllEpgEvents()
                deleteAllEpgHoles()
                deleteAllPlaylistItems()
                deleteAllEpgChannels()
            }
        }
    }

    override suspend fun performPing(flowSmartApi: FlowSmartApi) {
        require(flowSmartApi is PhoenixFlowSmartApi) { "Wrong type of FlowSmartApi" }
        //TODO remove invalid SmartModel interface
        flowSmartApi.ktSmartApi.ping()
    }

    override fun getSavedSession(setupDataString: String?): SmartSavedSession? {
        return try {
            Gson().fromJson(setupDataString, PhoenixSavedSession::class.java)
        } catch (e: JsonParseException) {
            Timber.tag("SessionStorage").e(e, "Could not parse session")
            null
        }
    }

    override fun getUserProfileReference(userName: String): SmartUserProfileReference {
        return PhoenixUserProfileReference(userName)
    }

    override fun gdprProfileChangeObject(isGdrpAccepted: Boolean?): SmartDataObject {
//        return SmartDataObject(Pair(EnvisionProfileMapper.GDPR_KEY, !isGdrpAccepted))
        return SmartDataObject(Pair("", false))
    }

    override fun isGdprAcceptedProfile(extras: SmartDataObject): Boolean {
//        return extras.getBoolean(EnvisionProfileMapper.GDPR_KEY, false)
        return false
    }

    override suspend fun updateUserParentalRule(flowSmartApi: FlowSmartApi): List<String>? {
       //TODO remove this custom interface
        //TODO smart model should NOT be accessed this way, only as a FlowSmartApi
        return ((flowSmartApi as? PhoenixFlowSmartApi)?.ktSmartApi)?.getParentalRuleList()
    }

    override val parentalRulesRefreshActivated: Boolean = true

//    override fun getSerialNumberFromBroadcast() = false //set to false when you want to use MAC address
    override fun getSerialNumberFromBroadcast() = true //set to false when you want to use MAC address

    override fun pinDialogWithSoftInput(): Boolean = true

    //TODO this method is calling invalid (Phoenix only) smart model
    override suspend fun verifyPurchasePin(flowSmartApi: FlowSmartApi, purchasePin: String) {
        require(flowSmartApi is PhoenixFlowSmartApi)
        flowSmartApi.ktSmartApi.verifyPurchasePin(purchasePin)
    }

    override fun showDialogs() = true

    override fun dialogAllowedForException(exception: Exception): Boolean =
            when (exception) {
                is AnonymousSessionException -> false
                is GeneralApiException -> {
                    when (exception.code.toIntOrNull()){
                        SERVICE_FORBIDDEN,
                        ASSET_NOT_FOUND,
                        INVALID_ACTION_PARAMETER,
                        MISSING_PARAMETER,
                        APP_TOKEN_NOT_FOUND,
                        INVALID_UDID -> false
                        else -> true
                    }
                }
                else -> true
            }

    override fun getErrorMessage(e: GeneralApiException, applicationContext: Context): String {
        val resourceRepository = ResourceRepository(applicationContext)
        return when (e.code) {
            ErrorCodes.NOT_ENTITLED.toString(), "NotEntitled" -> resourceRepository.getString(R.string.not_entitled, e.code)
            ErrorCodes.USER_SUSPENDED.toString() -> resourceRepository.getString(R.string.user_suspended, e.code)
            ErrorCodes.DEVICE_TYPE_NOT_ALLOWED.toString() -> resourceRepository.getString(R.string.device_type_not_allowed, e.code)
            ErrorCodes.KS_EXPIRED.toString() -> resourceRepository.getString(R.string.invalid_ks)
            ErrorCodes.DOMAIN_NOT_EXISTS.toString() -> resourceRepository.getString(R.string.domain_not_exists, e.code)
            ErrorCodes.HOUSEHOLD_USER_FAILED.toString() -> resourceRepository.getString(R.string.household_user_failed, e.code)
            ErrorCodes.MEDIA_CONCURRENCY_LIMITATION.toString(), "MediaConcurrencyLimitation" -> resourceRepository.getString(R.string.media_concurrency_limitation)
            ErrorCodes.CONCURRENCY_LIMITATION.toString(), "ConcurrencyLimitation" -> resourceRepository.getString(R.string.concurrency_limitation)
            else -> resourceRepository.getString(R.string.unexpected_error_common, e.code)
        }
    }

    override fun allowCancelableDialog(e: SuspendedException): Boolean {
        return when (e.code){
            ErrorCodes.HOUSEHOLD_SUSPENDED.toString() -> false
            else -> true
        }
    }

    override fun allowSubscriptionPurchaseScreen(): Boolean = true

    override val backendConnectionTestUrl
            = if(BuildConfig.IS_PRODUCTION) "http://rest-us.ott.kaltura.com" else "https://313.ort2.ott.kaltura.com"

    override val networkCapabilityMobile: String = SmartMediaMapper.KEY_NOT_RELEASED_ON_ATV_MOBILE
    override val networkCapabilityWifi: String = SmartMediaMapper.KEY_NOT_RELEASED_ON_ATV_WIFI
    override val networkCapabilityEth: String = SmartMediaMapper.KEY_NOT_RELEASED_ON_ATV_ETH
    override val deviceInfoConfig = DeviceInfoConfig("com.twentyfouri.entellauncher", "com.twentyfouri.setupwizard.entel")
    override val epgDatabaseMaximumSizeMB = EPG_DATABASE_MAXIMUM_SIZE_MB
    override val useSafeListOfApprovedNetworks: Boolean = true

    override val firmwareVersion = GlobalScope.async(Dispatchers.IO, CoroutineStart.LAZY) {
        //first check file sdmc.prop file because props could have same ids with invalid values
        //ro.sdmc.firmware.ver
        //ro.odm.sdmc.firmware.ver
        //then check props
        //ro.sdmc.firmware.ver
        //ro.build.version.incremental
        val fileIds = listOf("ro.sdmc.firmware.ver", "ro.odm.sdmc.firmware.ver")
        val propIds = listOf("ro.sdmc.firmware.ver", "ro.build.version.incremental", "ro.odm.sdmc.firmware.ver")

        try {
            BufferedReader(FileReader("/system/sdmc.prop")).readText().lines().forEach { line ->
                fileIds.forEach {
                    if(line.contains(it)) {
                        val value = line.split("=").getOrNull(1)
                        if(!value.isNullOrEmpty()) return@async value
                    }
                }
            }
        } catch (e: Exception) {
            //ignore any error, props will be searched
        }

        propIds.forEach {
            val result = DeviceProperties().getProperty(it)
            if(result?.isNotEmpty() == true) {
                return@async result
            }
        }
        return@async "not found"
    }

    override fun messageLicenceMobile(context: Context): String =
        FirebaseRemoteConfig.getInstance().getStringOrNull(FFTH_MOBILE_MESSAGE_KEY, OTT_MOBILE_MESSAGE_KEY)
            ?: super.messageLicenceMobile(context)

    override fun messageLicenceWiFi(context: Context): String =
        FirebaseRemoteConfig.getInstance().getStringOrNull(FFTH_WIFI_MESSAGE_KEY, OTT_WIFI_MESSAGE_KEY)
            ?: super.messageLicenceMobile(context)

    override fun messageLicenceEth(context: Context): String =
        FirebaseRemoteConfig.getInstance().getStringOrNull(FFTH_ETH_MESSAGE_KEY, OTT_ETH_MESSAGE_KEY)
            ?: super.messageLicenceMobile(context)

    private fun FirebaseRemoteConfig.getStringOrNull(ffthKey: String, ottKey: String): String? =
        when (SharedPreferencesUtils.getLastUserType()) {
            ProfileType.FTTH.value -> getString(ffthKey).let { if (it.isNotBlank()) it else null }
            ProfileType.OTT.value -> getString(ottKey).let { if (it.isNotBlank()) it else null }
            else -> null
        }

    override suspend fun getSessionInfo(context: Context): String {
        val setupData = SetupDataRepository.getInstance(context).getSetupData()
        val data = Gson().fromJson(setupData, PhoenixSavedSession::class.java)
        val userType = data?.activeProfile?.dynamicData?.get(DYNAMIC_DATA_KEY_FOR_CUSTOMER_PROFILE_TYPE)

        return StringBuilder()
                .appendLine("UDID: ${data?.udid}")
                .appendLine("OnNet: $onnet")
                .appendLine("Profile type: dynamic data -> $userType, type -> ${ProfileType.fromString(userType)} ")
                .appendLine("Master profile: ${data?.activeProfile?.isMaster}")
                .appendLine("User name: ${data?.activeProfile?.id}")
                .appendLine("Email: ${data?.activeProfile?.email}")
                .appendLine("First, last name: ${data?.activeProfile?.firstName}, ${data?.activeProfile?.lastName}")
                .appendLine("KS: ${data?.ks}")
                .appendLine("KS created: ${DateFormat.format("dd.MM.yyyy HH:mm:ss", data?.ksCreateTimeMs ?: 0)}")
                .appendLine("KS expire: ${DateFormat.format("dd.MM.yyyy HH:mm:ss", data?.ksExpiryTimeMs ?: 0)}")
                .appendLine("RT: ${data?.refreshToken}")
                .appendLine("RT id: ${data?.refreshTokenId}")
                .appendLine("RT created: ${DateFormat.format("dd.MM.yyyy HH:mm:ss", data?.refreshTokenCreateTimeMs ?: 0)}")
                .append("RT expire: ${DateFormat.format("dd.MM.yyyy HH:mm:ss", data?.refreshTokenExpiryTimeMs ?: 0)}")
                .toString()
    }

    override val shouldIgnoreNotFoundException = true
    override val offerOfflineIfMissingSession = false
    override val offerLoginIfInvalidSession = false
}

//Known info about Entel devices
//BIG box  VSB3930
//ro.build.version.incremental 11.1.0.3
//ro.odm.build.version.incremental 11.1.0.3
//ro.product.build.version.incremental 11.1.10.3
//ro.system.build.version.incremental 11.1.0.3
//ro.system_ext.build.version.incremental 11.1.0.3
//ro.vendor.build.version.incremental 11.1.0.3
//
//MIDDLE box  DIW586
//ro.product.name DIW586
//ro.product.odm.name DIW586
//ro.product.product.name DIW586
//ro.product.system.name DIW586
//ro.product.vendor.name DIW586
//sdmc.prop file available in system/
//ro.odm.sdmc.firmware.ver=11.3.42
//ro.product.name=DIW586
//ro.product.system.name=DIW586
//
//SMALL box (red)  DIW585
//ro.product.device DIW585
//ro.product.vendor.device DIW585
//ro.sdmc.firmware.ver 9.4.90
//sdmc.prop file available in system/
//ro.sdmc.firmware.ver=9.4.90
//ro.product.device=DIW585
//
//try key first
//ro.sdmc.firmware.ver
//ro.build.version.incremental
//then sdmc.prop file
//ro.sdmc.firmware.ver
//ro.odm.sdmc.firmware.ver