package com.twentyfouri.tvlauncher.viewmodels

import android.view.View
import android.widget.Button
import androidx.lifecycle.*
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.payment.SmartPurchase
import com.twentyfouri.smartmodel.model.payment.SmartSubscription
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.extensions.enlarge
import com.twentyfouri.tvlauncher.common.utils.PurchaseChecker
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.data.MetadataRepository
import timber.log.Timber

class SubscriptionsGridViewModel(
    val metadataRepository: MetadataRepository,
    val resourceRepository: ResourceRepository
) : ViewModel() {

    private val subscriptionChannel = MutableLiveData<SmartMediaItem>()
    private val _offerOneVisibility = MutableLiveData<Int>().apply { value = View.VISIBLE }
    private val _offerTwoVisibility = MutableLiveData<Int>().apply { value = View.GONE }
    private val _offerThreeVisibility = MutableLiveData<Int>().apply { value = View.GONE }
    private val _subscriptionBought = MediatorLiveData<SmartPurchase?>()
    private val _channelName = MutableLiveData<String>().apply { value = resourceRepository.getString(R.string.channel_not_subscribed)}
    private val _channelDescriptionText = MutableLiveData<String>().apply { value = resourceRepository.getString(R.string.subscription_selection_text)}
    private val _cancelButtonVisibility = MutableLiveData<Int>().apply { value = View.GONE }
    private val _subscriptionPrice = MutableLiveData<String>().apply { value = "" }
    private val _restartSubscriptionView = MutableLiveData<Boolean>(). apply { value = false }

    val cancelButtonVisibility: LiveData<Int> = _cancelButtonVisibility
    val subscriptionPackages: LiveData<List<SmartSubscription>>
    val channelName:  LiveData<String> = _channelName
    val channelDescriptionText:  LiveData<String> = _channelDescriptionText
    val offerOneButtonText: LiveData<String>
    val offerTwoButtonText: LiveData<String>
    val offerThreeButtonText: LiveData<String>
    val offerOneVisibility: LiveData<Int> = _offerOneVisibility
    val offerTwoVisibility: LiveData<Int> = _offerTwoVisibility
    val offerThreeVisibility: LiveData<Int> = _offerThreeVisibility
    val subscriptionBought: LiveData<SmartPurchase?> = _subscriptionBought
    val restartSubscriptionView: LiveData<Boolean> = _restartSubscriptionView
    val channelsIncludedInSubscription: LiveData<List<SmartMediaItem>>
    val subscriptionPrice : LiveData<String> = _subscriptionPrice
    private val indexLD = MutableLiveData<Int>(). apply { value = 0 }

    init {
        subscriptionPackages = Transformations.switchMap(subscriptionChannel) { metadataRepository.getSubscriptionsListLD(it?.reference) }
        channelsIncludedInSubscription = Transformations.switchMap(indexLD) { metadataRepository.getSubscriptionChannelsList(subscriptionPackages.value?.getOrNull(it))
        }
        offerOneButtonText = Transformations.map(subscriptionPackages){
            when {
                it.isNotEmpty() ->{
                    "Ver ${it[0].name}"
                }
                else -> ""
                }
            }

        offerTwoButtonText = Transformations.map(subscriptionPackages){
            when {
                it.size > 1 -> {
                    _offerTwoVisibility.value = View.VISIBLE
                    "Ver ${it[1].name}"
                }
                else -> ""
            }
        }

        offerThreeButtonText = Transformations.map(subscriptionPackages){
            when {
                it.size > 2 -> {
                    _offerThreeVisibility.value = View.VISIBLE
                    "Ver ${it[2].name}"
                }
                else -> ""
            }
        }
    }

    fun onFocusChange(v: View, hasFocus: Boolean) {
        if (hasFocus) {
            if(v is Button) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Subscription (${channelName.value}) button focused: ${v.text}")
            v.foreground = resourceRepository.getDrawable(com.twentyfouri.tvlauncher.R.color.transparent)
        }
        else v.foreground = resourceRepository.getDrawable(com.twentyfouri.tvlauncher.R.color.transparent_10)
        v.enlarge(hasFocus)
    }

    fun onOfferOneButtonClicked(v: View) {
        if(v is Button) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Subscription (${channelName.value}) button selected: ${v.text}")
        onOfferButtonClicked(0)
    }

    fun onOfferTwoButtonClicked(v: View) {
        if(v is Button) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Subscription (${channelName.value}) button selected: ${v.text}")
        onOfferButtonClicked(1)
    }

    fun onOfferThreeButtonClicked(v: View) {
        if(v is Button) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Subscription (${channelName.value}) button selected: ${v.text}")
        onOfferButtonClicked(2)
    }

    private fun onOfferButtonClicked(index: Int) {
        this.indexLD.value = index
        val subscriptionPackageName = subscriptionPackages.value?.get(index)?.name
        _channelName.value = subscriptionPackageName
        _channelDescriptionText.value = resourceRepository.getString(R.string.channel_included_list, subscriptionPackageName.toString())
        _cancelButtonVisibility.value = View.VISIBLE
        _subscriptionPrice.value = "${subscriptionPackages.value?.get(index)?.extras?.get("currencySign")?.getString("")}" +
                subscriptionPackages.value?.get(index)?.price?.amount?.toInt().toString()
    }
    fun onPurchaseButtonClicked(v: View) {
        if(v is Button) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Subscription (${channelName.value}) button selected: ${v.text}")
        val doAfterPinVerified: () -> Unit = {
            _subscriptionBought.addSource(metadataRepository.subscribeProduct(
                    indexLD.value?.let { subscriptionPackages.value?.get(it)?.reference },
                    indexLD.value?.let { subscriptionPackages.value?.get(it)?.isAutoRenew },
                    indexLD.value?.let { subscriptionPackages.value?.get(it) }
            )) { _subscriptionBought.value = it }
        }
        PurchaseChecker.checkPurchaseWithPurchasePin(v.context, doAfterPinVerified)
    }

    fun onCancelButtonClicked(v: View) {
        if(v is Button) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Subscription (${channelName.value}) button selected: ${v.text}")
        clearModel()
        restartSubscriptionView()
    }

    private fun restartSubscriptionView() {
        _restartSubscriptionView.value = true
        indexLD.value = 0
        setSubscriptionChannel(subscriptionChannel.value)
    }

    fun hideAllOffers() {
        _offerOneVisibility.value = View.GONE
        _offerTwoVisibility.value = View.GONE
        _offerThreeVisibility.value = View.GONE
    }

    fun clearModel(){
        _offerOneVisibility.value = View.VISIBLE
        _offerTwoVisibility.value = View.GONE
        _offerThreeVisibility.value = View.GONE
        _subscriptionBought.value = null
        _channelName.value = resourceRepository.getString(R.string.channel_not_subscribed)
        _channelDescriptionText.value = resourceRepository.getString(R.string.subscription_selection_text)
        _cancelButtonVisibility.value = View.GONE
        _subscriptionPrice.value = ""
        _restartSubscriptionView.value = false

    }

    fun setSubscriptionChannel(channel: SmartMediaItem?) { if (channel!=null) subscriptionChannel.value = channel!! }

    fun setSubscriptionBought(purchaseResult: SmartPurchase?) { _subscriptionBought.value = purchaseResult }

    fun setRestartSubscriptionView(boolean: Boolean){ _restartSubscriptionView. value = boolean}

    fun dismissPurchaseDialog() = PurchaseChecker.dismissPurchasePinDialog()
}