package com.twentyfouri.tvlauncher.viewmodels

import android.animation.ValueAnimator
import android.view.View
import androidx.lifecycle.*
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PlaylistType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.extensions.ifNotEqualTo
import com.twentyfouri.tvlauncher.data.HardcodedPlaylistReference
import com.twentyfouri.tvlauncher.data.RowPageRepository
import com.twentyfouri.tvlauncher.utils.CombinationTransformations
import com.twentyfouri.tvlauncher.utils.RowPageAnimationHelper
import com.twentyfouri.tvlauncher.utils.StringsMapper
import com.twentyfouri.tvlauncher.utils.ViewHolder
import kotlinx.coroutines.flow.*
import timber.log.Timber


open class RowViewModel(
    private val repository: RowPageRepository,
    private val resourceRepository: ResourceRepository,
    val sectionViewModel: SectionViewModel,
    val rowIndex: Int = NO_INDEX
) : ViewModel() {

    private val sectionIndex
        get() = sectionViewModel.sectionIndex
    val sectionStyle
        get() = sectionViewModel.pageSection.sectionStyle
    private val section
        get() = sectionViewModel.pageSection
    private val scrollType
        get() = sectionViewModel.scrollType
    val index
        get() = (sectionIndex*100) + rowIndex

    override fun equals(other: Any?): Boolean {
        val otherRVM = (other as? RowViewModel) ?: return false
        sectionIndex.ifNotEqualTo(otherRVM.sectionIndex) { return false }
        return rowIndex == otherRVM.rowIndex
    }

    fun equalsContent(other: RowViewModel): Boolean {
        //TODO
        return this == other
    }

    //these MutableLiveData are intentionally public, i've tried to find some way how to make them just private, but in this case we need to update the data
    //depending on what is currently in them, thus letting them be public was in this case the least problematic solution
    val rowData = MutableLiveData<List<Any>>()

    //this one is public because we have to add/remove LiveData sources in runtime
    val isFocused = MediatorLiveData<Boolean>().apply { value = false }
    val isFilterFocused = MediatorLiveData<Boolean>().apply { value = false }

    // mutable data for animated behavior
    private val useAnimatedLayout = MutableLiveData<Boolean>().apply { value = false }
    private val _sectionTitleFocused = MutableLiveData<Boolean>().apply { value = false }
    private val _sectionOptionsFocused = MutableLiveData<Boolean>().apply { value = false }
    private val _sectionTitleRowFocused = MutableLiveData<Boolean>().apply { value = false }
    private val _sectionOptionsRowFocused = MutableLiveData<Boolean>().apply { value = false }
    private val _hasSectionOptions = MutableLiveData<Boolean>().apply { value = false } //TODO: change this if row has left options
    private val _loadMoreButtonVisibility = MutableLiveData<Int>().apply { value = View.GONE }

    val headerText: LiveData<String>
    val headerVisibility: LiveData<Int>
    val filtersVisibility: LiveData<Int>
    val paddingVertical = MutableLiveData<Int>().apply { value = 0 }
    var paddingVerticalValueAnimator: ValueAnimator? = null
    val paddingVerticalBottom = MutableLiveData<Int>().apply { value = 0 }
    var paddingVerticalBottomValueAnimator: ValueAnimator? = null
    private val paddingVerticalBig = resourceRepository.getDimensionPixelSize(R.dimen.big_card_focused_padding)
    private val paddingVerticalMiddle = resourceRepository.getDimensionPixelSize(R.dimen.middle_card_focused_padding)
    private val paddingVerticalSmall = resourceRepository.getDimensionPixelSize(R.dimen.small_card_focused_padding)
    private val paddingVerticalBottomBig = resourceRepository.getDimensionPixelSize(R.dimen.big_card_focused_padding_bottom)
    private val paddingVerticalBottomMiddle = resourceRepository.getDimensionPixelSize(R.dimen.middle_card_focused_padding_bottom)
    private val paddingVerticalBottomSmall = resourceRepository.getDimensionPixelSize(R.dimen.small_card_focused_padding_bottom)

    val metadataVisibility: LiveData<Int>
    val appChannelMetadataVisibility: LiveData<Int>
    val metadataMarginBottom: LiveData<Int>
    var rowAlpha: LiveData<Float>
    val headerColor: LiveData<Int>
    val metadataMarginStart: LiveData<Int>
    val filterViewHolders: LiveData<List<ViewHolder<FilterViewModel>>?>

    //immutable data for animated behavior
    val sectionTitleRowFocused: LiveData<Boolean> = _sectionTitleRowFocused
    val sectionOptionsRowFocused: LiveData<Boolean> = _sectionOptionsRowFocused
    val hasSectionOptions: LiveData<Boolean> = _hasSectionOptions //TODO: change this if row has left options
    val bottomOptionsVisibility: LiveData<Int>
    val loadMoreButtonVisibility: LiveData<Int> = _loadMoreButtonVisibility
    val narrowFilterButtonVisibility: LiveData<Int>
    val sectionTitleVisibility: LiveData<Int>
    val generalBottomPadding: LiveData<Int>
    val sectionOptionsVisibility: LiveData<Int>
    val rowIsEmpty: LiveData<Boolean>
    val rowVisibility: LiveData<Int>
    val forceHide: MutableLiveData<Boolean> = MutableLiveData(false)

    private val _noDataVisibility = MutableStateFlow(View.GONE)
    val noDataVisibility: StateFlow<Int> = _noDataVisibility

    private val isRowFirst = rowIndex == 0

    init {
        headerText = MutableLiveData<String>().apply { value =  StringsMapper.translate(section.sectionStyle, section.label) }
        headerVisibility = Transformations.map(useAnimatedLayout) { animations ->
            when {
                isRowInGridButNotFirst() -> View.GONE
                section.sectionStyle == HardcodedPlaylistReference.Type.PLAY_STORE.name -> View.GONE
                animations == true -> View.GONE
                Flavor().getPlaylistType(section) == PlaylistType.GRID -> View.GONE
                else -> View.VISIBLE
            }
        }
        filtersVisibility = Transformations.map(useAnimatedLayout) { animations ->
            when {
                isRowInGridButNotFirst() -> View.GONE
                section.sectionStyle == HardcodedPlaylistReference.Type.PLAY_STORE.name -> View.GONE
                animations == true -> View.GONE
                Flavor().getPlaylistType(section) != PlaylistType.GRID -> View.GONE
                else -> View.VISIBLE
            }
        }
        filterViewHolders = Transformations.map(sectionViewModel.filterViewHolders) {
            if (isRowFirst) it
            else null
        }
        sectionTitleVisibility = Transformations.map(useAnimatedLayout) {
            if (it == true) View.VISIBLE
            else View.GONE
        }
        sectionOptionsVisibility = Transformations.map(hasSectionOptions) {
            if (it) View.VISIBLE
            else View.GONE
        }

        rowAlpha = CombinationTransformations.combineNullable(isFocused, useAnimatedLayout) { focused, animations ->
            if (animations == true || focused == true) 1.0f
            else 0.8f
        }
        headerColor = Transformations.map(isFocused) {
            if (it) resourceRepository.getColor(R.color.white)
            else resourceRepository.getColor(R.color.text_color_unfocused)
        }
        metadataVisibility = Transformations.map(isFocused) {
            if (it == true && scrollType == SectionViewModel.ScrollType.FIXED_LEFT_EDGE
                && Flavor().getPlaylistType(section) != PlaylistType.EXTERNAL_CHANNEL) View.VISIBLE
            else View.GONE
        }
        appChannelMetadataVisibility = Transformations.map(isFocused) {
            if (it == true && scrollType == SectionViewModel.ScrollType.FIXED_LEFT_EDGE
                && Flavor().getPlaylistType(section) == PlaylistType.EXTERNAL_CHANNEL) View.VISIBLE
            else View.GONE
        }

        metadataMarginBottom = Transformations.map(metadataVisibility) {
            when (it) {
                View.VISIBLE -> 0
                else -> resourceRepository.getDimensionPixelSize(R.dimen.metadata_invisible_margin)
            }
        }
        generalBottomPadding = Transformations.map(_sectionTitleRowFocused) {
            when {
                isRowInGridButNotLast() -> resourceRepository.getDimensionPixelSize(R.dimen.row_space_between_tiles)
                it -> resourceRepository.getDimensionPixelSize(R.dimen.row_margin_vertical_half)
                else -> resourceRepository.getDimensionPixelSize(R.dimen.row_margin_vertical)
            }
        }
        metadataMarginStart = Transformations.map(useAnimatedLayout) { if (it) 0 else resourceRepository.getDimensionPixelSize(R.dimen.row_first_item) }
        rowIsEmpty = Transformations.map(rowData) { it.isEmpty() }
        rowVisibility = CombinationTransformations.combineNonNullable(rowIsEmpty, forceHide, filtersVisibility) { empty, force, filtersVisibility ->
             when {
                sectionStyle == HardcodedPlaylistReference.Type.OFFLINE.name -> {
                    if(force) View.GONE
                    else View.VISIBLE
                }
                Flavor().getPlaylistType(section) == PlaylistType.GRID && filtersVisibility == View.VISIBLE -> {
                    //Visible even in case of empty row to keep the filter options displayed
                    View.VISIBLE
                }
                else -> {
                    if (force || empty) View.GONE
                    else View.VISIBLE
                }
            }
        }

        isFocused.asFlow().distinctUntilChanged().combine(isFilterFocused.asFlow().distinctUntilChanged()) { focused, filterFocused ->
            val playlistType = Flavor().getPlaylistType(sectionViewModel.pageSection)
            when {
                focused && (playlistType == PlaylistType.ON_NOW
                        || playlistType == PlaylistType.DO_NOT_MISS_CHANNELS
                        || playlistType == PlaylistType.FAVORITE_CHANNELS
                        || playlistType == PlaylistType.CATCHUP_PROGRAMS
                        || playlistType == PlaylistType.RECENT_RECORDINGS
                        || playlistType == PlaylistType.LAST_WATCHED_CHANNELS) -> Pair(paddingVerticalBig, paddingVerticalBottomBig)
                focused && filterFocused -> Pair(1, 1)
                focused && playlistType == PlaylistType.GRID -> Pair(paddingVerticalMiddle, paddingVerticalBottomMiddle)
                focused -> Pair(paddingVerticalSmall, paddingVerticalBottomSmall)
                else -> Pair(1, 1) //for reason which i was not able to uncover the initial padding here must be >0
            }
        }.onEach {
            paddingVerticalValueAnimator?.cancel()
            val paddingStart = paddingVertical.value ?: 0
            val paddingGoal = it.first
            if(paddingStart != paddingGoal) {
                paddingVerticalValueAnimator = ValueAnimator.ofInt(paddingStart, paddingGoal).apply {
                    duration = 150L //in millis
                    addUpdateListener { animation -> paddingVertical.value = animation.animatedValue as Int }
                    start()
                }
            }
            paddingVerticalBottomValueAnimator?.cancel()
            val paddingBottomStart = paddingVerticalBottom.value ?: 0
            val paddingBottomGoal = it.second
            if(paddingBottomStart != paddingBottomGoal) {
                paddingVerticalBottomValueAnimator = ValueAnimator.ofInt(paddingBottomStart, paddingBottomGoal).apply {
                    duration = 150L //in millis
                    addUpdateListener { animation -> paddingVerticalBottom.value = animation.animatedValue as Int }
                    start()
                }
            }
        }.launchIn(viewModelScope)

        if (isRowFirst) {
            rowIsEmpty.asFlow().combine(filtersVisibility.asFlow()) { isEmpty, filtersVisibility ->
                if (isEmpty) {
                    filtersVisibility
                } else {
                    View.GONE
                }
            }.onEach {
                _noDataVisibility.value = it
            }.launchIn(viewModelScope)
        }
        narrowFilterButtonVisibility = Transformations.map(loadMoreButtonVisibility){
            if (it == View.GONE && sectionItemLimitReached() && isRowInGridAndLast()) View.VISIBLE else View.GONE
        }
        bottomOptionsVisibility = CombinationTransformations.combineNonNullable(
            loadMoreButtonVisibility, narrowFilterButtonVisibility
        ) { loadMoreVisibility, focusFiltersVisibility ->
            if (loadMoreVisibility == View.VISIBLE || focusFiltersVisibility == View.VISIBLE) View.VISIBLE else View.GONE
        }
    }

    private fun isRowInGridButNotLast() = sectionViewModel.rowViewModels.value.let { (it?.size ?: 0) > 1 && it?.lastOrNull() != null && it.last() != this }

    private fun isRowInGridButNotFirst() = sectionViewModel.rowViewModels.value.let { (it?.size ?: 0) > 1 && it?.firstOrNull() != null && it.first() != this }

    private fun isRowInGridAndLast(): Boolean {
        (Flavor().getPlaylistType(section) == PlaylistType.GRID).ifFalse { return false }
        return sectionViewModel.rowViewModels.value.let { (it?.size ?: 0) > 1 && it?.lastOrNull() != null && it.last() == this }
    }

    private fun sectionItemLimitReached() =
        (section.selectedOptions?.totalCount ?: 0) >= Flavor().getPlaylistTotalSizeMax(section)

    private fun updateLoadMoreVisibility(){
        return when {
            isRowInGridAndLast() && Flavor().hasAnotherPage(section, sectionViewModel.getOldDataTotalCount()) -> _loadMoreButtonVisibility.postValue(View.VISIBLE)
            else -> _loadMoreButtonVisibility.postValue(View.GONE)
        }
    }

    fun onSectionTitleClicked(@Suppress("UNUSED_PARAMETER") v: View) {
        //TODO: add onClick action
    }

    fun isUsingFilters(): Boolean {
        return filterViewHolders.value?.isNotEmpty() == true
    }

    fun onSectionTitleFocus(v: View, hasFocus: Boolean) {
        _sectionTitleFocused.value = hasFocus //must happen immediately
        if (hasFocus) RowPageAnimationHelper.doOneCategoryFocused(v)
        else RowPageAnimationHelper.doOneCategoryNotFocused(v)
    }

    fun onSectionOptionsFocus(@Suppress("UNUSED_PARAMETER") v: View, hasFocus: Boolean) {
        if (_hasSectionOptions.value == true) _sectionOptionsFocused.value = hasFocus //must happen immediately
    }

    fun setUseAnimatedLayout(boolean: Boolean) { useAnimatedLayout.postValue(boolean) }

    fun setSectionTitleRowFocused(boolean: Boolean, post: Boolean = true) {
        if(post) _sectionTitleRowFocused.postValue(boolean) else _sectionTitleRowFocused.value = boolean
    }

    fun updateFromNewRowModels(newRowModels: List<RowViewModel>) {
        val newRowModel = newRowModels.find { it.index == index }
        val newRowData = newRowModel?.rowData?.value
        Timber.tag("Rows2").d("RowViewModel updateFromNewRowModels ... rowIndex:$index ... oldSize:${rowData.value?.size} ... newSize:${newRowData?.size}")
        if (newRowModel == null || newRowData.isNullOrEmpty()) {
            rowData.value = mutableListOf()
            return
        }
        rowData.value = newRowData!!
        updateBottomPadding()
        updateLoadMoreVisibility()
    }

    private fun updateBottomPadding() { _sectionTitleRowFocused.value = _sectionTitleRowFocused.value }

    private companion object {
        private const val NO_INDEX = -1
    }
}