package com.twentyfouri.tvlauncher.viewmodels

import android.view.View
import androidx.lifecycle.*
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.smartmodel.model.recording.SmartRecordingStatus
import com.twentyfouri.smartmodel.model.recording.SmartRecordingStorage
import com.twentyfouri.smartmodel.model.recording.SmartRecordingType
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.data.ColumnItem
import com.twentyfouri.tvlauncher.data.ColumnItemType
import com.twentyfouri.tvlauncher.data.RecordingsRepository
import com.twentyfouri.tvlauncher.utils.Navigator

class RecordingsViewModel(
    resourceRepository: ResourceRepository,
    val recordingsRepository: RecordingsRepository,
    val navigator: Navigator
) : ViewModel() {

    private val recordings = recordingsRepository.getRecordingsLD()
    val columnItems: LiveData<List<ColumnItem>>
    private val recordingsStorage = recordingsRepository.getRecordingsStorageLD()
    val availableHours: LiveData<String>
    val availableHoursPercentage: LiveData<Int>
    val emptyViewVisibility: LiveData<Int>
    val storageVisibility: LiveData<Int>
    private val _progressVisibility = MutableLiveData<Int>(View.GONE)
    val progressVisibility: LiveData<Int> = _progressVisibility
    private val _recordingsVisibility = MutableLiveData<Int>(View.VISIBLE)
    val recordingsVisibility: LiveData<Int> = _recordingsVisibility
    val recordingRefreshInProgress = recordingsRepository.getRefreshStatus()

    init {
        columnItems = Transformations.map(recordings) { it?.asColumnItems(resourceRepository) ?: arrayListOf() }
        availableHours = Transformations.map(recordingsStorage) { it.getAvailableHours(resourceRepository) }
        availableHoursPercentage = Transformations.map(recordingsStorage) { it.usedSize * 100 / it.totalSize }
        emptyViewVisibility = Transformations.map(columnItems) {
            if (it.isEmpty())
                View.VISIBLE
            else
                View.GONE
        }
        storageVisibility = Transformations.map(columnItems) {
            if (it.isEmpty())
                View.GONE
            else
                View.VISIBLE
        }
    }

    private fun SmartRecordingStorage.getAvailableHours(resourceRepository: ResourceRepository) = when (recordingType) {
        SmartRecordingType.MEDIA_BASED -> ""
        else -> resourceRepository.getString(
            R.string.recordings_settings_available_hours,
            (usedSize / SECONDS_IN_HOUR).toString(),
            (totalSize / SECONDS_IN_HOUR).toString()
        )
    }

    private fun List<SmartMediaItem>.asColumnItems(resourceRepository: ResourceRepository): ArrayList<ColumnItem> {
        val columnItems = ArrayList<ColumnItem>()
        val timeNow = TimeProvider.now().millis
        val recorded = filter { it.endDate != null && it.endDate!!.millis < timeNow && it.recordingData?.recordingStatus == SmartRecordingStatus.SUCCESS }
        val planned = filter { it.startDate != null && it.startDate!!.millis > timeNow && it.recordingData?.recordingStatus == SmartRecordingStatus.WAIT_RECORD }
        columnItems.addAll(recorded.asColumnItems(resourceRepository, R.string.recordings_settings_recordings_label))
        columnItems.addAll(planned.asColumnItems(resourceRepository, R.string.recordings_settings_planned_label))
        return columnItems
    }

    private fun List<SmartMediaItem>.asColumnItems(
        resourceRepository: ResourceRepository,
        labelResourceId: Int
    ): ArrayList<ColumnItem> {
        val columnItems = ArrayList<ColumnItem>()
        if (isNotEmpty()) {
            columnItems.add(
                ColumnItem(
                    type = ColumnItemType.LABEL,
                    text = resourceRepository.getString(labelResourceId, size.toString())
                )
            )
        }
        filterNot { it.type == SmartMediaType.AUTO_RECORDING }
            .sortedByDescending { it.startDate }
                .groupBy { it.seriesReference }
                .forEach { (t, u) ->
                    when {
                        t == null || u.size == 1 -> {
                            u.forEach { v ->
                                columnItems.add(
                                        ColumnItem(
                                                type = ColumnItemType.RECORDING,
                                                recording = v
                                        )
                                )
                            }
                        }
                        else -> {
                            columnItems.add(
                                ColumnItem(
                                    type = ColumnItemType.RECORDING_EPISODES,
                                    recording = u.first(),
                                    recordingEpisodes = u
                                )
                            )
                        }
                    }
                    Unit
                }
        return columnItems
    }

    fun openRecordingDetail(item: ColumnItem) {
        item.recording?.reference?.also { navigator.navigate(SmartNavigationTarget.toDetailPage(it)) }
    }

    fun showProgress() {
        _progressVisibility.postValue(View.VISIBLE)
        _recordingsVisibility.postValue(View.GONE)
    }

    companion object {
        private const val SECONDS_IN_HOUR = 60 * 60
    }

    fun refreshRecordings() {
        recordingsRepository.refreshRecordings()
    }
}