package com.twentyfouri.tvlauncher.viewmodels

import androidx.lifecycle.*
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.data.ColumnItem
import com.twentyfouri.tvlauncher.data.ColumnItemType
import com.twentyfouri.tvlauncher.data.RecordingsRepository
import com.twentyfouri.tvlauncher.utils.Navigator

class RecordingsEpisodesViewModel (
    resourceRepository: ResourceRepository,
    val recordingsRepository: RecordingsRepository,
    private val navigator: Navigator
): ViewModel() {

    private val parentColumnItem = MutableLiveData<ColumnItem>()

    val episodesTitle: LiveData<String>
    val childColumnItems: LiveData<List<ColumnItem>>

    init {
        episodesTitle = Transformations.map(parentColumnItem) {
            getEpisodeTitle(it) ?: resourceRepository.getString(R.string.recordings_episodes, "") //fallback 2
        }
        childColumnItems = Transformations.map(parentColumnItem) { it.recordingEpisodes?.asColumnItems(resourceRepository) ?: arrayListOf()}
    }

    private fun List<SmartMediaItem>.asColumnItems(resourceRepository: ResourceRepository): ArrayList<ColumnItem> {
        val columnItems = ArrayList<ColumnItem>()
        columnItems.add(
                ColumnItem(
                        type = ColumnItemType.LABEL,
                        text = resourceRepository.getString(R.string.recordings_episodes, size.toString()))
        )
        forEach { recording ->
            columnItems.add(
                    ColumnItem(
                            type = ColumnItemType.RECORDING,
                            recording = recording)
            )
        }
        return columnItems
    }

    fun openRecordingDetail(item: ColumnItem) {
        item.recording?.reference?.also {
            navigator.navigate(SmartNavigationTarget.toDetailPage(it))
        }
    }

    private fun getEpisodeTitle(columnItem: ColumnItem): String? =
        if (columnItem.recording?.type == SmartMediaType.RECORDING || columnItem.recording?.type == SmartMediaType.AUTO_RECORDING ){
            columnItem.recording.title
        } else ""

    fun setParentColumnItem(item: ColumnItem) { parentColumnItem.value = item }

    fun refreshRecordings(lifecycleOwner: LifecycleOwner) {
        val seriesRef = parentColumnItem.value?.recording?.seriesReference ?: return
        recordingsRepository.getRecordingsLD().observe(lifecycleOwner, Observer { recordings ->
            if (recordings.isNullOrEmpty().not()) {
                val parentItem = parentColumnItem.value!!
                parentItem.recordingEpisodes = recordings!!
                    .filter {
                        it.seriesReference == seriesRef
                                && it.recordingData?.recordingStatus == parentItem.recording?.recordingData?.recordingStatus
                    }
                    .sortedByDescending { it.startDate }
                setParentColumnItem(parentColumnItem.value!!)
            }
        })
    }
}