package com.twentyfouri.tvlauncher.viewmodels

import android.content.Context
import android.graphics.drawable.Drawable
import android.text.TextUtils
import android.util.Log
import android.view.View
import android.widget.Toast
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Transformations
import androidx.lifecycle.Transformations.map
import androidx.lifecycle.ViewModel
import com.bumptech.glide.load.engine.DiskCacheStrategy
import com.bumptech.glide.request.RequestOptions
import com.twentyfouri.androidcore.utils.EmptyImageSpecification
import com.twentyfouri.androidcore.utils.GlideImageSpecification
import com.twentyfouri.androidcore.utils.ImageSpecification
import com.twentyfouri.smartmodel.model.dashboard.SmartImages
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.media.SmartMediaDetail
import com.twentyfouri.smartmodel.model.media.SmartRestrictionType
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.ui.TvLauncherToast
import com.twentyfouri.tvlauncher.data.DateTimeRepository
import com.twentyfouri.tvlauncher.data.MetadataRepository
import com.twentyfouri.tvlauncher.data.RecordingsRepository
import com.twentyfouri.tvlauncher.extensions.isBrokenLive
import com.twentyfouri.tvlauncher.extensions.isCatchup
import com.twentyfouri.tvlauncher.extensions.isOnAirNow
import com.twentyfouri.tvlauncher.utils.CombinationTransformations
import com.twentyfouri.tvlauncher.utils.ExpirableGlideImageSpecification
import com.twentyfouri.tvlauncher.utils.RecordingsDialogsHelper
import com.twentyfouri.tvlauncher.utils.SingleLiveEvent
import java.util.Timer
import timber.log.Timber
import kotlin.concurrent.fixedRateTimer

class MetadataViewModel(
    private val repository: MetadataRepository,
    private val dateTimeRepository: DateTimeRepository,
    private val recordingsRepository: RecordingsRepository?,
    private val resourceRepository: ResourceRepository
) : ViewModel() {

    private var timer: Timer?
    var iconsViewModel: MediaRestrictionIconsViewModel? = null
    private var useOriginalRecordingState = true

    private val currentTime = MutableLiveData<Long>()
    private val mediaReference = MutableLiveData<SmartMediaReference>()
    private var fallbackMediaReference: SmartMediaReference? = null
    private val whereMetadataAreDisplayed = MutableLiveData(WhereMetadataAreDisplayed.DETAIL)
    private val _userTriggeredRecordingState = MutableLiveData<RecordingState>()
    private val triggerRecordingToast = MutableLiveData<Int>()
    private val isExpanded = MutableLiveData(false)

    val mediaDetail: LiveData<SmartMediaDetail?>
    val imageSpecification: LiveData<ImageSpecification>
    val title: LiveData<String>
    val time: LiveData<String>
    val category: LiveData<String>
    val date: LiveData<String>
    val description: LiveData<String>
    val liveVisibility: LiveData<Int>
    val recordVisibility: LiveData<Int>
    val imageVisibility: LiveData<Int>
    val descriptionMaxLines: LiveData<Int?>
    var recordingState : LiveData<RecordingState?> = SingleLiveEvent()
    val userTriggeredRecordingState: LiveData<RecordingState?> = _userTriggeredRecordingState
    private val recordings: LiveData<List<SmartMediaItem>?>
    val recordText: LiveData<String>
    val recordIconText: LiveData<String>
    val bottomMargin: LiveData<Int>
    val titleTextSize: LiveData<Int>
    val textWidth: LiveData<Int>
    private val endPadding: LiveData<Int>
    val topMargin: LiveData<Int>
    val timeDataVisibility: LiveData<Int>
    val catchupVisibility: LiveData<Int>
    val navigationVisibility: LiveData<Int>
    val infoVisibility: LiveData<Int>
    val channelReference: LiveData<SmartMediaReference?>
    val playability: LiveData<List<PlayabilityState>>
    private val watchingPosition: LiveData<Int>
    val navigationText: LiveData<String>
    val metadataTextFont: LiveData<String>
    val textLineSpacingExtra: LiveData<Int>
    val watchLiveIcon: LiveData<Drawable>
    val maxLines: LiveData<Int>
    val ellipsize: LiveData<TextUtils.TruncateAt?>

    init {
        Timber.tag("leaks").d("MetadataViewModel init")
        currentTime.value = TimeProvider.now().millis
        timer = fixedRateTimer(null, false, TICK_INTERVAL.toLong(), TICK_INTERVAL.toLong()) { currentTime.postValue(TimeProvider.now().millis) }
        mediaDetail = CombinationTransformations.switchMap(mediaReference) {
            if (Flavor().getIsRequestedInChannelRow(whereMetadataAreDisplayed.value)) {
                MutableLiveData()
            } else {
                val shouldLoadFullDetail = when (whereMetadataAreDisplayed.value) {
                    WhereMetadataAreDisplayed.EPG -> false
                    else -> true
                }
                repository.getMediaDetailLD(it, fallbackMediaReference, shouldLoadFullDetail)
            }
        }
        imageSpecification = CombinationTransformations.combineNullable(mediaDetail, whereMetadataAreDisplayed) { mediaDetail, whereMetadataAreDisplayed ->
            val appChannelDelegate = Flavor().getAppChannelsDelegate()
            val mediaRefValue = mediaReference.value

            (if (appChannelDelegate != null && mediaRefValue != null) {
                appChannelDelegate.getAppChannelSpecificImage(mediaRefValue, resourceRepository)
            } else null) ?: getImageSpecification(whereMetadataAreDisplayed, mediaDetail)
        }
        title = CombinationTransformations.combineNullable(mediaDetail, whereMetadataAreDisplayed) { detail, display ->
            //TODO do this differently (lines attribute)
            when {
                detail == null -> ""
                display != WhereMetadataAreDisplayed.DETAIL && detail.title?.length ?: 0 > 40 -> detail.title?.substring(0, 40) + "..."
                else -> detail.title ?: ""
            }
        }
        time = map(mediaDetail) { dateTimeRepository.formatDateRange(it?.startDate, it?.endDate) }
        category = map(mediaDetail) { it?.categories?.firstOrNull()?.label ?: "" }
        date = map(mediaDetail) { dateTimeRepository.formatDateTime(it?.startDate, it?.endDate) }
        description = map(mediaDetail) { it?.description ?: resourceRepository.getString(R.string.epg_catchup_unavailable) }
        liveVisibility = CombinationTransformations.combineNullable(mediaDetail, currentTime) { mediaDetail, _ ->
            mediaDetail?.let { if (dateTimeRepository.isLive(it.startDate, it.endDate)) View.VISIBLE else View.GONE } ?: View.GONE
        }
        imageVisibility = map(whereMetadataAreDisplayed) { if (it == WhereMetadataAreDisplayed.EPG) View.VISIBLE else View.GONE }
        descriptionMaxLines = CombinationTransformations.combineNullable(whereMetadataAreDisplayed, title) { where, title ->
            when {
                where == WhereMetadataAreDisplayed.DETAIL && title != null && title.length <= STANDARD_TITLE_LENGTH ->
                    resourceRepository.getInteger(R.integer.detail_view_detail_description_max_lines)
                else -> resourceRepository.getInteger(R.integer.detail_view_non_detail_description_max_lines)
            }
        }
        recordings = Transformations.switchMap(whereMetadataAreDisplayed) {
//            Log.d("leaks","MetadataViewModel recordings set")
            when (it) {
                WhereMetadataAreDisplayed.CHANNEL_ROW,
                WhereMetadataAreDisplayed.ROW -> MutableLiveData<List<SmartMediaItem>>()
                else -> recordingsRepository?.getRecordingsLD()
            }
        }

        recordingState = CombinationTransformations.zipCombineNullable(mediaDetail, recordings) { mediaDetail, recordings ->
            useOriginalRecordingState = true
            when {
                Flavor().isRecordingAllowed.not() -> RecordingState.NOTHING
                mediaDetail == null -> RecordingState.NOTHING
                recordings == null -> RecordingState.NOTHING
                mediaDetail.isRecordingAllowed().not() -> RecordingState.NOTHING
                else -> mediaDetail.getRecordingState(recordings)
            }
        }

        recordVisibility = CombinationTransformations.combineNullable(whereMetadataAreDisplayed, recordingState) { whereMetadataAreDisplayed, recordingState ->
            when {
                whereMetadataAreDisplayed != WhereMetadataAreDisplayed.EPG -> View.GONE
                recordingState == RecordingState.CAN_BE_RECORDED
                        || recordingState == RecordingState.CAN_BE_AUTO_RECORDED
                        || recordingState == RecordingState.WILL_BE_RECORDED
                        || recordingState == RecordingState.WILL_BE_AUTO_RECORDED -> View.VISIBLE
                else -> View.GONE
            }
        }
        recordText = CombinationTransformations.map(
                recordingState,
                { getRecordText(it)},
                userTriggeredRecordingState,
                { getRecordText(it) }
        )
        recordIconText = CombinationTransformations.map(
                recordingState,
                { getRecordIconText(it) },
                userTriggeredRecordingState,
                { getRecordIconText(it) }
        )
        bottomMargin = map(whereMetadataAreDisplayed) {
            when (it) {
                WhereMetadataAreDisplayed.EPG -> resourceRepository.getDimensionPixelSize(R.dimen.epg_view_metadata_bottom_offset)
                else -> 0
            }
        }
        titleTextSize = map(whereMetadataAreDisplayed) {
            when (it) {
                WhereMetadataAreDisplayed.DETAIL -> resourceRepository.getDimensionPixelSize(R.dimen.metadata_title_text_size_big)
                else -> resourceRepository.getDimensionPixelSize(R.dimen.metadata_title_text_size)
            }
        }
        textWidth = map(whereMetadataAreDisplayed) {
            when (it) {
                WhereMetadataAreDisplayed.DETAIL -> resourceRepository.getDimensionPixelSize(R.dimen.metadata_width_big)
                else -> resourceRepository.getDimensionPixelSize(R.dimen.metadata_width)
            }
        }
        endPadding = map(whereMetadataAreDisplayed) {
            when (it) {
                WhereMetadataAreDisplayed.DETAIL -> resourceRepository.getDimensionPixelSize(R.dimen.metadata_padding_end)
                else -> 0
            }
        }
        topMargin = map(whereMetadataAreDisplayed) {
            when (it) {
                WhereMetadataAreDisplayed.DETAIL -> resourceRepository.getDimensionPixelSize(R.dimen.metadata_margin_top)
                else -> 0
            }
        }
        catchupVisibility = CombinationTransformations.combineNullable(mediaDetail, currentTime) { detail, _ ->
            if (detail?.isCatchup() == true) View.VISIBLE else View.GONE
        }
        timeDataVisibility = map(mediaDetail) {
            when (it?.type) {
                SmartMediaType.LIVE_EVENT,
                SmartMediaType.RECORDING -> View.VISIBLE
                else -> View.GONE
            }
        }
        navigationVisibility = CombinationTransformations.combineNullable(whereMetadataAreDisplayed, recordingState) { whereMetadataAreDisplayed, recordingState ->
            when {
                whereMetadataAreDisplayed == WhereMetadataAreDisplayed.EPG
                        && recordingState != RecordingState.CAN_BE_RECORDED
                        && recordingState != RecordingState.CAN_BE_AUTO_RECORDED
                        && recordingState != RecordingState.WILL_BE_RECORDED
                        && recordingState != RecordingState.WILL_BE_AUTO_RECORDED -> View.VISIBLE
                else -> View.GONE
            }
        }
        infoVisibility = CombinationTransformations.combineNullable(navigationVisibility, recordVisibility) { navigation, record ->
            when {
                (navigation == View.VISIBLE || record == View.VISIBLE) && Flavor().canInfoButtonBeVisible() -> View.VISIBLE
                else -> View.GONE
            }
        }
        channelReference = map(mediaDetail) { it?.channelReference }
        playability = CombinationTransformations.combineNullable(mediaDetail, userTriggeredRecordingState, currentTime) { mediaDetail, recordingState, _ ->
            val states = ArrayList<PlayabilityState>()
            when (mediaDetail?.type) {
                SmartMediaType.EPISODE,
                SmartMediaType.MOVIE,
                SmartMediaType.SERIES -> states.add(PlayabilityState.VOD)
                else -> {
                    if (recordingState == RecordingState.WAS_RECORDED || recordingState == RecordingState.WAS_AUTO_RECORDED) states.add(PlayabilityState.WAS_RECORDED)
                    if (mediaDetail?.isCatchup() == true) states.add(PlayabilityState.IS_CATCHUP)
                    if (mediaDetail?.isOnAirNow() == true) states.add(PlayabilityState.IS_LIVE)
                    if (mediaDetail?.isBrokenLive() == true) states.add(PlayabilityState.BROKEN_LIVE)
                }
            }
            states
        }
        watchingPosition = map(mediaDetail) { it?.watchingPosition }
        navigationText = map(playability) {
            when {
                it.contains(PlayabilityState.IS_LIVE) -> resourceRepository.getString(R.string.navigation_live)
                mediaDetail.value?.isInFuture() == true -> resourceRepository.getString(R.string.navigation_show_detail)
                else -> resourceRepository.getString(R.string.navigation_look_back)
            }
        }
        watchLiveIcon = map(playability){
            when {
                it.contains(PlayabilityState.IS_LIVE) ->
                    Flavor().getWatchLiveIcon(resourceRepository) ?: resourceRepository.getDrawable(R.drawable.remote_control_cross_ok)
                else -> resourceRepository.getDrawable(R.drawable.remote_control_cross_ok)
            }
        }
        metadataTextFont = map(whereMetadataAreDisplayed) {
            when (it) {
                WhereMetadataAreDisplayed.DETAIL -> resourceRepository.getString(R.string.font_path_regular)
                else -> resourceRepository.getString(R.string.font_path_semibold)
            }
        }
        textLineSpacingExtra = map(whereMetadataAreDisplayed) {
            when(it) {
                WhereMetadataAreDisplayed.DETAIL -> resourceRepository.getDimensionPixelSize(R.dimen.metadata_text_line_spacing)
                else -> 0
            }
        }
        maxLines = map(isExpanded) {
            when {
                it && isDisplayedInDetail() -> Integer.MAX_VALUE
                !it && (isDisplayedInDetail() || isDisplayedInRow()) && (title.value?.length?.compareTo(STANDARD_TITLE_LENGTH) ?: -1) >= 0
                    -> resourceRepository.getInteger(R.integer.detail_view_non_detail_description_max_lines)
                else -> resourceRepository.getInteger(R.integer.detail_view_non_detail_description_max_lines)
            }
        }
        ellipsize = map(isExpanded) {
            when {
                it && isDisplayedInDetail() -> null
                !it && (isDisplayedInDetail() || isDisplayedInRow()) -> TextUtils.TruncateAt.END
                else -> TextUtils.TruncateAt.END
            }
        }
    }

    private fun getImageSpecification(whereMetadataAreDisplayed: WhereMetadataAreDisplayed?, mediaDetail: SmartMediaDetail?): ImageSpecification {
        val width = when(whereMetadataAreDisplayed) {
            WhereMetadataAreDisplayed.EPG -> 380
            WhereMetadataAreDisplayed.DETAIL -> 1920
            WhereMetadataAreDisplayed.ROW -> 480
            WhereMetadataAreDisplayed.CHANNEL_ROW -> 380
            else -> 380
        }
        val height = when(whereMetadataAreDisplayed) {
            WhereMetadataAreDisplayed.EPG -> 214
            WhereMetadataAreDisplayed.DETAIL -> 1080
            WhereMetadataAreDisplayed.ROW -> 270
            WhereMetadataAreDisplayed.CHANNEL_ROW -> 214
            else -> 214
        }

        val imageUrl = Flavor().pickBasedOnFlavor(mediaDetail?.images, width, height, SmartImages.DifferenceProperty(45))
        if (imageUrl.isNullOrEmpty()) return EmptyImageSpecification()

        return when (whereMetadataAreDisplayed) {
            WhereMetadataAreDisplayed.EPG -> ExpirableGlideImageSpecification(imageUrl, RequestOptions().placeholder(repository.getEpgImagePlaceholderId()))
            WhereMetadataAreDisplayed.DETAIL -> GlideImageSpecification(imageUrl, getDetailRequestOptions())
            else -> ExpirableGlideImageSpecification(imageUrl)
        }
    }

    private fun getDetailRequestOptions() = RequestOptions()
        .diskCacheStrategy(DiskCacheStrategy.NONE) // because file name is always same
        .skipMemoryCache(true)

    private fun getRecordIconText(it: RecordingState?) = when (it) {
        RecordingState.CAN_BE_RECORDED,
        RecordingState.CAN_BE_AUTO_RECORDED,
        RecordingState.WILL_BE_RECORDED,
        RecordingState.WILL_BE_AUTO_RECORDED -> resourceRepository.getString(R.string.record_icon)
        else -> ""
    }

    private fun getRecordText(it: RecordingState?) = when (it) {
        RecordingState.CAN_BE_RECORDED,
        RecordingState.CAN_BE_AUTO_RECORDED -> resourceRepository.getString(R.string.detail_start_button)
        RecordingState.WILL_BE_RECORDED,
        RecordingState.WILL_BE_AUTO_RECORDED-> resourceRepository.getString(R.string.detail_stop_button)
        else -> ""
    }

    private fun SmartMediaDetail.getRecordingState(recordings: List<SmartMediaItem>): RecordingState {
        val myRecording = recordings.find { belongsToRecording(it) } //this checks only the non-auto recordings
        if (myRecording != null) mediaDetail.value?.recordingReference = myRecording.reference
        val isInFuture = isInFuture()
        val hasAutoRecording = myRecording?.recordingReference != null || recordings.any { belongsToAutoRecording(it) } //check for auto-recording
        val nowMs = TimeProvider.nowMs()
        val isInPast = isInPast(nowMs)
        return when {
            myRecording != null && isInFuture && hasAutoRecording -> RecordingState.WILL_BE_AUTO_RECORDED
            myRecording != null && isInFuture -> RecordingState.WILL_BE_RECORDED
            myRecording != null && isInPast && (hasAutoRecording || areThereMoreThan1RecordedEpisodes(recordings, nowMs)) -> RecordingState.WAS_AUTO_RECORDED
            myRecording != null && isInPast -> RecordingState.WAS_RECORDED
            myRecording == null && isInFuture && seriesReference != null && hasAutoRecording.not() -> RecordingState.CAN_BE_AUTO_RECORDED
            myRecording == null && isInFuture -> RecordingState.CAN_BE_RECORDED
            else -> RecordingState.NOTHING
        }
    }

    private fun SmartMediaItem.areThereMoreThan1RecordedEpisodes(recordings: List<SmartMediaItem>, nowMs: Long): Boolean {
        return recordings.count { it.isInPast(nowMs) && it.seriesReference != null && it.seriesReference == seriesReference } > 1
    }

    private fun SmartMediaDetail.isRecordingAllowed(): Boolean =
        restrictions.find { it.type == SmartRestrictionType.RECORDING } == null

    private fun SmartMediaItem.isInPast(nowMs: Long = TimeProvider.nowMs()): Boolean {
        return endDate?.millis ?: 0 < nowMs
    }

    private fun SmartMediaDetail.isInFuture(): Boolean {
        return if (Flavor().isLiveRecordingAllowed)
            (endDate?.millis ?: 0 > TimeProvider.now().millis)
        else
            (startDate?.millis ?: 0 > TimeProvider.now().millis)
    }

    private fun SmartMediaDetail.belongsToRecording(recording: SmartMediaItem): Boolean {
        return when {
            type == SmartMediaType.RECORDING -> recording.reference == reference
            recording.type == SmartMediaType.AUTO_RECORDING -> false
            else -> Flavor().compareRecordingWithEvent(recording, this)
        }
    }

    private fun SmartMediaItem.belongsToAutoRecording(recording: SmartMediaItem): Boolean {
        if (recording.type != SmartMediaType.AUTO_RECORDING) return false
        if (seriesReference == null) return false
        return seriesReference == recording.seriesReference
    }

    private fun isDisplayedInDetail() = whereMetadataAreDisplayed.value == WhereMetadataAreDisplayed.DETAIL

    private fun isDisplayedInRow() = when(whereMetadataAreDisplayed.value) {
        WhereMetadataAreDisplayed.ROW,
        WhereMetadataAreDisplayed.CHANNEL_ROW -> true
        else -> false
    }

    fun setObserversInContext(context: Context) {
        mediaDetail.observe(context as LifecycleOwner) { detail -> iconsViewModel?.setContentIcons(detail?.ageRatings) }
        triggerRecordingToast.observe(context as LifecycleOwner) { resId ->
            if (resId != null)
                TvLauncherToast.makeText(
                    context,
                    resourceRepository.getString(resId),
                    Toast.LENGTH_SHORT
                )?.show()
        }
    }

    fun startStopRecording(lifecycleOwner: LifecycleOwner) {
        when (if (useOriginalRecordingState) recordingState.value else userTriggeredRecordingState.value) {
            RecordingState.CAN_BE_AUTO_RECORDED -> RecordingsDialogsHelper.showPlanSeriesRecordingDialog(
                resourceRepository,
                { startRecording(lifecycleOwner, false) },
                { startRecording(lifecycleOwner, true) }
            )
            RecordingState.CAN_BE_RECORDED -> startRecording(lifecycleOwner, false)
            RecordingState.WILL_BE_AUTO_RECORDED -> RecordingsDialogsHelper.showUnplanSeriesRecordingDialog(
                resourceRepository,
                { stopRecording(lifecycleOwner, false) },
                { stopRecording(lifecycleOwner, true) }
            )
            RecordingState.WILL_BE_RECORDED -> stopRecording(lifecycleOwner, false)
            RecordingState.WAS_AUTO_RECORDED -> if (recordingsRepository?.isAutoRecordingSingleEpisode(seriesReference = mediaDetail.value?.seriesReference) == true) {
                deleteRecording(lifecycleOwner, false)
            } else RecordingsDialogsHelper.showDeleteSeriesRecordingDialog(
                resourceRepository,
                { deleteRecording(lifecycleOwner, false) },
                { deleteRecording(lifecycleOwner, true) }
            )
            RecordingState.WAS_RECORDED -> deleteRecording(lifecycleOwner, false)
            RecordingState.NOTHING -> triggerRecordingToast.postValue(R.string.epg_recordings_unavailable)
            else -> _userTriggeredRecordingState.postValue(recordingState.value)
        }
        useOriginalRecordingState = false
    }

    private fun startRecording(lifecycleOwner: LifecycleOwner, autoRecording: Boolean) {
        recordingsRepository?.let { recordingsRepo ->
            recordingsRepo.startRecordingLD(
                mediaDetail.value,
                autoRecording,
                catchBlock = recordingsRepo.getStorageFullCatchBlock(resourceRepository)
            ).observe(lifecycleOwner) {
                triggerRecordingToast.postValue(R.string.epg_recording_planned)
                _userTriggeredRecordingState.postValue(
                    if (autoRecording) RecordingState.WILL_BE_AUTO_RECORDED
                    else RecordingState.WILL_BE_RECORDED
                )
                mediaDetail.value?.recordingReference =
                    if (autoRecording) it.recordingReference else it.reference
            }
        }
    }

    private fun stopRecording(lifecycleOwner: LifecycleOwner, autoRecording: Boolean) {
        recordingsRepository?.stopRecordingLD(mediaDetail.value, autoRecording, false)?.observe(
            lifecycleOwner
        ) {
            triggerRecordingToast.postValue(R.string.epg_recording_cancelled)
            _userTriggeredRecordingState.postValue(
                if (autoRecording) {
                    RecordingState.CAN_BE_AUTO_RECORDED
                } else {
                    mediaDetail.value?.getRecordingState(emptyList())
                }
            )
        }
    }

    private fun deleteRecording(lifecycleOwner: LifecycleOwner, autoRecording: Boolean) {
        recordingsRepository?.stopRecordingLD(mediaDetail.value, autoRecording, true)?.observe(
            lifecycleOwner
        ) {
            _userTriggeredRecordingState.postValue(RecordingState.NOTHING)
        }
    }

    fun setMediaReference(reference: SmartMediaReference, fallbackReference: SmartMediaReference?) {
        fallbackMediaReference = fallbackReference
        mediaReference.value = reference
    }

    fun setWhereDisplayed(where: WhereMetadataAreDisplayed) { whereMetadataAreDisplayed.value = where }

    fun setExpanded(boolean: Boolean) { isExpanded.value = boolean }

    override fun onCleared() {
        timer?.cancel()
        timer = null
        super.onCleared()
        Timber.tag("leaks").d("MetadataViewModel onCleared")
    }

    fun onDestroyView() {
        onCleared()
    }

    enum class WhereMetadataAreDisplayed {
        ROW,
        EPG,
        DETAIL,
        CHANNEL_ROW
    }

    enum class RecordingState {
        CAN_BE_RECORDED,
        CAN_BE_AUTO_RECORDED,
        WILL_BE_RECORDED,
        WILL_BE_AUTO_RECORDED,
        WAS_RECORDED,
        WAS_AUTO_RECORDED,
        NOTHING
    }

    enum class PlayabilityState {
        VOD,
        WAS_RECORDED,
        IS_LIVE,
        IS_CATCHUP,
        BROKEN_LIVE
    }

    private companion object {
        private const val TICK_INTERVAL = 10 * 1000
        private const val STANDARD_TITLE_LENGTH = 40
        private const val TAG = "MetadataViewModel"
    }
}