package com.twentyfouri.tvlauncher.viewmodels

import android.content.res.ColorStateList
import android.view.View
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Transformations
import androidx.lifecycle.ViewModel
import com.twentyfouri.smartmodel.model.menu.SmartMenuItem
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.MenuType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_UI_LOG
import com.twentyfouri.tvlauncher.data.ContentResolverRepository
import com.twentyfouri.tvlauncher.data.HardcodedMenuMapper
import com.twentyfouri.tvlauncher.utils.CombinationTransformations
import com.twentyfouri.tvlauncher.utils.Navigator
import timber.log.Timber

class MenuItemViewModel(
    var title: String = "title",
    var target: SmartNavigationTarget? = null,
    private val resourceRepository: ResourceRepository
) : ViewModel() {

    private val _isSelected = MutableLiveData<Boolean>().apply { value = false }
    private val _isFocused = MutableLiveData<Boolean>().apply { value = false }
    private val menuItem = MutableLiveData<SmartMenuItem>()
    private lateinit var contentResolverRepository: ContentResolverRepository

    val isSelected: LiveData<Boolean> = _isSelected
    val isFocused: LiveData<Boolean> = _isFocused
    val icon: LiveData<Int>
    val innerTextColor: LiveData<ColorStateList>
    private val isEnabled: LiveData<Boolean>
    private val isUnread: LiveData<Boolean>
    val innerText: LiveData<String>
    val visibility: LiveData<Int>

    init {
        isUnread = Transformations.switchMap(menuItem) {
            if (it.id == HardcodedMenuMapper.Type.NOTIFICATIONS.name) contentResolverRepository.getHasUnreadLD()
            else MutableLiveData<Boolean>().apply { value = false }
        }
        icon = CombinationTransformations.combineNonNullable(menuItem, isUnread) { menuItem, isUnread ->
            getIconFor(menuItem, isUnread)
        }
        isEnabled = Transformations.switchMap(menuItem) {
            if (it.id == HardcodedMenuMapper.Type.NOTIFICATIONS.name) contentResolverRepository.getEnabledLD()
            else MutableLiveData<Boolean>().apply { value = true}
        }
        innerText = Transformations.switchMap(menuItem) {
            if (it.id == HardcodedMenuMapper.Type.NOTIFICATIONS.name) contentResolverRepository.getInnerTextLD()
            else MutableLiveData<String>().apply { value = ""}
        }
        innerTextColor = Transformations.map(isUnread) {
            if (it) {
                ColorStateList.valueOf(resourceRepository.getColor(R.color.notification_text_unseen_color))
            } else {
                resourceRepository.getColorStateList(R.color.text_color_selector_2)
            }
        }
        visibility = Transformations.map(isEnabled) { if (it) View.VISIBLE else View.GONE }
    }

    fun getOnClick() = View.OnClickListener {
        Timber.tag(TAG_UI_LOG).d("Selected menu: $title")
        Navigator.getInstance().navigate(target)
    }

    fun getOnFocusChangeListener() = View.OnFocusChangeListener { _: View?, hasFocus: Boolean ->
        if (hasFocus) Timber.tag(TAG_UI_LOG).d("Focused menu: $title")
        _isFocused.postValue(hasFocus)
    }

    fun setFromMenuItem(menuItem: SmartMenuItem, contentResolverRepository: ContentResolverRepository) {
        this.contentResolverRepository = contentResolverRepository
        this.menuItem.value = menuItem
    }

    fun setIsSelected(boolean: Boolean) { _isSelected.value = boolean }

    fun setIsFocused(boolean: Boolean) { _isFocused.value = boolean }

    private fun getIconFor(menuItem: SmartMenuItem, isUnread: Boolean): Int {
        return when (Flavor().getMenuType(menuItem.id)) {
            MenuType.SEARCH -> R.drawable.ic_search
            MenuType.NOTIFICATIONS -> if (isUnread) R.drawable.ic_notifications_bg_unread else R.drawable.ic_notifications_bg
            MenuType.VOICE -> R.drawable.ic_voice
            MenuType.PERSONAL -> R.drawable.ic_account
            MenuType.SETTINGS -> R.drawable.ic_settings
            MenuType.DIAGNOSTICS -> R.drawable.ic_diagnostics
            else -> 0
        }
    }
}