package com.twentyfouri.tvlauncher.viewmodels

import android.content.Context
import android.os.Handler
import android.view.View
import android.view.animation.Animation
import android.view.animation.AnimationUtils
import androidx.fragment.app.commit
import androidx.fragment.app.setFragmentResultListener
import androidx.lifecycle.ViewModel
import com.twentyfouri.smartmodel.model.dashboard.SmartPlaylistFilter
import com.twentyfouri.smartmodel.model.dashboard.SmartPlaylistFilterReference
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.data.ListPickerItem
import com.twentyfouri.tvlauncher.extensions.isToday
import com.twentyfouri.tvlauncher.ui.ListPickerFragment
import com.twentyfouri.tvlauncher.ui.MainActivity
import com.twentyfouri.tvlauncher.ui.RowPageFragment
import com.twentyfouri.tvlauncher.widgets.SettingsItemType
import org.joda.time.DateTime

open class FilterViewModel(
    val type: Type,
    val filters: List<SmartPlaylistFilter>,
    val selectedFiltersReferences: Collection<SmartPlaylistFilterReference>,
    val resourceRepository: ResourceRepository
) : ViewModel() {

    enum class Type {
        CHANNEL,
        DATE,
        INITIAL,
        CONTENT,
        RESET,
        NONE
    }

    var isFirst = false

    //values used to prevent unwanted animation when filters row is focused and logical item is focused but focus quickly jump to last focused filter item
    private var lastFocusState = false
    private var isScaledIn = false
    fun getOnFocusChangedListener() = View.OnFocusChangeListener { v, hasFocus ->
        lastFocusState = hasFocus
        val animId = if (isFirst) {
            if (hasFocus) R.anim.scale_in_card_from_left else R.anim.scale_out_card_from_left
        } else {
            if (hasFocus) R.anim.scale_in_card_from_center else R.anim.scale_out_card_from_center
        }
        val anim = AnimationUtils.loadAnimation(v?.context, animId)
        v?.elevation = if(hasFocus) 5f else 0f //value just to bring focused button higher then others in linear layout
        doAnimation(v, anim, hasFocus)
    }

    private fun doAnimation(view: View?, animation: Animation, hasFocus: Boolean) {
        Handler().post {
            if(lastFocusState == hasFocus && isScaledIn != hasFocus) {
                view?.startAnimation(animation)
                animation.fillAfter = true
                isScaledIn = hasFocus
            }
        }
    }

    protected fun Type.getAllTitle(): String = when (this) {
        Type.CHANNEL -> resourceRepository.getString(R.string.filter_channels_all)
        Type.DATE -> resourceRepository.getString(R.string.filter_date_all)
        Type.INITIAL -> resourceRepository.getString(R.string.filter_initial_all)
        Type.CONTENT -> resourceRepository.getString(R.string.filter_content_all)
        else -> ""
    }

    private fun Type.getDefaultTitle(): String = when (this) {
        Type.CHANNEL -> resourceRepository.getString(R.string.filter_channels)
        Type.DATE -> resourceRepository.getString(R.string.filter_date)
        Type.INITIAL -> resourceRepository.getString(R.string.filter_initial)
        Type.CONTENT -> resourceRepository.getString(R.string.filter_content)
        else -> ""
    }

    open fun getTitle(): String {
        val selectedFilterReference = selectedFiltersReferences.find { Flavor().getFilterType(it) == type }
        val selectedFilter = filters.find { it.reference == selectedFilterReference }
        return selectedFilter?.reference?.title ?: type.getAllTitle()
    }

    fun getOnClick() = View.OnClickListener {
        val context = it.context
        if (type == Type.RESET) {
            resetSelectedFilters(context)
            return@OnClickListener
        }
        val selectedFilterReference: SmartPlaylistFilterReference? = selectedFiltersReferences.find { Flavor().getFilterType(it) == type }
        var currentItem = ListPickerFragment.DEFAULT_POSITION_TOP
        val data = ArrayList<ListPickerItem>().apply {
            add(
                ListPickerItem(
                    id = type.name,
                    label = type.getAllTitle(),
                    title = "",
                    subtitle = "",
                    type = SettingsItemType.LABEL
                )
            )
            for (aFilter in filters) {
                (aFilter.reference == selectedFilterReference).ifTrue { currentItem = size }
                add(getPickerItem(aFilter))
            }
            if (currentItem == ListPickerFragment.DEFAULT_POSITION_TOP && type == Type.DATE) {
                currentItem =
                    indexOf(
                        find { listItem ->
                            filters.find { filter ->
                                DateTime.parse(filter.reference.title).isToday
                            }?.reference?.hashCode().toString() == listItem.id
                        }
                    )
            }

        }
        val fragment = ListPickerFragment.newInstance(
            title = type.getDefaultTitle(),
            defaultPosition = currentItem,
            data = data,
            selectedPosition = ListPickerFragment.SELECTED_POSITION_MIDDLE
        )
        (context as? MainActivity)?.supportFragmentManager?.commit {
            add(R.id.fullscreen_frame, fragment, ListPickerFragment::class.java.simpleName)
            addToBackStack(null)
        }
        ((context as? MainActivity)?.supportFragmentManager?.findFragmentById(R.id.content_frame) as? RowPageFragment)?.also { rpf ->
            fragment.setFragmentResultListener(ListPickerFragment.REQUEST_KEY) { _, bundle ->
                (bundle.get(ListPickerFragment.BUNDLE_KEY_RESULT) as? ListPickerItem)?.let { listPickerItem ->
                    rpf.onPicked(listPickerItem)
                }
            }
            rpf.saveCurrentFocus()
        }
    }

    protected open fun getPickerItem(filter: SmartPlaylistFilter): ListPickerItem {
        return ListPickerItem(
            id = filter.reference.hashCode().toString(),
            label = filter.reference.title,
            title = "",
            subtitle = "",
            type = SettingsItemType.LABEL
        )
    }

    private fun resetSelectedFilters(context: Context) {
        val item = ListPickerItem(
            id = Type.RESET.name,
            label = "",
            title = "",
            subtitle = "",
            type = SettingsItemType.LABEL
        )
        ((context as? MainActivity)?.supportFragmentManager?.findFragmentById(R.id.content_frame) as? RowPageFragment)?.also { rpf ->
            rpf.onPicked(item)
        }
    }
}