package com.twentyfouri.tvlauncher.viewmodels

import android.content.Context
import android.graphics.Rect
import android.util.Log
import android.view.View
import androidx.lifecycle.*
import com.twentyfouri.androidcore.epg.model.EpgChannel
import com.twentyfouri.androidcore.epg.model.EpgData
import com.twentyfouri.androidcore.epg.model.EpgEvent
import com.twentyfouri.androidcore.epg.model.EpgEventLabel
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.media.SmartMediaDetail
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.data.*
import com.twentyfouri.tvlauncher.data.PopulateDatabaseWorker.Companion.Progress
import com.twentyfouri.tvlauncher.extensions.addDataSuperSafely
import com.twentyfouri.tvlauncher.extensions.getEventsClone
import org.joda.time.DateTime
import java.util.*
import kotlin.collections.ArrayList

class EpgViewModel(
    private val repository: EpgRepository,
    private val recordingsRepository: RecordingsRepository,
    private val resourceRepository: ResourceRepository,
    context: Context
) : ViewModel() {

    private val focusedEvent = MutableLiveData<EpgEvent?>().apply { value = null }
    private val _epgData = MutableLiveData<EpgData>()
    private val _datePickText = MutableLiveData<String>().apply { value = "" }

    val descriptionVisibility: LiveData<Int>
    val epgData: LiveData<EpgData> = _epgData
    val datePickText: LiveData<String> = _datePickText
    val channelNumbers = mutableMapOf<Int, Int>()

    var epgAbsoluteStartTime = DateTime()
    var epgAbsoluteEndTime = DateTime()
    var currentDay: Date? = null
    val databaseProgress: LiveData<Int>
    val isDatabasePopulated: LiveData<Boolean>
    val epgLoaderVisibility: LiveData<Boolean>

    init {
        repository.loadChannelsForEpg(LOADER_INDICATOR_DELAY) { channels ->
            var position = 0
            val converted: List<EpgChannel> = channels.map {
                EpgChannelExt(it, resourceRepository).apply {
                    channelNumbers[it.channelNumber] = position++
                    channelNumber = it.channelNumber
                }
            }
            //Log.d("maxiEpg", "epgData loadChannelsForEpg posted")
            _epgData.postValue(EpgData(ArrayList(converted)))
        }

        descriptionVisibility = Transformations.map(focusedEvent) { when(it) { null -> View.GONE else -> View.VISIBLE } }

        databaseProgress = Transformations.map(PopulateDatabaseWorkManager.getWorkInfosLD(context)) {
            it.firstOrNull()
                ?.progress?.getInt(Progress, 100)
                ?: 100
        }

        isDatabasePopulated = Transformations.map(databaseProgress) { it == 100 }

        epgLoaderVisibility = Transformations.map(isDatabasePopulated){ it }
    }

    fun getChannel(channelPosition: Int): EpgChannel? {
        //uncomment if you want to have just one channel in EPG
        //if (channelPosition > 2) return null
        if (channelPosition < 0 || channelPosition >= _epgData.value?.channelCount ?: 0) return null
        return _epgData.value?.getChannel(channelPosition)
    }

    fun setFocusedEvent(event: EpgEvent?) { focusedEvent.value = event }

    private fun print(channels: List<SmartMediaItem>, fromTime: DateTime, toTime: DateTime): String
            = "ch:${channels.joinToString(separator = ",") { it.channelNumber.toString() }}:s:${fromTime.hourOfDay}:${fromTime.minuteOfHour}:e:${toTime.hourOfDay}:${toTime.minuteOfHour}"

    private fun print(date: DateTime?): String {
        if (date == null) return "00:00"
        val a = date.hourOfDay
        val b = if (a < 10) "0$a" else a.toString()
        val c = date.minuteOfHour
        val d = if (c < 10) "0$c" else c.toString()
        return "$b:$d"
    }

    fun loadEvents(channels: List<SmartMediaItem>, fromTime: Long, toTime: Long) {
        repository.loadListOfListOfEpgEventsForEpg(
                channels,
                DateTime(fromTime),
                DateTime(toTime)
        ) { events ->
//            Log.d("maxiEpg", "loadListOfListOfEpgEventsForEpg collector " +
//                    events.joinToString(separator = " # ") { eventsOnChannel ->
//                        "${eventsOnChannel.firstOrNull()?.channelNumber.toString()}," +
//                                "${print(eventsOnChannel.minByOrNull { e -> e.startDate?.millis ?: 0 }?.startDate)}," +
//                                print(eventsOnChannel.maxByOrNull { e -> e.endDate?.millis ?: 0 }?.endDate)
//                    })
//            events
//                .firstOrNull { it.firstOrNull()?.channelNumber == 4 } //filter just one channel
//                ?.sortedBy { it.startDate }
//                ?.forEach { Log.d("maxiEpg", "channel:${it.channelNumber}, ${print1(it.startDate!!.millis, it.endDate!!.millis)} collector") }

            val recordings = recordingsRepository.getRecordingsLD().value
            val newEpgData3 = _epgData.value!!.addDataSuperSafely(
                channels = channels,
                events = events,
                fromTime = fromTime,
                toTime = toTime,
                getRecordingDrawableBlock = { smartMediaItem -> recordings?.let { getRecordingDrawable(recordings, smartMediaItem) } },
                epgAbsoluteStartTime = epgAbsoluteStartTime,
                epgAbsoluteEndTime = epgAbsoluteEndTime
            )

            _epgData.postValue(newEpgData3)
        }
    }

    private fun getRecordingDrawable(
        recordings: List<SmartMediaItem>,
        event: SmartMediaItem
    ): EpgEventLabel? {
        if (recordings.any { aRecording -> Flavor().compareRecordingWithEvent(aRecording, event) })
            return getRecordingEpgEventLabel()
        return null
    }

    fun getRecordingEpgEventLabel() = EpgEventLabel(
        EpgEventExt.LABEL_ID_RECORDINGS,
        resourceRepository.getDrawable(R.drawable.ic_record_red),
        resourceRepository.getDimensionPixelSize(R.dimen.epg_view_recording_dot),
        resourceRepository.getDimensionPixelSize(R.dimen.epg_view_recording_dot),
        Rect(
            resourceRepository.getDimensionPixelSize(R.dimen.epg_view_recording_dot_margin),
            0,
            resourceRepository.getDimensionPixelSize(R.dimen.epg_view_recording_dot_margin),
            0
        )
    )

    fun clearRanges() {
        for (epgChannelIndex in 0 until (_epgData.value?.channelCount ?: 0)) {
            (getChannel(epgChannelIndex) as? EpgChannelExt)?.clearLoadedRanges()
            //Log.d("maxiEpg", "epgData clearRanges")
            _epgData.value?.getEvents(epgChannelIndex)?.clear()
        }
    }

    fun setDatepickText(text: String) { _datePickText.value = text }

    fun addRecordingLabelsToSeriesEvents(
        focusedChannelPosition: Int?,
        focusedEvent: EpgEventExt?
    ) {
        if (focusedChannelPosition == null) return
        if (focusedEvent?.labels?.find {it.id == EpgEventExt.LABEL_ID_RECORDINGS  } != null) return //Recording label already added
        val recordings = recordingsRepository.getRecordingsLD().value
        epgData.value?.getEventsClone(focusedChannelPosition)?.forEach { event ->
            if (recordings?.find { aRecording ->
                    Flavor().compareRecordingWithEvent(
                        aRecording,
                        (event as? EpgEventExt)?.smartMediaItem
                    )
                } != null
                && (event as? EpgEventExt)?.smartMediaItem?.endDate != null
                && event.smartMediaItem.endDate!!.millis > TimeProvider.now().millis) {
                (event as? EpgEventExt)?.addLabel(getRecordingEpgEventLabel())
            }
        }
    }

    fun removeRecordingLabelsFromSeriesEvents(
        focusedChannelPosition: Int?,
        focusedEvent: EpgEventExt?
    ) {
        if (focusedChannelPosition == null) return
        if (focusedEvent?.labels?.find { it.id == EpgEventExt.LABEL_ID_RECORDINGS } == null) return //Recording label already removed
        epgData.value?.getEventsClone(focusedChannelPosition)?.forEach { event ->
            (event as? EpgEventExt)?.let { eventExt ->
                if (eventExt.smartMediaItem.seriesReference == null) return@forEach
                if (eventExt.smartMediaItem.seriesReference == focusedEvent.smartMediaItem.seriesReference) {
                    eventExt.removeRecordingLabel()
                }
            }
        }
    }

    fun getItemDetail(item: SmartMediaItem): LiveData<SmartMediaDetail?> = repository.getDetailLD(item,"OpenPlayerFromEPG")

    companion object {
        private const val LOADER_INDICATOR_DELAY = 1000L
    }
}

class EventComparator : Comparator<SmartMediaItem> {
    override fun compare(o1: SmartMediaItem?, o2: SmartMediaItem?): Int {
        if (o1 === o2) return 0
        if (o1 == null || o1.startDate?.isBefore(o2?.startDate) == true) return -1
        return 1
    }
}