package com.twentyfouri.tvlauncher.utils

import android.content.ActivityNotFoundException
import android.content.Intent
import android.provider.Settings
import android.util.Log
import android.widget.Toast
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentTransaction
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.dashboard.SmartPageReference
import com.twentyfouri.smartmodel.model.menu.SmartNavigationAction
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.tvlauncher.BuildConfig
import com.twentyfouri.tvlauncher.common.utils.NavigatorCommon
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PageType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.ui.TvLauncherToast
import com.twentyfouri.tvlauncher.common.utils.NetworkConnectionState
import com.twentyfouri.tvlauncher.data.ACTION_OPEN_NOTIFICATIONS
import com.twentyfouri.tvlauncher.extensions.isVisible
import com.twentyfouri.tvlauncher.extensions.putSerializable
import com.twentyfouri.tvlauncher.ui.*
import com.twentyfouri.tvlauncher.ui.MainActivity.Companion.REQUEST_CODE_FAVORITE_APP_SELECTION
import com.twentyfouri.tvlauncher.ui.RowPageFragment.Companion.ARG_PAGE_REFERENCE
import timber.log.Timber

private const val EXTRA_SEARCH_TYPE = "search_type"
private const val SEARCH_TYPE_VOICE = 1
private const val SEARCH_TYPE_KEYBOARD = 2

class Navigator: NavigatorCommon() {

    interface NavigatorContract: NavigatorContractCommon {
        fun navigatePlayer(location: PlayerLocation, reference: SmartMediaReference, playerPosition: Int)
        fun navigateCustom(custom: String)
    }

    //do NOT make this public
    private var navigatorContract: NavigatorContract?
        set(value) {
            navigatorContractCommon = value
            if (value != null) {
                if (pendingNavigationTarget != null) {
                    navigate(pendingNavigationTarget)
                    pendingNavigationTarget = null
                }
                if (pendingNavigationDialogFragment != null) {
                    navigateDialog(pendingNavigationDialogFragment!!)
                    pendingNavigationDialogFragment = null
                }
            }
        }
        get() = navigatorContractCommon as? NavigatorContract

    fun setContract(contract: NavigatorContract?) {
        navigatorContract = contract
    }

    override fun navigate(target: SmartNavigationTarget?) {
        if (navigatorContract == null) {
            pendingNavigationTarget = target
            return
        }

        navigatorContract?.also {
            when (target?.action) {
                SmartNavigationAction.SEARCH -> {
                    /* Don't remove. This snippet is commented in relation to: PRJ1210ENT-341.
                       TODO This should be executed only in an alternative build type with automatic installation enabled for search packages and setup wizard.
                       For this build type we will have to check whether the search packages are installed or not when the user navigates to the search
                       screen to show a pop up with the option to install them. This can be done with the AppManager.installSearchingUtilities() method.

                         AppManager.installSearchingUtilities(it.getContext()) {
                          it.navigate(
                              Intent(Intent.ACTION_ASSIST).apply {
                                  addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_RESET_TASK_IF_NEEDED)
                                  putExtra(EXTRA_SEARCH_TYPE, SEARCH_TYPE_KEYBOARD)
                              }
                          )
                      } */
                    it.navigate(
                        Intent(Intent.ACTION_ASSIST).apply {
                            addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_RESET_TASK_IF_NEEDED)
                            putExtra(EXTRA_SEARCH_TYPE, SEARCH_TYPE_KEYBOARD)
                        }
                    )
                }
                SmartNavigationAction.VOICE_SEARCH -> {
                    it.navigate(
                        Intent(Intent.ACTION_ASSIST).apply {
                            addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_RESET_TASK_IF_NEEDED)
                            putExtra(EXTRA_SEARCH_TYPE, SEARCH_TYPE_VOICE)
                        }
                    )
                }
                SmartNavigationAction.FAVORITE_APPS_SCREEN -> it.navigate(target)
                SmartNavigationAction.SETTINGS -> it.navigate(Intent(Settings.ACTION_SETTINGS))
                SmartNavigationAction.DIAGNOSTICS -> it.navigate(target)
                SmartNavigationAction.NOTIFICATIONS -> it.navigate(target)
                SmartNavigationAction.SUBMENU -> {}
                SmartNavigationAction.AVATAR_DROPDOWN -> {}
                SmartNavigationAction.BROWSE_PAGE -> it.navigate(target) //MainActivity goes to RowPageFragment
                SmartNavigationAction.BROWSE_PLAYLIST -> {}
                SmartNavigationAction.LIVE_CHANNELS,
                SmartNavigationAction.LIVE_EPG_SIMPLE,
                SmartNavigationAction.LIVE_EPG_FULL -> it.navigate(target) //MainActivity goes to EpgFragment
                SmartNavigationAction.LANGUAGE -> {}
                SmartNavigationAction.USER_PROFILE -> it.navigate(target) //MainActivity goes to PersonalSettingsActivity
                SmartNavigationAction.USER_RECORDINGS -> it.navigate(target) //MainActivity goes to PersonalSettingsActivity, then shows Recordings Fragment
                SmartNavigationAction.FAVORITES -> {}
                SmartNavigationAction.WEB_VIEW -> {}
                SmartNavigationAction.DETAIL_PAGE -> it.navigate(target) //MainActivity goes to DetailFragment in fullscreen
                SmartNavigationAction.PLAYER -> {
                    target.mediaReference?.let { ref ->
                        var location = PlayerLocation.values().find { loc -> loc.name == target.extras.getString() }
                        if (location == null) location = PlayerLocation.UNCHANGED
                        it.navigatePlayer(location, ref, target.playerPosition)
                    }
                }
                SmartNavigationAction.SWITCH_PROFILE -> {}
                SmartNavigationAction.LOGIN -> it.navigate(target)
                SmartNavigationAction.LOGOUT,
                SmartNavigationAction.SPLASHSCREEN,
                SmartNavigationAction.PROFILE_PIN,
                SmartNavigationAction.PARENTAL_PIN,
                SmartNavigationAction.FORGOT_PASSWORD,
                SmartNavigationAction.UNKNOWN,
                null -> {}
            }
        }
    }

    fun navigateCustom(custom: String) {
        navigatorContract?.navigateCustom(custom)
    }

    companion object {
        const val NAVIGATE_CUSTOM_FORCE_RESTART = "FORCE_RESTART"
        const val NAVIGATE_CUSTOM_SETTINGS_RESTART = "SETTINGS_RESTART"
        const val NAVIGATE_CUSTOM_OFFLINE = "OFFLINE"
        const val NAVIGATE_CUSTOM_APPS_PAGE = "APPS_PAGE"

        init {
            insertInstance(getInstance())
        }

        private var instance: Navigator? = null
        fun getInstance() =
            instance ?: synchronized(this) {
                instance ?: Navigator().also { instance = it }
            }
    }
}

internal fun MainActivity.internalNavigate(target: SmartNavigationTarget) {
    when(target.action) {
        SmartNavigationAction.USER_PROFILE -> {
            startActivity(
                    Intent(this, PersonalSettingsActivity::class.java)
                            .addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            )
        }
        SmartNavigationAction.USER_RECORDINGS -> {
            startActivity(
                    Intent(this, PersonalSettingsActivity::class.java).apply {
                        putExtra(EXTRA_FRAGMENT_NAVIGATION_CODE, USER_RECORDINGS_CODE)
                    }
            )
        }
        SmartNavigationAction.FAVORITE_APPS_SCREEN -> {
            startActivityForResult(
                Intent (this, FavoriteAppsActivity::class.java).apply {
                    putExtra(SELECTED_APP_ID, target.extras.getString())
                },
                REQUEST_CODE_FAVORITE_APP_SELECTION
            )
        }
        SmartNavigationAction.DIAGNOSTICS -> startActivity(Intent(this, DiagnosticsActivity::class.java))
        SmartNavigationAction.BROWSE_PAGE -> {
            getTopBar()?.selectByPageReference(target.pageReference, true)
            internalNavigateRowPageFragment(target.pageReference)
        }
        SmartNavigationAction.LIVE_CHANNELS,
        SmartNavigationAction.LIVE_EPG_SIMPLE,
        SmartNavigationAction.LIVE_EPG_FULL -> internalNavigateEpgFragment()
        SmartNavigationAction.LOGIN -> {
            try {
                startActivityForResult(
                        Intent(getString(R.string.credentials_activity_name)),
                        MainActivity.REQUEST_CODE_LOGIN_AND_RESTART
                )
            } catch (ex: ActivityNotFoundException) {
                TvLauncherToast.makeText(
                        this,
                        ex.message ?: "SetupWizard not installed",
                        Toast.LENGTH_SHORT
                )?.show()
            }
        }
        SmartNavigationAction.DETAIL_PAGE -> {
            target.mediaReference?.let { ref ->
                //do not navigate to detail if no connectivity
                if (NetworkConnectionState.instance.state != NetworkConnectionState.State.ONLINE) {
                    TvLauncherToast.makeText(this, R.string.no_internet_error, Toast.LENGTH_LONG)?.show()
                } else {
                    openFragmentInFullscreen(DetailFragment.newInstance(ref))
                }
            }
        }
        SmartNavigationAction.NOTIFICATIONS -> {
            @Suppress("ConstantConditionIf")
            if (BuildConfig.IS_APP.not()) {
                navigate(
                        Intent(ACTION_OPEN_NOTIFICATIONS).apply {
                            setPackage(getString(R.string.own_package_name))
                        }
                )
            }
        }
    }
}

internal fun MainActivity.internalNavigateCustom(custom: String) {
    when (custom) {
        Navigator.NAVIGATE_CUSTOM_FORCE_RESTART -> {
            startActivity(Intent(this, MainActivity::class.java).apply { putExtra(MainActivity.EXTRA_FORCE_RESTART, true) })
        }
        Navigator.NAVIGATE_CUSTOM_SETTINGS_RESTART -> {
            startActivityForResult(Intent(Settings.ACTION_SETTINGS), MainActivity.REQUEST_CODE_LOGIN_AND_RESTART)
        }
        Navigator.NAVIGATE_CUSTOM_OFFLINE -> setOfflineMode()
        Navigator.NAVIGATE_CUSTOM_APPS_PAGE -> {
            val pageReference = getPageReference(PageType.APPS)
            navigate(SmartNavigationTarget.toBrowsePage(pageReference))
        }
    }
}

private fun MainActivity.openFragmentInFullscreen(fragment: Fragment) {
    Timber.tag("fragments").d("add RowPageFragment in openFragmentInFullScreen")
    // Exit the player, if it's open
    playerStop(true)
    val tagExt = if (fragment is DetailFragment) {
        DetailFragment::class.java.simpleName
    } else {
        getTagExtFromPageReference(fragment.arguments?.getSerializable(ARG_PAGE_REFERENCE) as? SmartPageReference)
    }
    val transaction = supportFragmentManager.beginTransaction()
    hideContentFragment(transaction)
    /*fix attempt for
    * java.lang.IllegalStateException: Can not perform this action after onSaveInstanceState
    * at com.twentyfouri.tvlauncher.utils.NavigatorKt.openFragmentInFullscreen(Navigator.kt)
    * using commitAllowingStateLoss() */
    // TODO: Remove if no affect is achieved
    Timber.tag("MainActivity.openFragmentInFullscreen").d("commitAllowingStateLoss $fragment")
    FirebaseCrashlytics.getInstance()
        .log("MainActivity.openFragmentInFullscreen: commitAllowingStateLoss $fragment")
    transaction
        .replace(R.id.fullscreen_frame, fragment, tagExt)
        .addToBackStack(null)
        .commitAllowingStateLoss()
}

private fun MainActivity.hideContentFragment(transaction: FragmentTransaction) {
    val contentFragment = supportFragmentManager.findFragmentById(R.id.content_frame)
    if (contentFragment != null && contentFragment.isHidden.not()) {
        transaction.hide(contentFragment)
        if (contentFragment is BaseFragment) contentFragment.saveCurrentFocus()
    }
    getTopBar()?.also {
        if(it.isHidden.not()) transaction.hide(it)
    }
    if (binding.viewModel?.showDebugButtons?.value == true) {
        binding.viewModel?.showDebugButtons(false)
        transaction.setBreadCrumbShortTitle(MainActivity.TRANSACTION_BREADCRUMB_BUTTONS_HIDE)
    }
}

private fun MainActivity.removeFullscreenFragment(transaction: FragmentTransaction) {
    Timber.tag("fragments").d("removeFullScreenFragment")
    val fullscreenFragment = supportFragmentManager.findFragmentById(R.id.fullscreen_frame)
    fullscreenFragment?.also {
        Timber.tag("MainActivity.removeFullscreenFragment").d("commit $it")
        FirebaseCrashlytics.getInstance().log("MainActivity.removeFullscreenFragment: commit $it")
        supportFragmentManager.beginTransaction().remove(it).commit()
    }
    getTopBar()?.also { if (it.isHidden) transaction.show(it) }
}

internal fun MainActivity.internalNavigateEpgFragment() {
    // Exit the player, if it's open
    // but leave it if we are heading to EPG and player should play behind
    if (Flavor().displayPlayerBehindEPG) {
        if (playerAction.isPlayingOrShouldResume() && lastPlayerFragmentLocation == PlayerLocation.FOREGROUND) togglePlayerFrontBack()
    } else {
        playerStop(false)
    }

    val tagExt = getTagExtFromPageReference(getPageReference(PageType.EPG))
    val transaction = supportFragmentManager.beginTransaction()
    removeFullscreenFragment(transaction)

    val currentFragment = supportFragmentManager.findFragmentById(R.id.content_frame)
    if (currentFragment is EpgFragment) { // If we're already on the page, stay
        if (currentFragment.isHidden) transaction.show(currentFragment)
        Timber.tag("MainActivity.internalNavigateEpgFragment").d("commit1")
        FirebaseCrashlytics.getInstance().log("MainActivity.internalNavigateEpgFragment: commit1")
        transaction.addToBackStack(null).commit()
    } else {
        val hiddenFragment = supportFragmentManager.findFragmentByTag(tagExt)
        if (hiddenFragment != null) {
            Timber.tag("fragments").d("internal navigate EpgFragment")
            transaction.replace(R.id.content_frame, hiddenFragment, tagExt)
            if (hiddenFragment.isHidden) transaction.show(hiddenFragment)
            Timber.tag("MainActivity.internalNavigateEpgFragment").d("commit2")
            FirebaseCrashlytics.getInstance()
                .log("MainActivity.internalNavigateEpgFragment: commit2")
            transaction
                .addToBackStack(null)
                .commit()
        } else {
            val newFragment = EpgFragment().apply {
                putSerializable(Flavor().getEpgConfigValues.REQUEST_FOCUS,
                    true)
            }
            if (newFragment.showTopbarWhenNavigated) showTopBar()
            Timber.tag("MainActivity.internalNavigateEpgFragment").d("commit3")
            FirebaseCrashlytics.getInstance()
                .log("MainActivity.internalNavigateEpgFragment: commit3")
            transaction
                .replace(R.id.content_frame, newFragment, tagExt)
                .addToBackStack(null)
                .commit()
        }
    }
}

private fun Fragment.equalsReference(pageReference: SmartPageReference?): Boolean {
    if (this is RowPageFragment) {
        val currentPageReference: SmartPageReference? = arguments?.getSerializable(ARG_PAGE_REFERENCE) as? SmartPageReference
        if (currentPageReference == pageReference && pageReference != null) {
            return true
        }
    }
    return false
}

internal fun MainActivity.internalNavigateRowPageFragment(pageReference: SmartPageReference?) {
    playerStop(false)
    val tagExt = getTagExtFromPageReference(pageReference)
    val transaction = supportFragmentManager.beginTransaction()
    removeFullscreenFragment(transaction)

    val currentFragment = supportFragmentManager.findFragmentById(R.id.content_frame)
    if (currentFragment?.equalsReference(pageReference) == true) { // If we're already on the page, stay
        Timber.tag("fragments").d("add RowPageFragment in navigator")
        if (Flavor().shouldHomeButtonFocusHomeTab && pageReference == getPageReference(PageType.MAIN)) {
            val fragment = supportFragmentManager.findFragmentById(R.id.content_frame)
            (fragment as? RowPageFragment)?.reset()
        }
        if (currentFragment.isHidden) transaction.show(currentFragment)
        Timber.tag("MainActivity.internalNavigateRowPageFragment").d("commitAllowingStateLoss")
        FirebaseCrashlytics.getInstance()
            .log("MainActivity.internalNavigateRowPageFragment: commitAllowingStateLoss")
        transaction.addToBackStack(null).commitAllowingStateLoss()
    } else {
        Timber.tag("fragments").d("add RowPageFragment in navigator 2")
        val hiddenFragment = supportFragmentManager.findFragmentByTag(tagExt)
        if (hiddenFragment?.isAdded == true) {
            hiddenFragment.isHidden.ifTrue { transaction.show(hiddenFragment) }
            Log.d("MainActivity.internalNavigateRowPageFragment", "commitNowAllowingStateLoss1")
            FirebaseCrashlytics.getInstance().log("MainActivity.internalNavigateRowPageFragment: commitNowAllowingStateLoss1")
            transaction.runOnCommit {
                (hiddenFragment as? RowPageFragment)?.reset()
            }.commitNowAllowingStateLoss()
        } else if (pageReference != null) {
            val rowPageFragment = RowPageFragment.newInstance(pageReference)
            if (rowPageFragment.showTopbarWhenNavigated) showTopBar()
            rowPageFragment.isAdded.ifFalse {
                Timber.tag("MainActivity.internalNavigateRowPageFragment")
                    .d("commitNowAllowingStateLoss2")
                FirebaseCrashlytics.getInstance()
                    .log("MainActivity.internalNavigateRowPageFragment: commitNowAllowingStateLoss2")
                transaction
                    .replace(R.id.content_frame, rowPageFragment, tagExt)
                    .runOnCommit { rowPageFragment?.reset() } //leave the ?., it can happen somehow
                    .commitNowAllowingStateLoss()
            }
        }
    }
}

internal fun MainActivity.shouldCallOnBackPressed(): Boolean {
    isListPickerOnTop().ifTrue { return true }
    isRowPageFragmentOnTop().ifFalse { return true } //HOME and APPS cannot be closed by Back
    isPlayerOnTop().ifTrue { return true }
    return false
}

private fun MainActivity.isRowPageFragmentOnTop(): Boolean {
    val contentFragment = supportFragmentManager.findFragmentById(R.id.content_frame) ?: return false
    if ((contentFragment is RowPageFragment).not()) return false
    if (contentFragment.isHidden) return false
    return true
}

private fun MainActivity.isPlayerOnTop(): Boolean = getBindingSafe()?.playerContainer?.isVisible() ?: false

private fun MainActivity.isListPickerOnTop(): Boolean {
    val fullscreenFragment = supportFragmentManager.findFragmentById(R.id.fullscreen_frame) ?: return false
    if ((fullscreenFragment is ListPickerFragment).not()) return false
    if (fullscreenFragment.isHidden) return false
    return true
}

internal fun MainActivity.isEpgFragmentDisplayedOverPlayer(): Boolean =
    Flavor().displayPlayerBehindEPG
            && playerAction.isPlayingOrShouldResume()
            && getEpgFragment()?.isVisible == true
            && lastPlayerFragmentLocation == PlayerLocation.BACKGROUND
            && getListPickerFragment()?.isHidden != false