package com.twentyfouri.tvlauncher.utils

import android.graphics.Typeface
import android.view.LayoutInflater
import android.view.View
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import androidx.databinding.BindingAdapter
import androidx.databinding.DataBindingUtil
import androidx.databinding.ViewDataBinding
import androidx.leanback.widget.HorizontalGridView
import androidx.leanback.widget.OnChildViewHolderSelectedListener
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import com.bumptech.glide.Glide
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.tvlauncher.BR
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.widgets.PinEntryView
import com.twentyfouri.tvlauncher.widgets.MetadataView
import com.twentyfouri.tvlauncher.widgets.RowItemView

@BindingAdapter("onFocusChange")
fun onFocusChange(view: View, listener: View.OnFocusChangeListener?) {
    if (view.onFocusChangeListener == null) view.onFocusChangeListener = listener
}

@BindingAdapter("viewHolders")
fun <T : ViewModel> LinearLayout.setViewHolders(viewHolders: List<ViewHolder<T>>?) {
    viewHolders ?: return
    //Log.d("mapar","focus ${findFocus()}")
    var i = 0
    val viewsToRemove: MutableList<View> = mutableListOf()
    val currentChildCount = childCount
    while (i < currentChildCount || i < viewHolders.size ?: 0) {
        val view = getViewAt(i)
        val viewHolder = viewHolders.getOrElse(i) {null}
        if (viewHolder == null) {
            view?.also { //Log.d("mapar","removeView")
                viewsToRemove.add(it) }
            i++
            continue
        }
        if (view != null) {
            val dataBinding = DataBindingUtil.getBinding<ViewDataBinding>(view)
            viewHolder.view = view
            dataBinding?.setVariable(BR.viewModel, viewHolder.model)
            i++
            continue
        }
        val dataBinding = DataBindingUtil.inflate<ViewDataBinding>(LayoutInflater.from(context), viewHolder.layoutId, this, false)
        //Log.d("mapar","addView")
        addView(dataBinding.root)
        viewHolder.view = dataBinding.root
        dataBinding?.setVariable(BR.viewModel, viewHolder.model)
        dataBinding.lifecycleOwner = context as LifecycleOwner
        i++
    }
    for (v in viewsToRemove) removeView(v)

    //The following code is commented as a reminder that it was done like this before and it is intentionally not done like that any more
//    removeAllViews()
//    viewHolders?.forEach {
//        val dataBinding = DataBindingUtil.inflate<ViewDataBinding>(LayoutInflater.from(context), it.layoutId, this, false)
//        addView(dataBinding.root)
//        it.view = dataBinding.root
//        dataBinding?.setVariable(BR.viewModel, it.model)
//        dataBinding.lifecycleOwner = context as LifecycleOwner
//    }
}

private fun LinearLayout.getViewAt(i: Int): View? {
    if (i >= childCount) return null
    return getChildAt(i)
}

@BindingAdapter("imageFromUrl")
fun ImageView.bindImageFromUrl(imageUrl: MutableLiveData<String>) {
    if (!imageUrl.value.isNullOrEmpty()) {
        Glide.with(this.context)
            .load(imageUrl.value)
            .into(this)
    }
}

@BindingAdapter("selected")
fun View.setSelectedState(isSelected: Boolean) {
    this.isSelected = isSelected
    if (this is RowItemView){
        this.viewModel.setIsSelected(isSelected)
    }
}

@BindingAdapter("onPinEnteredListener")
fun PinEntryView.onPinEntered(listener: PinEntryView.OnPinEnteredListener) {
    this.setOnPinEnteredListener(listener)
}

@BindingAdapter("mediaReference")
fun MetadataView.setMediaReference(mediaReference: SmartMediaReference) {
    this.bind(aMediaReference = mediaReference)
}

@BindingAdapter("isExpanded")
fun MetadataView.setIsExpanded(expanded: Boolean) {
    this.setExpanded(expanded)
}

@BindingAdapter("isBold")
fun setBold(view: TextView, isBold: Boolean) {
    val context = view.context
    if (isBold) {
        view.typeface = Typeface.createFromAsset(context.assets, context.getString(R.string.font_path_extrabold))
    } else {
        view.typeface = Typeface.createFromAsset(context.assets, context.getString(R.string.font_path_bold))
    }
}

@BindingAdapter("onChildViewHolderSelectedListener")
fun  setChildViewHolderSelectedListener(view: HorizontalGridView, listener: OnChildViewHolderSelectedListener?){
    if (listener != null) {
        view.addOnChildViewHolderSelectedListener(listener)
    }
}

@BindingAdapter("font")
fun setFont (view: TextView, fontPath: String?){
    fontPath?.let{view.typeface = Typeface.createFromAsset(view.context.assets, fontPath)}
    }