package com.twentyfouri.tvlauncher.ui

import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.databinding.DataBindingUtil
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.LifecycleOwner
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.payment.SmartPurchaseState
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.adapters.SubscriptionChannelsAdapter
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogCodes
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogDismissListener
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogFragment
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogModel
import com.twentyfouri.tvlauncher.common.utils.NavigatorCommon
import com.twentyfouri.tvlauncher.common.ui.messagedialog.*
import com.twentyfouri.tvlauncher.databinding.FragmentSubscriptionsGridBinding
import com.twentyfouri.tvlauncher.utils.RemoteControlState
import com.twentyfouri.tvlauncher.viewmodels.SubscriptionsGridViewModel
import org.koin.androidx.viewmodel.ext.android.getViewModel
import org.koin.androidx.viewmodel.ext.android.viewModel
import timber.log.Timber

class SubscriptionsGridFragment: BaseFragment() {

    private var nullableBinding: FragmentSubscriptionsGridBinding? = null
    private val binding: FragmentSubscriptionsGridBinding get() = nullableBinding ?: throw IllegalStateException("trying to access uninitialized binding")
    private val subscriptionChannelsList: MutableList<SmartMediaItem> = arrayListOf()
    private val gridAdapter = SubscriptionChannelsAdapter(subscriptionChannelsList)

    override fun onCreateView(
            inflater: LayoutInflater,
            container: ViewGroup?,
            savedInstanceState: Bundle?
    ): View = FragmentSubscriptionsGridBinding.inflate(
                layoutInflater,
                container,
                false
        ).apply {
            lifecycleOwner = this@SubscriptionsGridFragment.viewLifecycleOwner
            gridview.adapter = gridAdapter
            nullableBinding = this
        }.root

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        binding.viewModel = getViewModel()

        val channelItem = arguments?.getSerializable(ARG_CHANNEL_ITEM) as? SmartMediaItem
        bind(channelItem, viewLifecycleOwner)
    }

    fun bind(subscriptionChannel: SmartMediaItem?, lifecycleOwner: LifecycleOwner) {
        binding.root.visibility = View.GONE
        clearChannelList()
        binding.viewModel?.setSubscriptionChannel(subscriptionChannel)
        ((context as FragmentActivity) as MainActivity).remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION)
        binding.offerOne.requestFocus()
        binding.viewModel?.subscriptionBought?.observe(lifecycleOwner, {
            it?.let { purchaseResult ->
                if (purchaseResult.state == SmartPurchaseState.OK) {
                    binding.viewModel?.setSubscriptionBought(null)
                    showPurchaseSuccessDialog()
                }

                if (purchaseResult.state == SmartPurchaseState.FAILED) {
                    //todo: need access to https://tvdev.atlassian.net/wiki/spaces/TED/pages/591953929/CR+New+Purchase+on+screen+flow
                    when(purchaseResult.failReasonCode) {
                        20 -> {
                            //show Insufficient Funds dialog
                            binding.viewModel?.setSubscriptionBought(null)
                            showInsufficientFundsDialog(purchaseResult.failReasonCode.toString())

                        }
                        22 -> {
                            //show User does not exist dialog
                            binding.viewModel?.setSubscriptionBought(null)
                            showUserDoesNotExistsDialog(purchaseResult.failReasonCode.toString())
                        }
                        else -> {
                            //show generic error dialog? do nothing?
                            binding.viewModel?.setSubscriptionBought(null)
                            showPurchaseFailedDialog(purchaseResult.failReasonCode.toString())
                        }
                    }
                }
            }
        })
        binding.viewModel?.cancelButtonVisibility?.observe(lifecycleOwner, {
            binding.executePendingBindings()
            if (it == View.VISIBLE) {
                binding.cancelButton.requestFocus() // binding.cancelButton.requestFocus() does not work here
                binding.viewModel?.hideAllOffers()
            }
        })
        binding.viewModel?.restartSubscriptionView?.observe(lifecycleOwner, {
            if (it == true) {
                clearChannelList()
                binding.viewModel?.setRestartSubscriptionView(false)
                binding.offers.requestFocus()
            }
        })
        binding.viewModel?.channelsIncludedInSubscription?.observe(lifecycleOwner, {
            if (binding.viewModel?.cancelButtonVisibility?.value == View.VISIBLE) {
                binding.gridview.layoutParams.height = binding.gridview.minimumHeight
                if (it.size > 8) { binding.gridview.layoutParams.height = binding.gridview.layoutParams.height + it.size * 15 } //stretch GridView when too many channels
                subscriptionChannelsList.clear()
                subscriptionChannelsList.addAll(it ?: emptyList())
                gridAdapter.notifyDataSetChanged()
            }
        })
        binding.viewModel?.subscriptionPackages?.observe(lifecycleOwner, {
            if (it.isEmpty()) {
                binding.root.visibility = View.GONE
                showNoSubscriptionsAvailableDialog()
                binding.viewModel?.subscriptionPackages?.removeObservers(lifecycleOwner)
            } else {
                binding.root.visibility = View.VISIBLE
                ((context as FragmentActivity) as MainActivity).playerAction.showControls(true)
            }
        })
    }

    override fun onDestroyView() {
        binding.viewModel?.clearModel()
        clearChannelList()
        binding.viewModel?.setSubscriptionChannel(null)
        binding.viewModel?.dismissPurchaseDialog()
        super.onDestroyView()
        nullableBinding = null
    }

    private fun showPurchaseSuccessDialog() {
        val messageDialogModel = MessageDialogModel(
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_completed_title),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_completed_message),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_completed_close),
                MessageDialogCodes.purchaseSuccess
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        dialogFragment.mDismissListener = object : MessageDialogDismissListener {
            override fun onDismiss() {
                ((context as FragmentActivity) as MainActivity).refreshPlayerAfterSubscriptionBought()
            }
        }
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    private fun showInsufficientFundsDialog(code: String) {
        val messageDialogModel = MessageDialogModel(
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_insufficient_funds_title),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_insufficient_funds_message, code),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_insufficient_funds_close),
                MessageDialogCodes.insufficientFunds
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    private fun showPurchaseFailedDialog(code: String) {
        val messageDialogModel = MessageDialogModel(
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_failed_title),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_failed_message, code),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_failed_close),
                MessageDialogCodes.purchaseFailed
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    private fun showUserDoesNotExistsDialog(code: String) {
        val messageDialogModel = MessageDialogModel(
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_user_does_not_exists_title),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_user_does_not_exists_message, code),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_user_does_not_exists_close),
                MessageDialogCodes.userDoesNotExists
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    private fun showNoSubscriptionsAvailableDialog() {
        val messageDialogModel = MessageDialogModel(
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.no_subscriptions_available),
                null,
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_completed_close),
                MessageDialogCodes.noSubscriptionsAvailable
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        dialogFragment.mDismissListener = object : MessageDialogDismissListener {
            override fun onDismiss() {
                Timber.tag("SubscriptionsGridFragment.showNoSubscriptionsAvailableDialog.onDismiss")
                    .d("commit ${this@SubscriptionsGridFragment}")
                FirebaseCrashlytics.getInstance()
                    .log("SubscriptionsGridFragment.showNoSubscriptionsAvailableDialog.onDismiss: commit ${this@SubscriptionsGridFragment}")
                parentFragmentManager.beginTransaction().remove(this@SubscriptionsGridFragment)
                    .commitAllowingStateLoss()
            }
        }
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    override fun onPause() {
        parentFragmentManager.beginTransaction().remove(this@SubscriptionsGridFragment).commit()
        super.onPause()
    }

    private fun clearChannelList(){
        subscriptionChannelsList.clear()
        gridAdapter.notifyDataSetChanged()
    }

    companion object {
        private const val ARG_CHANNEL_ITEM = "ARG_CHANNEL_ITEM"

        fun newInstance(channelItem: SmartMediaItem): SubscriptionsGridFragment {
            val args = Bundle(1)
            args.putSerializable(ARG_CHANNEL_ITEM, channelItem)
            val fragment = SubscriptionsGridFragment()
            fragment.arguments = args
            return fragment
        }
    }
}