package com.twentyfouri.tvlauncher.ui

import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.leanback.widget.BaseGridView
import androidx.lifecycle.lifecycleScope
import androidx.recyclerview.widget.RecyclerView
import com.google.firebase.analytics.FirebaseAnalytics
import com.twentyfouri.smartmodel.model.dashboard.SmartPageReference
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.adapters.RowAdapter
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.ui.MainActivityAction
import com.twentyfouri.tvlauncher.common.utils.NetworkConnectionState
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.data.HardcodedPlaylistReference
import com.twentyfouri.tvlauncher.data.ListPickerItem
import com.twentyfouri.tvlauncher.databinding.FragmentRowPageBinding
import com.twentyfouri.tvlauncher.receiver.ScreenOnOffReceiver
import com.twentyfouri.tvlauncher.utils.setSelectedState
import com.twentyfouri.tvlauncher.viewmodels.RowPageViewModel
import com.twentyfouri.tvlauncher.widgets.RowView
import com.twentyfouri.tvlauncher.widgets.leanback.OnChildViewHolderSelectedListener
import com.twentyfouri.tvlauncher.widgets.leanback.VerticalGridView
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.getViewModel
import timber.log.Timber

class RowPageFragment : BaseFragment() {

    override val showTopbarWhenNavigated = true
    private var nullableBinding: FragmentRowPageBinding? = null
    private val viewLifecycleIsActive
        get() = nullableBinding != null
    val binding: FragmentRowPageBinding get() = nullableBinding ?: throw IllegalStateException("trying to access uninitialized binding")
    private val rowPageViewModel
        get() = nullableBinding?.viewModel
    private var focusFirstRowWhenLoaded: Boolean = false
    private var focusFirstRowWhenLoadedDone: Boolean = false
    private var isHomePage: Boolean = false
    private fun getBindingSafe() = nullableBinding
    private var delayedRefreshOnResumeJob: Job? = null

    private var pageIdForLog = "UNKNOWN"

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View = FragmentRowPageBinding.inflate(
        inflater,
        container,
        false
    ).apply {
        lifecycleOwner = this@RowPageFragment.viewLifecycleOwner
        nullableBinding = this
    }.root

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val pageReference = requireArguments().getSerializable(ARG_PAGE_REFERENCE) as? SmartPageReference
        val viewModelFromFactory = getViewModel<RowPageViewModel>()

        pageReference?.also {
            viewModelFromFactory.updatePageReference(it)
            (activity as? MainActivity)?.getTopBar()?.selectByPageReference(pageReference, true)
        }

        isHomePage = Flavor().isHomePage(pageReference)

        pageIdForLog = when {
            Flavor().isHomePage(pageReference) -> "HOME"
            Flavor().isAppsPage(pageReference) -> "APPS"
            else -> tag ?: "unknown"
        }

        binding.viewModel = viewModelFromFactory

        val adapter = RowAdapter(DEFAULT_LAYOUT_ANIMATION)
        binding.rowPageGrid.adapter = adapter
        setScrollDefault(binding.rowPageGrid)
        setupPageDataObserver(adapter)
    }

    override fun onDestroyView() {
        //this triggers onDetached on RowViewViewHolder, which triggers unbind on RowView, which releases RowItemAdapter
        binding.rowPageGrid.adapter = null
        super.onDestroyView()
        nullableBinding = null
    }

    fun hideOfflineRow(hide: Boolean) {
        rowPageViewModel?.rowViewModels?.value?.let { models ->
            val offrow = models.find { it.sectionStyle == HardcodedPlaylistReference.Type.OFFLINE.name } ?: return //no offline rows so exit
            if(offrow.forceHide.value == hide) return //already done

            var wasFocused = false
            val firstVisibleBefore = (binding.rowPageGrid.adapter as? RowAdapter)?.getFirstVisibleRowPosition() ?: -1
            models.filter { it.sectionStyle == HardcodedPlaylistReference.Type.OFFLINE.name }.forEach {
                it.forceHide.value = hide
                if(it.isFocused.value == true) wasFocused = true
            }
            val firstVisibleAfter = (binding.rowPageGrid.adapter as? RowAdapter)?.getFirstVisibleRowPosition() ?: -1
            if(hide) {
                if(binding.rowPageGrid.findFocus() == null && firstVisibleAfter >= 0) {
                    binding.rowPageGrid.smoothScrollToPosition(firstVisibleAfter)
                } else if(wasFocused && firstVisibleAfter >= 0) {
                    binding.rowPageGrid.getChildAt(firstVisibleAfter).requestFocus()
                } else if(models.getOrNull(firstVisibleAfter)?.isFocused?.value == true){
                    (activity as? MainActivity)?.showTopBar()
                    return@let
                }
            } else {
                if(models.getOrNull(firstVisibleBefore)?.isFocused?.value == true) {
                    (activity as? MainActivity)?.hideTopBar()
                }
                if(binding.rowPageGrid.findFocus() == null) {
                    binding.rowPageGrid.smoothScrollToPosition(firstVisibleAfter)
                }
            }
        }
    }

    private fun setupPageDataObserver(adapter: RowAdapter) {
        //pageSections must be MutableLiveData, because on another place we have to change it, therefore we have to update it before we can observe it
        rowPageViewModel?.updatePageSectionsOnLifecycle(viewLifecycleOwner)
        rowPageViewModel?.rowViewModels?.observe(viewLifecycleOwner) { rowModels ->
            Timber.tag("Rows2")
                .d("RowPageFragment pageDataSections loaded ... countOfRows:${rowModels.size}")
            if (rowModels.isEmpty().not()) {
                focusFirstRowWhenLoaded = true
                adapter.customSubmitList(rowModels)
                binding.rowPageGrid.setItemViewCacheSize(rowModels.size)
            } else {
                focusFirstRowWhenLoaded = false
            }
        }

        binding.rowPageGrid.setOnChildLaidOutListener { _, view, position, _ ->
            val firstVisible = adapter.getFirstNotForceHiddenRowPosition()
            if (position == firstVisible && focusFirstRowWhenLoaded && !focusFirstRowWhenLoadedDone && adapter.isRowVisible(firstVisible)) {
                view.requestFocus()
                binding.rowPageGrid.smoothScrollToPosition(firstVisible)
                focusFirstRowWhenLoaded = false
                focusFirstRowWhenLoadedDone = true
            }
        }
    }

    private fun setScrollDefault(rowPageGrid: VerticalGridView) {
        rowPageGrid.apply {
            itemAlignmentOffset = 0
            itemAlignmentOffsetPercent = VerticalGridView.WINDOW_ALIGN_OFFSET_PERCENT_DISABLED
            isItemAlignmentOffsetWithPadding = true
            windowAlignment = BaseGridView.WINDOW_ALIGN_LOW_EDGE
            windowAlignmentOffset = resources.getDimensionPixelSize(R.dimen.topbar_offset_for_content)
            windowAlignmentOffsetPercent = VerticalGridView.WINDOW_ALIGN_OFFSET_PERCENT_DISABLED
            // disable recycling of rows
            recycledViewPool.setMaxRecycledViews(0, 0)

            addOnChildViewHolderSelectedListener(object : OnChildViewHolderSelectedListener() {
                override fun onChildViewHolderSelected(
                    parent: RecyclerView?,
                    child: RecyclerView.ViewHolder?,
                    position: Int,
                    subposition: Int
                ) {
                    if((parent?.adapter as? RowAdapter)?.isRowFirstVisible(position) == true) {
                        (activity as MainActivity).showTopBar()
                    } else {
                        when (position) {
                            0 -> (activity as MainActivity).showTopBar()
                            else -> (activity as MainActivity).hideTopBar()
                        }
                    }
                }
            })
        }
    }

    override fun saveCurrentFocus() {
        if (binding.root is ViewGroup) {
            lastFocusedView = (binding.root as ViewGroup).findFocus()
        }
    }

    override fun reset() {
        lastFocusedView = (activity as? MainActivity)?.getTopBar()?.view
        (binding.rowPageGrid.adapter as? RowAdapter)?.let {adapter ->
            viewLifecycleOwner.lifecycleScope.launch {
                delay(TEST_DELAY)
                adapter.notifyDataSetChanged()
                val findFirstPositionToScrollTo = adapter.getFirstVisibleRowPosition()
                if (findFirstPositionToScrollTo > -1) binding.rowPageGrid.smoothScrollToPosition(findFirstPositionToScrollTo)
            }
        }
    }

    override fun onBackPressed(topbarFocused: Boolean): BackPressAction {
        val focusedView = binding.rowPageGrid.findFocus()
        if(focusedView?.isSelected == true){
            // If currently focused view is selected, unselect it first
            focusedView.setSelectedState(false)
            return BackPressAction.RETURN
        }
        (binding.rowPageGrid.adapter as? RowAdapter)?.let {adapter ->
            if(adapter.isUsingFilters()) {
                binding.rowPageGrid.smoothScrollToPosition(0)
                adapter.tryFocusFilters()
            } else {
                val findFirstPositionToScrollTo = adapter.getFirstVisibleRowPosition()
                if (findFirstPositionToScrollTo > -1) binding.rowPageGrid.smoothScrollToPosition(findFirstPositionToScrollTo)
            }
        }
        return super.onBackPressed(topbarFocused)
    }

    override fun onResume() {
        super.onResume()
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Page resumed: $pageIdForLog")
        FirebaseAnalytics.getInstance(requireContext()).logEvent(FirebaseAnalytics.Event.SCREEN_VIEW, bundleOf(
            FirebaseAnalytics.Param.SCREEN_NAME to tag,
            FirebaseAnalytics.Param.SCREEN_CLASS to "RowPageFragment"
        ))
        NetworkConnectionState.instance.waitForConnection(lifecycleOwner = viewLifecycleOwner, skipWaiting = !ScreenOnOffReceiver.screenWasOff) {
            Timber.tag("Flow").d("refreshOnResume ... RowPageFragment onResume")
            val focusedView = getBindingSafe()?.rowPageGrid?.findFocus()
            getBindingSafe()?.viewModel?.updatePageSectionsOnResume(
               skipUpdateForRowIndex = if (focusedView?.isSelected == true) binding.rowPageGrid.selectedPosition else null
            )
        }
        if(isHomePage) (activity as? MainActivityAction)?.provideNQC()?.isHomeVisible = true
    }

    override fun onPause() {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Page paused: $pageIdForLog")
        Timber.tag("Flow").d("refreshOnResume ... RowPageFragment onPause")
        stopPageSectionsUpdates()
        if (isHomePage) (activity as? MainActivityAction)?.provideNQC()?.isHomeVisible = false
        super.onPause()
    }

    override fun onHiddenChanged(hidden: Boolean) {
        super.onHiddenChanged(hidden)
        Timber.tag("Flow").d("refreshOnResume ... RowPageFragment onHiddenChanged: $hidden")
        if (hidden) {
            stopPageSectionsUpdates()
        } else {
            if ((activity as? MainActivity)?.isForegroundPlaybackActive() == false) rowPageViewModel?.updatePageSectionsOnResume()
        }
    }

    fun onPlayerStop() {
        Timber.tag("Flow").d("refreshOnResume ... RowPageFragment onPlayerStop")
        rowPageViewModel?.updatePageSectionsOnResume()
        viewLifecycleIsActive.ifFalse {
            Timber.tag("Flow")
                .d("refreshOnResume ... delayed refresh cancelled because view is destroyed")
            return
        }
        //it happens that the player stores a bookmark AFTER the bookmark row on RowPageFragment is reloaded
        //thus the just viewed asset is not displayed in the row, thus here the rows which should be reloaded
        //on resume are reloaded again after some empiric number of milliseconds
        delayedRefreshOnResumeJob = viewLifecycleOwner.lifecycleScope.launchDelayedJob(3000) {
            Timber.tag("Flow").d("refreshOnResume ... RowPageFragment onPlayerStop delayed job")
            rowPageViewModel?.updatePageSectionsOnResume()
        }
    }

    private fun CoroutineScope.launchDelayedJob(
        refreshTime: Int,
        block: () -> Unit
    ) = this.launch (Dispatchers.IO) {
        if (refreshTime > 0) {
            if (isActive) {
                delay(refreshTime.toLong())
                block()
            }
        } else {
            block()
        }
    }

    private fun stopPageSectionsUpdates() {
        delayedRefreshOnResumeJob?.cancel()
        delayedRefreshOnResumeJob = null
        rowPageViewModel?.stopPageSectionsUpdates()
    }

    fun onPlayerStart() {
        Timber.tag("Flow").d("refreshOnResume ... RowPageFragment onPlayerStart")
        stopPageSectionsUpdates()
    }

    fun onPicked(listPickerItem: ListPickerItem) {
        rowPageViewModel?.sectionViewModels?.value?.forEach { it.selectFilter(listPickerItem) }
    }

    companion object {
        const val ARG_PAGE_REFERENCE = "ARG_PAGE_REFERENCE"
        private const val DEFAULT_LAYOUT_ANIMATION = RowView.ROW_VIEW_LAYOUT_BASIC

        fun newInstance(pageReference: SmartPageReference): RowPageFragment {
            val args = Bundle(1)
            args.putSerializable(ARG_PAGE_REFERENCE, pageReference)
            val fragment = RowPageFragment()
            fragment.arguments = args
            return fragment
        }
    }
}