package com.twentyfouri.tvlauncher.ui

import android.content.Context
import android.content.Intent
import android.content.res.ColorStateList
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Rect
import android.graphics.Typeface
import android.graphics.drawable.ColorDrawable
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import android.util.Log
import android.view.KeyEvent
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.animation.AnimationUtils
import android.widget.Toast
import androidx.core.content.ContextCompat
import androidx.core.os.bundleOf
import androidx.fragment.app.FragmentActivity
import androidx.fragment.app.commit
import androidx.fragment.app.setFragmentResultListener
import androidx.lifecycle.lifecycleScope
import com.google.firebase.analytics.FirebaseAnalytics
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.twentyfouri.androidcore.epg.EpgStyle
import com.twentyfouri.androidcore.epg.EpgView
import com.twentyfouri.androidcore.epg.TextStyle
import com.twentyfouri.androidcore.epg.model.EpgChannel
import com.twentyfouri.androidcore.epg.model.EpgEvent
import com.twentyfouri.androidcore.epg.model.EpgLabelPos
import com.twentyfouri.androidcore.epg.model.EpgSettings
import com.twentyfouri.androidcore.epg.model.EpgTimeProvider
import com.twentyfouri.androidcore.epg.model.EpgTimeline
import com.twentyfouri.androidcore.epg.model.FocusSettings
import com.twentyfouri.androidcore.utils.styling.TemplateColors
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.smartmodel.serialization.SmartDataValue
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PageType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ApiMissingException
import com.twentyfouri.tvlauncher.common.extensions.ifElse
import com.twentyfouri.tvlauncher.common.extensions.ifNotNull
import com.twentyfouri.tvlauncher.common.extensions.ifNull
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.ui.TvLauncherToast
import com.twentyfouri.tvlauncher.common.ui.messagedialog.CANCEL
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogAction
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogCodes
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogDismissListener
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogFragment
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogFragmentListener
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogModel
import com.twentyfouri.tvlauncher.common.ui.messagedialog.OPTION_A
import com.twentyfouri.tvlauncher.common.utils.NetworkConnectionState
import com.twentyfouri.tvlauncher.common.utils.RestrictionChecker
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.data.EpgChannelExt
import com.twentyfouri.tvlauncher.data.EpgEventExt
import com.twentyfouri.tvlauncher.data.ListPickerItem
import com.twentyfouri.tvlauncher.databinding.FragmentEpgBinding
import com.twentyfouri.tvlauncher.extensions.NO_DATA_ID
import com.twentyfouri.tvlauncher.extensions.buildNoData
import com.twentyfouri.tvlauncher.extensions.dateToString
import com.twentyfouri.tvlauncher.extensions.getEventsClone
import com.twentyfouri.tvlauncher.extensions.isSameDay
import com.twentyfouri.tvlauncher.extensions.seekingRuleAllowsStartover
import com.twentyfouri.tvlauncher.extensions.toDateTime
import com.twentyfouri.tvlauncher.receiver.ScreenOnOffReceiver
import com.twentyfouri.tvlauncher.ui.actions.ActivityPlayerAction
import com.twentyfouri.tvlauncher.utils.MutexProvider
import com.twentyfouri.tvlauncher.utils.Navigator
import com.twentyfouri.tvlauncher.utils.isEpgFragmentDisplayedOverPlayer
import com.twentyfouri.tvlauncher.viewmodels.EpgViewModel
import com.twentyfouri.tvlauncher.viewmodels.MetadataViewModel
import com.twentyfouri.tvlauncher.widgets.SettingsItemType
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.joda.time.DateTime
import org.joda.time.DateTimeZone
import org.joda.time.format.DateTimeFormat
import org.koin.androidx.viewmodel.ext.android.getViewModel
import java.util.Date
import java.util.Locale
import timber.log.Timber
import java.sql.Time
import java.util.*
import kotlin.collections.ArrayList
import kotlin.math.max
import kotlin.math.min

class EpgFragment : BaseFragment() {

    override val showTopbarWhenNavigated = true
    private var inactivityStart: Long = TimeProvider.nowMs()
    private var binding: FragmentEpgBinding? = null
    private var focusedChannelPosition: Int? = null
    private var lastFocusedEventId: String? = null
    private var focusedEvent: EpgEventExt? = null
    private var epgFragmentVisible = true

    private var pendingForceFocusStoredEvent = false;

    private val config = Flavor().getEpgConfigValues

    private var shouldFocusLive = false
    private var isUserScrolling = false
    private var pendingFocusAfterFragmentAttached = true
    private var rememberCurrentFocusOnResume = false
    private var focusResolved: Boolean = false
    private var focusResolvingInProgress: Boolean = false

    private val epgTimeProvider = object : EpgTimeProvider {

        // uses short time for TIMEZONE_ID time zone
        private val dtfShortTime =
            DateTimeFormat.shortTime().withZone(DateTimeZone.forID(Flavor().getTimezoneID()))

        // always uses 24h time omitting locale configuration for TIMEZONE_ID time zone
        private val dtfShortTime24h = DateTimeFormat.forPattern("HH:mm")
            .withZone(DateTimeZone.forID(Flavor().getTimezoneID()))

        override fun getShortTime(timeMillis: Long, force24hTime: Boolean) =
            if (force24hTime) dtfShortTime24h.print(timeMillis) else dtfShortTime.print(timeMillis)

        override fun getNow() = TimeProvider.nowMs()
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?,
    ): View = FragmentEpgBinding.inflate(
        inflater,
        container,
        false
    ).apply {
        lifecycleOwner = this@EpgFragment.viewLifecycleOwner
        binding = this
    }.root

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        try {
            val epgViewModel = getViewModel<EpgViewModel>()

            (activity as? MainActivity)?.getTopBar()?.selectByPageType(PageType.EPG, true)
            binding?.viewModel = epgViewModel

            epgViewModel.isDatabasePopulated.observe(viewLifecycleOwner) {
                if (it) {
                    Timber.tag("Flow").d("isDatabasePopulated ... initEpg")
                    initEpg(binding!!, epgViewModel)
                    setupRecordingObservers()
                    epgViewModel.isDatabasePopulated.removeObservers(viewLifecycleOwner)
                }
            }
        } catch (t: ApiMissingException) {
            Navigator.getInstance().navigateCustom(Navigator.NAVIGATE_CUSTOM_OFFLINE)
        }
    }

    override fun onDestroyView() {
        binding = null
        super.onDestroyView()
    }

    private fun setupRecordingObservers() {
        (activity as? MainActivity)?.getBindingSafe()?.viewModel?.recordingClicked?.observe(
            viewLifecycleOwner
        ) {
            if (epgFragmentVisible) binding?.epgDescription?.startStopRecording()
        }
        //TODO make it with less encapsulation breaks
        binding?.epgDescription?.binding?.viewModel?.userTriggeredRecordingState?.observe(
            viewLifecycleOwner
        ) {
            when (it) {
                MetadataViewModel.RecordingState.WILL_BE_RECORDED,
                MetadataViewModel.RecordingState.WILL_BE_AUTO_RECORDED,
                -> focusedEvent?.addLabel(binding?.viewModel?.getRecordingEpgEventLabel())

                else -> focusedEvent?.removeRecordingLabel()
            }
            binding?.epgView?.postInvalidateOnAnimation()
        }
        //TODO make it with less encapsulation breaks
        binding?.epgDescription?.binding?.viewModel?.recordingState?.observe(viewLifecycleOwner) {
            when (it) {
                MetadataViewModel.RecordingState.WILL_BE_RECORDED -> {
                    if (lastFocusedEventId == focusedEvent?.id) {
                        focusedEvent?.addLabel(binding?.viewModel?.getRecordingEpgEventLabel())
                    }
                }

                MetadataViewModel.RecordingState.WILL_BE_AUTO_RECORDED -> {
                    binding?.viewModel?.addRecordingLabelsToSeriesEvents(
                        focusedChannelPosition,
                        focusedEvent
                    )
                }

                else -> {
                    if (lastFocusedEventId == focusedEvent?.id) {
                        focusedEvent?.removeRecordingLabel()
                    }
                    binding?.viewModel?.removeRecordingLabelsFromSeriesEvents(
                        focusedChannelPosition,
                        focusedEvent
                    )
                }
            }
            lastFocusedEventId = focusedEvent?.id
            binding?.epgView?.postInvalidateOnAnimation()
        }
    }

    private fun initEpg(binding: FragmentEpgBinding, epgViewModel: EpgViewModel) {
        val epgView = binding.epgView
        //by providing non-null interface we force EpgView to use remembering of last focused event approach
        epgView.epgFocusedEventStore = (context as? EpgView.EpgFocusedEventStore)
        val epgSettings = buildEpgSettings()
        val pxPerDp = resources.displayMetrics.density
        setRangeData()

        val epgStyle = EpgStyle(context, TemplateColors.getInstance().colorPalette)
        epgStyle.channelLayoutMargin = (0.3 * pxPerDp).toInt()
        epgStyle.channelLayoutWidth =
            resources.getDimensionPixelSize(R.dimen.epg_view_channels_width)
        epgStyle.channelLayoutHeight = resources.getDimensionPixelSize(R.dimen.epg_view_line_height)
        epgStyle.channelLayoutPadding =
            resources.getDimensionPixelSize(R.dimen.epg_view_channels_padding)
        epgStyle.channelNumberArea = Rect(
            resources.getDimensionPixelSize(R.dimen.epg_view_channel_number_offset),
            0,
            resources.getDimensionPixelSize(R.dimen.epg_view_channel_number_offset) + resources.getDimensionPixelSize(
                R.dimen.epg_view_channel_number_width
            ),
            epgStyle.channelLayoutHeight
        )
        epgStyle.channelImageArea = Rect(
            resources.getDimensionPixelSize(R.dimen.epg_view_channel_image_offset),
            0,
            resources.getDimensionPixelSize(R.dimen.epg_view_channel_image_area_width),
            epgStyle.channelLayoutHeight
        )
        epgStyle.timeBarAreaPadding =
            Rect(resources.getDimensionPixelSize(R.dimen.epg_view_date_picker_width), 0, 0, 0)
        epgStyle.borderColor = ColorStateList(arrayOf(intArrayOf()),
            intArrayOf(ContextCompat.getColor(requireContext(), R.color.epg_border)))
        epgStyle.borderWidth = (1.25 * pxPerDp).toInt()
        epgStyle.borderWidthSelected = epgStyle.borderWidth
        epgStyle.borderWidthFocused = (-1) * (3.75 * pxPerDp).toInt()
        epgStyle.channelBorderColor = ColorStateList(arrayOf(intArrayOf()),
            intArrayOf(ContextCompat.getColor(requireContext(), R.color.epg_border)))
        epgStyle.timeLineDrawableForTimebar = ColorDrawable(Color.TRANSPARENT)
        epgStyle.timeLineDrawableForEventArea =
            resources.getDrawable(R.drawable.live_line_indicator, null)
        epgStyle.timeLineWidth = resources.getDimension(R.dimen.epg_view_timeline_width).toInt()
        epgStyle.timeBarBackgroundColor = ContextCompat.getColor(requireContext(), R.color.epg_timeline)
        epgStyle.timeBarLeftMargin = 0
        epgStyle.timeTitleBackgroundColor =
            ContextCompat.getColor(requireContext(), R.color.epg_event_background_color)
        epgStyle.channelNumberTextStyle = TextStyle(
            ContextCompat.getColor(requireContext(), R.color.epg_channel_number_text_color),
            Typeface.DEFAULT,
            resources.getDimensionPixelSize(R.dimen.epg_view_channel_number_text_size)
        ).apply { textAlign = Paint.Align.LEFT }
        epgStyle.timeBarTextStyle = TextStyle(
            ContextCompat.getColor(requireContext(), R.color.epg_timeline_text_color),
            Typeface.DEFAULT,
            resources.getDimensionPixelSize(R.dimen.epg_view_event_label_text_size)
        ).apply { textAlign = Paint.Align.CENTER }
        epgStyle.loadingTextStyle = TextStyle(
            resources.getColorStateList(R.color.epg_loading_color_selector, null),
            Typeface.DEFAULT,
            resources.getDimensionPixelSize(R.dimen.epg_view_loading_text_size)
        )
        epgStyle.eventBackgroundDrawable =
            resources.getDrawable(R.drawable.epg_event_background, null)
        epgStyle.channelBackgroundDrawable = ColorDrawable().apply {
            color = ContextCompat.getColor(requireContext(), R.color.epg_channel_background_color)
        }
        epgStyle.channelBackgroundDrawableSelected = ColorDrawable().apply {
            color = ContextCompat.getColor(requireContext(), R.color.epg_channel_background_color)
        }
        epgStyle.channelLayoutShadowWidth =
            resources.getDimensionPixelSize(R.dimen.epg_channel_list_shadow_width)
        epgStyle.channelListShadowDrawable =
            resources.getDrawable(R.drawable.epg_channel_list_shadow_background, null)
        epgStyle.eventTextStyle = TextStyle(
            resources.getColorStateList(R.color.epg_text_color_selector, null),
            Typeface.DEFAULT,
            resources.getDimensionPixelSize(R.dimen.epg_view_event_label_text_size)
        )
        val liveTextStyle =
            TextStyle(ContextCompat.getColor(requireContext(), R.color.epg_text_color_selector),
                Typeface.DEFAULT,
                30)
        epgStyle.timeBarLiveTextStyle = liveTextStyle

        epgViewModel.epgData.observe(viewLifecycleOwner) { epgData ->
            //Log.d("maxiEpg", "epgData observer updated")
            epgView.epgData
                .ifNull { epgView.setEpgData(epgData, epgSettings, epgStyle) }
                .ifNotNull {
                    if (shouldFocusLive && epgData.hasData()) {
                        scrollToLive()
                    }
                    if (epgView.focusedChannelPosition >= 0
                        && !isUserScrolling
                        && (epgView.focusedEvent == null || epgView.focusedEvent.id == NO_DATA_ID)
                    ) {
                        epgView.focusedChannelPosition =
                            (context as? EpgView.EpgFocusedEventStore)?.storedChannelPosition
                                ?: epgView.focusedChannelPosition
                    }
                    epgView.redraw()
                }
            if ((activity as? MainActivity)?.isEpgFragmentDisplayedOverPlayer() == true && pendingFocusAfterFragmentAttached) {
                if (pendingForceFocusStoredEvent) {
                    pendingForceFocusStoredEvent = false
                    forceFocusStoredEvent()
                } else {
                    focusStoredChannelAndEvent(shouldFocusLive)
                }
            }
        }

        epgView.addEpgDayChangeListener({ date, _ ->
            Timber.tag("EpgFragment").d(date.toString())
            epgViewModel.currentDay = date
            epgViewModel.setDatepickText(date.dateToString(true, requireContext()))
        }, Locale("en-us"))

        epgView.addChannelsVisibleListener { firstVisible, lastVisible, fromTime, toTime ->
            var modifiedFrom: Long
            var modifiedTo: Long
            val halfDay = config.HORIZONTAL_PAGE_SIZE_MS
            val centerTime = (fromTime + toTime) / 2
            val nearestHalfDayInPast = centerTime - (centerTime % halfDay) // round to 12 hours
            val nearestHalfDayInFuture = nearestHalfDayInPast + halfDay // round to 12 hours
            if (centerTime - nearestHalfDayInPast < nearestHalfDayInFuture - centerTime) {
                // we are closer to half day in past
                modifiedFrom = nearestHalfDayInPast - halfDay
                modifiedTo = nearestHalfDayInPast + halfDay
            } else {
                // we are closer to half day in future
                modifiedFrom = nearestHalfDayInFuture - halfDay
                modifiedTo = nearestHalfDayInFuture + halfDay
            }
            modifiedFrom = max(modifiedFrom, epgViewModel.epgAbsoluteStartTime.millis)
            modifiedTo = min(modifiedTo, epgViewModel.epgAbsoluteEndTime.millis)
            val channelsToLoad = ArrayList<SmartMediaItem>()

            val firstVisibleWithBumper = max(0, firstVisible - 2)
            val lastVisibleWithBumper = max(0, lastVisible + 2)
            val firstChannelOnPage =
                firstVisibleWithBumper - (firstVisibleWithBumper % config.VERTICAL_PAGE_SIZE_CHANNEL_COUNT) // round to first
            val lastChannelOnPage =
                lastVisibleWithBumper + config.VERTICAL_PAGE_SIZE_CHANNEL_COUNT - (lastVisibleWithBumper % config.VERTICAL_PAGE_SIZE_CHANNEL_COUNT) // round to first

            for (index in firstChannelOnPage..lastChannelOnPage) {
                epgViewModel.getChannel(index)
                    ?.apply {
                        val channel = this as EpgChannelExt
                        val rangeToLoad = channel.getRangeLeftToBeLoaded(modifiedFrom, modifiedTo)
                        if (rangeToLoad.getWidth() > 0 && rangeToLoad.status == EpgChannelExt.Status.EMPTY) {
                            channel.markLoadingRange(modifiedFrom, modifiedTo)
                            channelsToLoad.add(channel.smartMediaItem)
                        }
                    }
                //?: Log.d("maxiEpg", "ERROR channel $index missing - cannot be loaded")
            }

//            Log.d("maxiEpg", "channelVisibleListener loadEvents " +
//                    "first:$firstVisible, " +
//                    "last:$lastVisible, " +
//                    "from:${DateTime(fromTime).let{"${it.hourOfDay}:${it.minuteOfHour}"}}, " +
//                    "to:${DateTime(toTime).let{"${it.hourOfDay}:${it.minuteOfHour}"}}"
//            )
            if (channelsToLoad.isNotEmpty()) epgViewModel.loadEvents(channelsToLoad,
                modifiedFrom,
                modifiedTo)
            epgView.postInvalidateOnAnimation()
        }

        epgView.addOnEventFocusedListener { channelPosition, event ->
            focusedChannelPosition = channelPosition
            focusedEvent = null
            isUserScrolling = false
            if (epgView.isFocused && event == null) {
                epgViewModel.setFocusedEvent(buildNoData(1, 2))
            } else {
                epgViewModel.setFocusedEvent(event)
            }
            logSelectedEvent(event, channelPosition, epgViewModel, false)
            when {
                event is EpgEventExt && event.id != NO_DATA_ID -> {
                    focusedEvent = event
                    binding.epgDescription.bindOnEpg(event.smartMediaItem.reference, event.smartMediaItem.channelReference)
                }
                event != null -> {
                    val channel = epgViewModel.getChannel(channelPosition)
                    val isAppChannel =
                        Flavor().getAppChannelsDelegate()?.getAppChannelIDs()?.contains(channel?.id)

                    isAppChannel.ifTrue {
                        channel?.let {
                            Flavor().getAppChannelsDelegate()?.getAppChannelMediaReference(it)
                        }?.let { binding.epgDescription.bindOnEpg(it) }
                    }.ifElse {
                        // just pass pure reference -> will be interpreted as UNKNOWN media in repository
                        binding.epgDescription.bindOnEpg(
                            aMediaReference = object : SmartMediaReference() {
                                override fun equals(other: Any?): Boolean {
                                    return false
                                }

                                override fun hashCode(): Int {
                                    return 0
                                }
                            },
                            fallbackMediaReference = (event as? EpgEventExt)?.smartMediaItem?.channelReference
                        )
                    }
                    // just debug toast message
//                    val time = DateTimeRepository(context!!).formatDateRange(DateTime(event.startTime), DateTime(event.endTime))
//                    Toast.makeText(context, "No Data event focused - $time", Toast.LENGTH_LONG).show()
                }
                else -> {
                    // just pass pure reference -> will be interpreted as UNKNOWN media in repository
                    binding.epgDescription.bindOnEpg(object : SmartMediaReference() {
                        override fun equals(other: Any?): Boolean {
                            return false
                        }

                        override fun hashCode(): Int {
                            return 0
                        }
                    })
                }
            }
        }

        epgView.setOnKeyListener { _, _, event ->
            (event.keyCode == KeyEvent.KEYCODE_INFO && event.action == KeyEvent.ACTION_UP && focusedChannelPosition != null).ifTrue {
                val channel = epgViewModel.getChannel(focusedChannelPosition!!)
                if (channel is EpgChannelExt) {
                    focusedEvent?.let { epgEvent ->
                        if (Flavor().shouldEpgStoreFocusOnDetailOpen) {
                            focusedChannelPosition?.let { channelPosition ->
                                (context as? EpgView.EpgFocusedEventStore)?.storeEventAndChannelPosition(
                                    epgEvent,
                                    channelPosition)
                                rememberCurrentFocusOnResume = true
                            }
                        }
                        Navigator.getInstance()
                            .navigate(SmartNavigationTarget.toDetailPage(epgEvent.smartMediaItem.reference))
                        lastFocusedViewBeforeSubscriptions = binding.epgView
                    } ?: TvLauncherToast.makeText(
                        requireContext(),
                        resources.getString(R.string.epg_action_not_available),
                        Toast.LENGTH_SHORT
                    )?.show()
                }
            }
        }

        epgView.addEpgViewClickListener(object : EpgView.EpgViewClickListener {
            override fun onChannelClicked(channelPosition: Int, epgChannel: EpgChannel?) {
                // can not happen with focus navigation
            }

            override fun onEventClicked(
                channelPosition: Int,
                programPosition: Int,
                epgEvent: EpgEvent,
            ) {
                // for no data events, that are currently "live", at least navigate to player directly
                logSelectedEvent(epgEvent, channelPosition, epgViewModel, true)
                val eventChannel = epgView.epgData.getChannel(channelPosition)

                val skipDetail = (Flavor().navigationEpgLiveSkipDetail || epgEvent.id == NO_DATA_ID) && epgEvent.isLive(TimeProvider.nowMs())
                val isFlavorSpecificAppChannel = (Flavor().getAppChannelsDelegate()?.getAppChannelIDs()?.contains(eventChannel.id) == true)

                when {
                    isFlavorSpecificAppChannel.not() && skipDetail -> {
                        (epgView.epgData.getChannel(channelPosition) as? EpgChannelExt)?.smartMediaItem?.let { channelMediaItem ->
                            RestrictionChecker.checkRestrictions(
                                context = requireContext(),
                                mediaItem = channelMediaItem,
                                doOnSuccess = {
                                    val target =
                                        SmartNavigationTarget.toPlayer(channelMediaItem.reference,
                                            null)
                                    target.extras =
                                        SmartDataValue.from(PlayerLocation.FOREGROUND.name)
                                    Navigator.getInstance().navigate(target)
                                }
                            )
                        }
                    }
                    isFlavorSpecificAppChannel -> {
                        val prevChID: String =
                            if (channelPosition == 0) "-1" else epgView.epgData.getChannel(
                                channelPosition - 1).id
                        val nextChID: String =
                            // We subtract one to channel count in the equality check because position starts at 0 while count at 1.
                            if (channelPosition == epgView.epgData.channelCount - 1) "-1" else epgView.epgData.getChannel(
                                channelPosition + 1).id
                        val launchIntent: Intent? = Flavor().getAppChannelsDelegate()
                            ?.getAppChannelIntent(eventChannel.id, prevChID, nextChID)
                        launchIntent?.let { startActivity(it) }
                    }
                    epgEvent.id == NO_DATA_ID -> {
                        //do not try to open detail if event id is "no_data" because it will case call
                        //to get detail with nonsense value. Live event is handled by `skipDetail` branch
                    }
                    else -> (epgEvent as? EpgEventExt)?.openDetail(epgEvent, channelPosition)
                }
            }

            private fun EpgEventExt.openDetail(epgEvent: EpgEvent, channelPosition: Int) {
                if (Flavor().shouldEpgStoreFocusOnDetailOpen) {
                    (context as? EpgView.EpgFocusedEventStore)?.storeEventAndChannelPosition(epgEvent, channelPosition)
                    rememberCurrentFocusOnResume = true
                }
                Navigator.getInstance()
                    .navigate(SmartNavigationTarget.toDetailPage(smartMediaItem.reference))
                lastFocusedViewBeforeSubscriptions = binding.epgView
            }

        })

        binding.epgDatePick.onFocusChangeListener = View.OnFocusChangeListener { v, hasFocus ->
            if (hasFocus) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("EPG day button focused")
            val animId =
                if (hasFocus) R.anim.scale_in_card_from_left else R.anim.scale_out_card_from_left
            val anim = AnimationUtils.loadAnimation(v?.context, animId)
            Handler().post {
                v?.startAnimation(anim)
                anim.fillAfter = true
            }
        }

        binding.epgDatePick.setOnClickListener {
            Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("EPG day button selected")
            val currentSelectedDay = epgViewModel.currentDay?.toDateTime()
            var currentItem = ListPickerFragment.DEFAULT_SELECTED_ITEM_INDEX_MIDDLE
            val data = ArrayList<ListPickerItem>().apply {
                val today = TimeProvider.now()
                for (i in (-1 * config.EPG_DAYS_INTO_PAST)..(config.EPG_DAYS_INTO_FUTURE)) {
                    val dateTime = today.plusDays(i)
                    val date = dateTime.toDate()
                    currentSelectedDay?.isSameDay(dateTime).ifTrue { currentItem = size }
                    add(
                        DateListPickerItem(
                            id = i.toString(),
                            label = date.dateToString(false, requireContext()),
                            date = date
                        )
                    )
                }
            }
            val fragment = ListPickerFragment.newInstance(
                title = getString(R.string.epg_date_picker_dialog_title),
                defaultPosition = currentItem,
                data = data,
                selectedPosition = ListPickerFragment.SELECTED_POSITION_MIDDLE
            )
            Timber.tag("EpgFragment.initEpg").d("commit $fragment")
            FirebaseCrashlytics.getInstance().log("EpgFragment.initEpg: commit $fragment")
            activity?.supportFragmentManager?.commit {
                add(R.id.fullscreen_frame, fragment, ListPickerFragment::class.java.simpleName)
                addToBackStack(null)
            }
            fragment.setFragmentResultListener(ListPickerFragment.REQUEST_KEY) { _, bundle ->
                val dateListPickerItem = bundle.get(ListPickerFragment.BUNDLE_KEY_RESULT) as? DateListPickerItem
                if (dateListPickerItem != null) {
                    (context as? EpgView.EpgFocusedEventStore)?.storeEventAndChannelPosition(
                        null,
                        focusedChannelPosition ?: 0
                    )
                    /*fix attempt for
                    * NullPointerException
                    * at com.twentyfouri.androidcore.epg.EpgView.scrollToDate(EpgView.java:1761)
                      at com.twentyfouri.tvlauncher.ui.EpgFragment.onActivityResult(EpgFragment.kt:639)
                    * If EpgView is not available, we don't scroll there*/
                    try {
                        binding.epgView.scrollToDate(Date(dateListPickerItem.date.time - 30 * 60 * 1000)) //subtract half an hour to move desired time into better position on screen
                    } catch (e: NullPointerException) {
                        e.printStackTrace()
                    }
                }
                binding.epgDatePick.requestFocus()
            }
        }

        //this focus needs to be requested to update focus in topbar - needed for GUIDE button handling
        //but ignore it if there is stored event which should be focused
        if (arguments?.getSerializable(config.REQUEST_FOCUS) == true && !isChannelAndEventStored()) {
            binding.epgDatePick.requestFocus()
        }
    }

    private fun logSelectedEvent(event: EpgEvent?, channelPosition: Int, model: EpgViewModel, clicked: Boolean) {
        val chInfo = "(CH:${model.getChannel(channelPosition)?.channelNumber})"
        val eventInfo = when(event) {
            null -> {
                ""
            }
            else -> {
                val now = TimeProvider.nowMs()
                val eventTime = when {
                    event.startTime > now -> "→"
                    event.endTime < now -> "←"
                    else -> "↓"
                }
                "${event.title} ($eventTime ${Flavor().getLogInfoFromEPGEvent(event)})"
            }
        }
        val action = if(clicked) "selected" else "focused"
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("EPG event $action: $eventInfo $chInfo")
    }

    private fun dateToString(date: Date, dayOnly: Boolean): String {
        val dateTime = date.toDateTime()
        val today = TimeProvider.now()
        return when {
            dateTime.isSameDay(today) -> getString(R.string.epg_date_today)
            dateTime.isSameDay(today.minusDays(1)) -> getString(R.string.epg_date_yesterday)
            dateTime.isSameDay(today.plusDays(1)) -> getString(R.string.epg_date_tomorrow)
            else -> {
                val pattern = if (dayOnly) "EEEE" else "EEEE, d MMMM"
                val formatter = DateTimeFormat.forPattern(pattern).withLocale(Locale.getDefault())
                formatter.print(dateTime).capitalize()
            }
        }
    }

    fun stayOnItemAfterResume() {
        rememberCurrentFocusOnResume = true
    }

    private fun resolveEpgFocusOnScreenReturn() {
        if (focusResolvingInProgress.not() && focusResolved.not()) {
            focusResolvingInProgress = true
            if (rememberCurrentFocusOnResume) {
                shouldFocusLive = false
                rememberCurrentFocusOnResume = false
                if (Flavor().shouldEpgStoreFocusOnDetailOpen) {
                    if (binding?.epgView?.epgData == null) {
                        pendingForceFocusStoredEvent = true
                    } else {
                        forceFocusStoredEvent()
                    }
                } else {
                    scrollToLive()
                }
            } else {
                scrollToLive()
            }
            focusResolvingInProgress = false
            focusResolved = true
        }
    }

    override fun onPause() {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Page paused: EPG")
        super.onPause()
        focusResolved = false
        inactivityStart = TimeProvider.nowMs()
        MutexProvider.clear()
    }

    override fun onResume() {
        super.onResume()
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Page resumed: EPG")
        if (this.isHidden) return
        FirebaseAnalytics.getInstance(requireContext()).logEvent(FirebaseAnalytics.Event.SCREEN_VIEW, bundleOf(
            FirebaseAnalytics.Param.SCREEN_NAME to "EPG",
            FirebaseAnalytics.Param.SCREEN_CLASS to "EpgFragment"
        ))
        NetworkConnectionState.instance.waitForConnection(lifecycleOwner = viewLifecycleOwner,
            skipWaiting = !ScreenOnOffReceiver.screenWasOff) {
            if (isChannelAndEventStored()) {
                resolveEpgFocusOnScreenReturn()
            } else if (TimeProvider.nowMs() > inactivityStart + config.INACTIVITY_REFRESH_MS) {
                // clear loaded ranges -> refresh epg from-to
                binding?.viewModel?.clearRanges()
                setRangeData()
                binding?.epgView?.recalculateAndRedraw(false, false)
                // scroll by 1px to force channel visible listener
                binding?.epgView?.safeScrollBy(1, 0)
            }
            if ((activity as? MainActivity)?.isEpgFragmentDisplayedOverPlayer() != true
                && Flavor().resetEpgFragmentOnButtonClick
            ) {
                reset()
            }
        }
    }


    override fun onHiddenChanged(hidden: Boolean) {
        epgFragmentVisible = hidden.not()
        if (hidden) focusResolved = false
        if ((activity as MainActivity).getEpgPressed()) {
            //EPG button was pressed. Focus will be resolved in onResume()
            return super.onHiddenChanged(hidden)
        }
        if (!hidden) {
            resolveEpgFocusOnScreenReturn()
        }
        super.onHiddenChanged(hidden)
    }

    override fun reset() {
        rememberCurrentFocusOnResume = false
        shouldFocusLive = true
    }

    //unfortunately direct set of focus is not enough because other elements around epg view are trying to steal
    //focus (like topbar) so direct attempt is made and one more is postponed
    fun forceFocusStoredEvent() {
        val shouldFocusLive = this.shouldFocusLive
        binding?.epgView?.requestFocus()
        forceJumpToTime(shouldFocusLive)
        focusStoredChannelAndEvent(shouldFocusLive)

        Handler(Looper.getMainLooper()).post {
            binding?.epgView?.requestFocus()
            forceJumpToTime(shouldFocusLive)
            focusStoredChannelAndEvent(shouldFocusLive)
        }
    }

    fun scrollToLive() {
        val epgView = binding?.epgView ?: return
        if (!epgView.hasFocus()) epgView.requestFocus()
        val channelPosition = (context as? EpgView.EpgFocusedEventStore)?.storedChannelPosition ?: 0
        epgView.scrollToTimeAndChannel(
            channelPosition,
            getViewportLiveCenterTimeMs()
        )
        selectChannel(channelPosition, true)
    }

    private fun getViewportLiveCenterTimeMs() =
        DateTime.now().millis - resources.getInteger(R.integer.epg_seconds_in_viewport).toLong() * 1000 / 2

    private fun forceJumpToTime(shouldFocusLive: Boolean) {
        val channelPosition = (context as? EpgView.EpgFocusedEventStore)?.storedChannelPosition
        if (shouldFocusLive)
            scrollToLive()
        else {
            val event = (context as? EpgView.EpgFocusedEventStore)?.storedEvent
            if (channelPosition != null && event != null) {
                binding?.epgView?.scrollToTimeAndChannel(channelPosition, findProperTimeToScrollTo(event))
            }
        }
    }

    private fun focusStoredChannelAndEvent(shouldFocusLive: Boolean) {
        if (shouldFocusLive.not() && isUserScrolling.not()) {
            val epgView = binding?.epgView
            val channelPosition = (context as? EpgView.EpgFocusedEventStore)?.storedChannelPosition
            val storedEvent = (context as? EpgView.EpgFocusedEventStore)?.storedEvent
            if (storedEvent != null && channelPosition != null && epgView != null) {
                val ownEvent = epgView.epgData?.getEventsClone(channelPosition)?.find {
                    it.startTime == storedEvent.startTime && it.endTime == storedEvent.endTime && it.title == storedEvent.title
                }
                epgView.scrollToTimeAndChannel(channelPosition, findProperTimeToScrollTo(storedEvent))
                if (ownEvent != null) {
                    epgView.setFocusedEvent(channelPosition, ownEvent)
                    pendingFocusAfterFragmentAttached = false
                }
            }
        }
    }

    // if event happens to be live horizontal scroll is set that epgView scrolls to
    // current live time instead of start of the event
    private fun findProperTimeToScrollTo(event: EpgEvent?): Long {
        event ?: return getViewportLiveCenterTimeMs()
        return if (event.isLive(TimeProvider.nowMs())) {
            getViewportLiveCenterTimeMs()
        } else {
            event.startTime
        }
    }

    private fun isChannelAndEventStored(): Boolean {
        (context as? EpgView.EpgFocusedEventStore)?.let {
            return it.storedEvent != null
        }
        return false
    }

    fun openPlayer(startFromBeginning: Boolean = false) {
        focusedEvent?.smartMediaItem?.let { mediaItem ->
            mediaItem.isPastEvent().ifTrue { rememberCurrentFocusOnResume = true }
            mediaItem.isNotFutureEvent().ifTrue {
                if (!mediaItem.seekingRuleAllowsStartover(isCatchup = mediaItem.isPastEvent())) {
                    //Startover block detected, check event entitlement
                    if (mediaItem.isPastEvent() || startFromBeginning) {
                        TvLauncherToast.makeText(
                            context as Context,
                            R.string.player_action_not_available_channel,
                            Toast.LENGTH_SHORT
                        )?.show()
                        return
                    }
                }
                if (startFromBeginning) {
                    showStartOverConfirmationDialog(mediaItem)
                    return
                }
                val target = SmartNavigationTarget.toPlayer(mediaItem.reference, null)
                target.extras = SmartDataValue.from(PlayerLocation.FOREGROUND.name)
                Navigator.getInstance().navigate(target)
            }
        }
    }

    private fun SmartMediaItem.isNotFutureEvent() = (startDate != null && startDate!!.isBeforeNow)
    private fun SmartMediaItem.isPastEvent(): Boolean =
        (endDate?.millis ?: 0 < TimeProvider.nowMs())

    private fun setRangeData() {
        binding?.viewModel?.epgAbsoluteStartTime = TimeProvider.now().minusDays(config.EPG_DAYS_INTO_PAST)
        binding?.viewModel?.epgAbsoluteEndTime = TimeProvider.now().plusDays(config.EPG_DAYS_INTO_FUTURE)
    }

    private fun buildEpgSettings(): EpgSettings {
        val hoursForwardMillis = (config.EPG_DAYS_INTO_FUTURE * config.MILLISECONDS_IN_DAY).toLong()
        val hoursBackwardMillis = (config.EPG_DAYS_INTO_PAST * config.MILLISECONDS_IN_DAY).toLong()
        return EpgSettings.EpgSettingsBuilder()
            .isRTL(binding?.epgView?.layoutDirection == View.LAYOUT_DIRECTION_RTL)
            .hoursInViewport(resources.getInteger(R.integer.epg_seconds_in_viewport)
                .toLong() * 1000)
            .setEpgTitleProvider(null)
            .setEpgTimeProvider(epgTimeProvider)
            .showTimeLine(EpgTimeline.ON_TOP, false)
            .daysForwardMillis(hoursForwardMillis)
            .daysBackwardMillis(hoursBackwardMillis)
            .setForce24hTime(true)
            .timeBarLiveText(null)
            .setTimeBarWithBorders(true)
            .showBorderAroundTitle(true)
            .setSingleDirectionFling(true)
            .setSingleDirectionScroll(true)
            .isInShortenWordsMode(true)
            .setFocusedEventOnTopEverything(true)
            .setEventLabelMaxLines(1)
            .setLabelPosition(EpgLabelPos.AFTER_TEXT)
            .loadingChannelEventsText(getString(R.string.epg_loading_channel))
            .setFocusSettings(FocusSettings().apply {
                focusStartAreaPercentY = 35
                focusEndAreaPercentY = 50
                focusStartAreaPercentX = 5
                focusEndAreaPercentX = 60
            })
            .isStickyText(true)
            .build()
    }

    override fun onBackPressed(topbarFocused: Boolean): BackPressAction {
        if ((context as? MainActivity)?.isEpgFragmentDisplayedOverPlayer() == true) {
            (context as? MainActivity)?.togglePlayerFrontBack()
            return BackPressAction.RETURN
        }
        if (binding?.epgView?.hasFocus() == true) {
            binding?.epgDatePick?.requestFocus()
            return BackPressAction.RETURN
        }
        if (binding?.epgDatePick?.hasFocus() != true) (context as? ActivityPlayerAction)?.playerStop(
            false)
        val backPressAction = super.onBackPressed(topbarFocused)
//        val backPressAction = BackPressAction.POP_BACK_STACK //uncomment if back on EPG should close the EPG
        return backPressAction
    }

    fun scrollEPGPageUp() {
        val epgView = binding?.epgView ?: return
        if (!epgView.hasFocus()) epgView.requestFocus()
        var currentChannelPosition = epgView.focusedChannelPosition
        if (currentChannelPosition < 0) currentChannelPosition = 0
        var futureChannelPosition = currentChannelPosition + config.SCROLL_PAGE_SIZE
        var futureChannelScrollPosition = currentChannelPosition + config.SCROLL_PAGE_SIZE
        val maxChannelPosition = epgView.epgData?.channelCount?.minus(1) ?: 0
        val maxChannelScrollPosition =
            epgView.epgData?.channelCount?.minus(config.SCROLL_PAGE_SIZE) ?: 0
        if (maxChannelPosition < futureChannelPosition) futureChannelPosition =
            currentChannelPosition + maxChannelPosition - currentChannelPosition
        if (maxChannelScrollPosition < futureChannelScrollPosition) {
            if (currentChannelPosition > maxChannelScrollPosition) currentChannelPosition -= config.SCROLL_PAGE_SIZE - 1
            futureChannelScrollPosition =
                currentChannelPosition + maxChannelScrollPosition - currentChannelPosition
        }
        selectChannel(futureChannelPosition, false)
        scrollToChannel(futureChannelScrollPosition)
    }

    fun scrollEPGPageDown() {
        val epgView = binding?.epgView ?: return
        if (!epgView.hasFocus()) epgView.requestFocus()
        var currentChannelPosition = epgView.focusedChannelPosition
        if (currentChannelPosition < 0) currentChannelPosition = 0
        var channelsDecrement = config.SCROLL_PAGE_SIZE
        if (currentChannelPosition < config.SCROLL_PAGE_SIZE) channelsDecrement =
            currentChannelPosition
        val futureChannelPosition = currentChannelPosition - channelsDecrement
        val maxChannelScrollPosition =
            epgView?.epgData?.channelCount?.minus(config.SCROLL_PAGE_SIZE) ?: 0
        if (currentChannelPosition > maxChannelScrollPosition) currentChannelPosition -= config.SCROLL_PAGE_SIZE - 1
        val futureChannelScrollPosition = currentChannelPosition - channelsDecrement
        selectChannel(futureChannelPosition, false)
        scrollToChannel(futureChannelScrollPosition)
    }

    fun scrollEPGDay(where: Int) {
        val epgView = binding?.epgView ?: return
        if (!epgView.hasFocus()) epgView.requestFocus()
        val current = DateTime(binding?.epgView?.dayDateTime)
        when (where) {
            SCROLL_DAY_FUTURE -> epgView.scrollToDate(current.plusDays(1).toDate())
            SCROLL_DAY_PAST -> epgView.scrollToDate(current.minusDays(1).toDate())
        }
        selectChannel(epgView.focusedChannelPosition, false)
    }


    private fun selectChannel(channelNumber: Int, shouldFocusLive: Boolean) {
        isUserScrolling = true
        val epgView = binding?.epgView
        epgView?.requestLayout()

        //Because of unknown reason it can happen that channelNumber = -1 which cause crash (suspected in the case is focusedChannelPosition)
        //as fallback in this case we will try to select channel 0
        //in case there is not even one channel then exit
        val channelCount = epgView?.epgData?.channelCount ?: 0
        val fixedChannelNumber = when {
            channelNumber in 0 until channelCount -> channelNumber
            channelNumber < 0 && channelCount > 0 -> {
                Timber.tag("EpgFragment")
                    .e("Trying to select channel index $channelNumber. Fallback to channel 0.")
                0
            }
            else -> {
                Timber.tag("EpgFragment")
                    .e("Trying to select channel index $channelNumber while there are $channelCount channels. Request ignored.")
                return
            }
        }

        var selectedEventOnChannel: EpgEvent? = null
        loadEventsIfReady(
            {
                if (shouldFocusLive) {
                    epgView?.epgData?.getEventsClone(fixedChannelNumber)?.forEach { epgEvent ->
                        if (epgEvent.isLive(DateTime.now().millis)) {
                            selectedEventOnChannel = epgEvent
                        }
                    }
                } else {
                    epgView?.epgData?.getEventsClone(fixedChannelNumber)?.forEach { epgEvent ->
                        if (epgEvent.startTime > epgView.dayDateTime.millis && epgEvent.endTime < epgView.dayDateTime.millis) {
                            selectedEventOnChannel = epgEvent
                        }

                    }
                }
                //Do not select loaded event in case that user moved focus outside of epgView
                if (binding?.epgView?.hasFocus() == true) {
                    epgView?.setFocusedEvent(
                        fixedChannelNumber, selectedEventOnChannel
                    )
                    epgView?.selectedEvent = selectedEventOnChannel
                    if (selectedEventOnChannel is EpgEventExt) focusedEvent =
                        selectedEventOnChannel as EpgEventExt?

                    if (shouldFocusLive &&
                        epgView?.dayDateTime?.millis !in DateTime.now()
                            .minusHours(1).millis..DateTime.now().plusHours(1).millis
                    ) {
                        //Sometimes the epgView.recalculateAndRedraw interrupts scrolling to live
                        //and forces viewport scroll back to previously focused event. This branch
                        //makes sure that viewport is really scrolled into live.
                        scrollToLive()
                    }
                }
                this.shouldFocusLive = false
            },
            {
                (epgView?.epgData?.getEvents(fixedChannelNumber)?.size ?: 0) > 0
            }
        )
        val channelToSelect = epgView?.epgData?.getChannel(fixedChannelNumber)
        epgView?.selectedChannel = channelToSelect

        focusedChannelPosition = fixedChannelNumber
    }

    private fun loadEventsIfReady(loadProcedure: () -> Unit, checkIfReadyProcedure: () -> Boolean) {
        try {
            viewLifecycleOwner.lifecycleScope.launch {
                delay(LOAD_EVENTS_POLLING_INTERVAL_MS)
                if (checkIfReadyProcedure()) {
                    delay(LOAD_EVENTS_TIME_PADDING_MS)
                    withContext(Dispatchers.Main) {
                        loadProcedure()
                    }
                } else {
                    loadEventsIfReady(loadProcedure, checkIfReadyProcedure)
                }
            }
        } catch (e: Exception) {
            // fix for java.lang.IllegalStateException
            // Can't access the Fragment View's LifecycleOwner when getView() is null i.e., before onCreateView() or after onDestroyView()
            // loadEventsIfReady is skipped
            e.printStackTrace()
        }
    }

    fun scrollToChannelViaNumberButtons(number: Int) {
        val closestChannelNumber =
            binding?.viewModel?.channelNumbers?.keys?.minByOrNull { if (it >= number) it - number else Int.MAX_VALUE }
        val position = binding?.viewModel?.channelNumbers?.get(closestChannelNumber)
        if (position != null) {
            scrollToChannel(position)
            selectChannel(position, true)
            binding?.epgView?.scrollToTimeAndChannel(
                position,
                getViewportLiveCenterTimeMs()
            )
        }
    }

    private fun scrollToChannel(toChannel: Int) {
        val channelHeight = resources.getDimensionPixelSize(R.dimen.epg_view_line_height)
        val to = toChannel * channelHeight
        val currentScrollY = binding?.epgView?.scrollY
        binding?.epgView?.safeScrollBy(0, to - (currentScrollY ?: 0))
    }

    private fun showStartOverConfirmationDialog(mediaItem: SmartMediaItem) {
        val messageDialogModel = MessageDialogModel(
            message = getString(R.string.player_ok_longpress_action_dialog_event, mediaItem.title),
            description = null,
            optionsButtonText = arrayOf(getString(R.string.button_yes)),
            cancelButtonText = getString(R.string.cancel),
            code = MessageDialogCodes.reproduceFromBeginning
        )
        val confirmMessageDialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        confirmMessageDialogFragment.mDismissListener = object : MessageDialogDismissListener {
            override fun onDismiss() {} //do nothing
        }
        confirmMessageDialogFragment.mListener = object : MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean {
                if (answer !is MessageDialogAction.Result) {
                    return false
                }
                when (answer.type) {
                    OPTION_A -> {
                        val target = SmartNavigationTarget.toPlayer(mediaItem.reference, null, 0)
                        target.extras = SmartDataValue.from(PlayerLocation.FOREGROUND.name)
                        Navigator.getInstance().navigate(target)
                    }
                    CANCEL -> {
                    } //cancel, do nothing
                    else -> {
                    }
                }
                return false
            }
        }
        (view?.context as? FragmentActivity)?.supportFragmentManager?.also {
            confirmMessageDialogFragment.show(
                it,
                "tag_dialog_startover"
            )
        }
    }

    class DateListPickerItem(
        id: String,
        label: String,
        val date: Date,
    ) : ListPickerItem(id, label, "", "", type = SettingsItemType.LABEL)

    companion object {
        val SCROLL_DAY_FUTURE = 1
        val SCROLL_DAY_PAST = -1
        const val LOAD_EVENTS_POLLING_INTERVAL_MS = 200L
        const val LOAD_EVENTS_TIME_PADDING_MS = 1000L
    }

    abstract class EPGConfigValues {
        abstract val SCROLL_PAGE_SIZE: Int
        abstract val INACTIVITY_REFRESH_MS: Int
        abstract val HORIZONTAL_PAGE_SIZE_MS: Int
        abstract val VERTICAL_PAGE_SIZE_CHANNEL_COUNT: Int
        abstract val EPG_DAYS_INTO_PAST: Int
        abstract val EPG_DAYS_INTO_FUTURE: Int
        abstract val MILLISECONDS_IN_DAY: Int
        abstract val REQUEST_FOCUS: String
    }
}