package com.twentyfouri.tvlauncher.ui

import android.os.Bundle
import androidx.databinding.DataBindingUtil
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ApiMissingException
import com.twentyfouri.tvlauncher.common.ui.SemaphoreState
import com.twentyfouri.tvlauncher.common.utils.InfoUpdated
import com.twentyfouri.tvlauncher.common.utils.NetworkInfo
import com.twentyfouri.tvlauncher.databinding.DiagnosticsNetworkActivityBinding
import com.twentyfouri.tvlauncher.extensions.beGone
import com.twentyfouri.tvlauncher.extensions.beVisible
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.Timer
import kotlin.concurrent.fixedRateTimer

class DiagnosticsNetworkActivity: BaseActivity() {

    lateinit var binding: DiagnosticsNetworkActivityBinding
    private var networkInfo: NetworkInfo? = null
    private var activeConnectionInfoTimer: Timer? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        try {
            binding = DataBindingUtil.setContentView<DiagnosticsNetworkActivityBinding>(
                    this,
                    R.layout.diagnostics_network_activity
            )
                    .apply {
                        lifecycleOwner = this@DiagnosticsNetworkActivity
                    }
        } catch (t: ApiMissingException) {
            finish()
        }

        binding.sideInfoHeader.text = getString(R.string.di_header_network)
        binding.mainInfoText.text = ""
        binding.ethernetInfoText.text = ""
        binding.wifiInfoText.text = ""

        networkInfo = NetworkInfo(this, object : InfoUpdated {
            override suspend fun onInfoUpdated() {
                withContext(Dispatchers.Main) {
                    binding.mainInfoText.text = networkInfo?.toString(this@DiagnosticsNetworkActivity)
                    binding.ethernetInfoText.text = networkInfo?.ethernet?.toString(this@DiagnosticsNetworkActivity)
                    binding.wifiInfoText.text = networkInfo?.wifi?.toString(this@DiagnosticsNetworkActivity)
                }
            }
        })
    }

    override fun onResume() {
        super.onResume()
        networkInfo?.startAutoUpdate(this)

        if (Flavor().shouldCheckNetworkConnectionQuality) {
            activeConnectionInfoTimer?.cancel()
            activeConnectionInfoTimer = null
            activeConnectionInfoTimer = fixedRateTimer("connection_semaphore", true, 0, 15000) {
                val state = NetworkInfo.getConnectionSemaphoreState()
                CoroutineScope(Dispatchers.Main).launch { binding.semaphore.setLights(state) }
                setSemaphoreWarning(state)
            }
        } else {
            binding.semaphore.setLights(SemaphoreState.OFF)
            setSemaphoreWarning(SemaphoreState.OFF)
        }
    }

    private fun setSemaphoreWarning(state: SemaphoreState) {
        val warning = when (state) {
            SemaphoreState.OFF,
            SemaphoreState.GREEN_LAN,
            SemaphoreState.GREEN_WIFI -> null
            SemaphoreState.ORANGE_WIFI,
            SemaphoreState.ORANGE_LAN -> getString(R.string.quality_warning_yellow)
            SemaphoreState.RED_WIFI,
            SemaphoreState.RED_LAN -> getString(R.string.quality_warning_red)
        }
        CoroutineScope(Dispatchers.Main).launch {
            if (warning.isNullOrEmpty()) {
                binding.semaphoreWarning.beGone()
            } else {
                binding.semaphoreWarning.beVisible()
                binding.semaphoreWarning.text = warning
            }
        }
    }

    override fun onPause() {
        networkInfo?.stopAutoUpdate()
        activeConnectionInfoTimer?.cancel()
        activeConnectionInfoTimer = null
        super.onPause()
    }
}
