package com.twentyfouri.tvlauncher.ui

import android.os.Bundle
import android.util.Log
import android.view.View
import androidx.annotation.CallSuper
import androidx.lifecycle.lifecycleScope
import com.twentyfouri.tvlauncher.common.utils.NetworkConnectionState
import com.twentyfouri.tvlauncher.data.PopulateDatabaseWorkManager
import com.twentyfouri.tvlauncher.receiver.ScreenOnOffReceiver
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import org.koin.androidx.scope.ScopeFragment
import org.koin.core.component.KoinComponent
import kotlin.time.Duration
import kotlin.time.ExperimentalTime
import org.koin.core.component.get
import timber.log.Timber

@OptIn(ExperimentalTime::class)
open class BaseFragment : ScopeFragment(), BackPressListener, KoinComponent {

    protected var lastFocusedView: View? = null
    var lastFocusedViewBeforeSubscriptions: View? = null
    open val showTopbarWhenNavigated = false

    open fun reset() {
        // no action by default
    }

    @CallSuper
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        try {
            super.onViewCreated(view, savedInstanceState)
        } catch (ise: IllegalStateException) {
            // https://github.com/InsertKoinIO/koin/issues/1433
        }
    }

    override fun onResume() {
        super.onResume()
        startListeningToBackPressed()
        NetworkConnectionState.instance.waitForConnection(lifecycleOwner = viewLifecycleOwner, skipWaiting = !ScreenOnOffReceiver.screenWasOff) {
            viewLifecycleOwner.lifecycleScope.launch {
                delay(TEST_DELAY)
                Timber.tag("Flow").d("populateDatabase called from BaseFragment")
                context?.also { PopulateDatabaseWorkManager.enqueue(it) }
            }
        }
    }

    override fun onPause() {
        stopListeningToBackPressed()
        super.onPause()
    }

    open fun saveCurrentFocus() {
    }

    open fun resumeFocus(){
        lastFocusedView?.requestFocus()
    }

    override fun onHiddenChanged(hidden: Boolean) {
        if (!hidden) lastFocusedView?.requestFocus()
        super.onHiddenChanged(hidden)
    }

    override fun onBackPressed(topbarFocused: Boolean): BackPressAction {
        return BackPressAction.NOTHING
    }

    private fun startListeningToBackPressed() = (context as? MainActivity)?.addBackPressedListener(this)

    private fun stopListeningToBackPressed() = (context as? MainActivity)?.removeBackPressedListener(this)

    protected companion object {
        /**
         * Delay some execution, so the UsageManager can settle in RESUMED state to let some GTS tests pass
         *
         * https://jira.24i.com/browse/PRJ1010YOU-2644
         *
         * 0.25 seconds is too low, 1/3 is ok, but let's leave some headroom
         */
        @JvmStatic
        protected val TEST_DELAY = Duration.seconds(0.5)
    }
}