package com.twentyfouri.tvlauncher.receiver

import android.app.Activity
import android.content.BroadcastReceiver
import android.content.Context
import android.content.ContextWrapper
import android.content.Intent
import android.os.Handler
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.user.SmartSessionState
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.data.SetupDataRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.ui.MainActivity
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.joda.time.DateTime
import org.koin.core.component.KoinComponent
import org.koin.core.component.get
import timber.log.Timber
import kotlin.math.abs

class TimeChangeReceiver : BroadcastReceiver(), KoinComponent {

    override fun onReceive(context: Context, intent: Intent) {
        val action = intent.action
        when (action) {
            Intent.ACTION_TIME_CHANGED -> {
                val autoTime = android.provider.Settings.Global.getInt(
                    context.contentResolver,
                    android.provider.Settings.Global.AUTO_TIME,
                    0
                )
                // if new provided time differs more than 3 minutes from last one
                val diff = abs(lastTimeTick - DateTime.now().millis)
                when {
                    diff > 60*3000 -> refreshApiOffset(context)
                    autoTimeSettings != autoTime -> {
                        // let system have some time to retrieve correct time
                        Handler().postDelayed({
                            refreshApiOffset(context)
                        }, 2000)
                    }
                }
                autoTimeSettings = autoTime
            }
            Intent.ACTION_TIMEZONE_CHANGED -> {
                refreshApiOffset(context)
            }
            Intent.ACTION_TIME_TICK -> {
                lastTimeTick = DateTime.now().millis
            }
        }
    }

    private fun refreshApiOffset(context: Context) {
        // refresh API offset
        val setupDataRepository = SetupDataRepository.getInstance(context)
        val api = get<FlowSmartApi>()
        CoroutineScope(Dispatchers.IO).launch {
            val sessionData = setupDataRepository.getSetupDataObject()?.sessionData
            if (sessionData == null || !sessionData.contains(SmartSessionState.LOGGED_IN.name)) return@launch
            try {
                Flavor().performPing(api)
            } catch (e: Exception) {
                Timber.e(e.stackTraceToString())
            }
            TimeProvider.refreshUsingOffset(api.serverTimeOffset)
            val intent = Intent(context, MainActivity::class.java).apply {
                putExtra(MainActivity.EXTRA_FORCE_RESTART, true)
            }
            val isActivityContext = when (context) {
                is Activity -> true
                is ContextWrapper -> context.baseContext is Activity
                else -> false
            }
            if (!isActivityContext) {
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            }
            Timber.tag("Restart").i("TimeChangeReceiver.refreshApiOffset")
            context.startActivity(intent)
        }
    }

    companion object {
        private var lastTimeTick = DateTime.now().millis
        var autoTimeSettings = 0
    }
}
