package com.twentyfouri.tvlauncher.receiver

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.provider.Settings
import android.widget.Toast
import com.twentyfouri.tvlauncher.services.RemoteLoggingService
import org.koin.core.component.KoinComponent
import timber.log.Timber
import java.security.MessageDigest

class RemoteLoggingReceiver : BroadcastReceiver(), KoinComponent {
    /*
     * Can be tested via adb commands:
     *
     * adb shell "am broadcast -a com.twentyfouri.tvlauncher.intent.REMOTE_LOGGING -p com.twentyfouri.entellauncher --es 'auth' '<key>' --es 'command' 'enable' --ei 'time' 60 --ei 'storage' 300"
     * adb shell "am broadcast -a com.twentyfouri.tvlauncher.intent.REMOTE_LOGGING -p com.twentyfouri.entellauncher --es 'auth' '<key>' --es 'command' 'disable'"
     * adb shell "am broadcast -a com.twentyfouri.tvlauncher.intent.REMOTE_LOGGING -p com.twentyfouri.entellauncher --es 'auth' '<key>' --es 'command' 'get_status'"
     * adb shell "am broadcast -a com.twentyfouri.tvlauncher.intent.REMOTE_LOGGING -p com.twentyfouri.entellauncher --es 'auth' '<key>' --es 'command' 'get_detailed_status'"
     */

    override fun onReceive(context: Context, intent: Intent) {
        Timber.d("onReceive")
        /*
         * Since this is effectively a kind of backdoor for debugging purposes, we disable it
         * on devices where developer mode is not enabled. This should limit the level to which
         * it can be exploited in case of a leak.
         */
        if (!context.getIsDeveloperModeEnabled()) {
            Timber.d("Developer mode not enabled")
            return
        }

        val key = intent.getStringExtra("auth") ?: return
        if (!isKeyValid(key)) {
            return
        }

        val bridge = RemoteLoggingService.RemoteLoggingBridge()

        when (intent.getStringExtra("command")) {
            "enable" -> bridge.enableRemoteLogging(
                time = intent.getIntExtra("time", 0),
                storage = intent.getIntExtra("storage", 0)
            )
            "disable" -> bridge.disableRemoteLogging()
            "get_status" ->
                showMessage(context,
                    "getRemoteLoggingStatus: ${bridge.getRemoteLoggingStatus()}",
                    Toast.LENGTH_SHORT)
            "get_detailed_status" ->
                showMessage(context,
                    "getDetailedRemoteLoggingStatus: ${bridge.getDetailedRemoteLoggingStatus()}",
                    Toast.LENGTH_LONG)
        }
    }

    private fun showMessage(context: Context, message: String, toastLength: Int) {
        Timber.d(message)
        Toast.makeText(context, message, toastLength).show()
    }

    private fun Context.getIsDeveloperModeEnabled(): Boolean {
        return Settings.Secure.getInt(contentResolver,
            Settings.Global.DEVELOPMENT_SETTINGS_ENABLED,
            0
        ) != 0
    }

    companion object {
        // The key is "Rq6_x_ibYUsbkX=LMzOZoxoopE6M.Y"
        private const val VALID_KEY_HASH = "3ccd93b871c1e2b61da48c4e457e7aeb89e8a4194aeee30b8d0c0c3a287e683c"
        private const val HASH_SALT = "ozWrzHgc+E6i1Er-Ts7oOaeDs7n6_r"

        private fun isKeyValid(key: String): Boolean {
            return hashCode(key + HASH_SALT) == VALID_KEY_HASH
        }

        private fun hashCode(text: String): String {
            val md = MessageDigest.getInstance("SHA-256")
            val bytes = md.digest(text.toByteArray(Charsets.UTF_8))
            return bytes.joinToString(separator = "") { "%02x".format(it) }
        }
    }
}