package com.twentyfouri.tvlauncher.provider

import android.app.SearchManager
import android.content.ContentProvider
import android.content.ContentValues
import android.content.UriMatcher
import android.database.Cursor
import android.database.MatrixCursor
import android.net.Uri
import android.provider.BaseColumns
import com.twentyfouri.androidcore.utils.Log
import com.twentyfouri.smartmodel.model.dashboard.SmartImages
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.SearchGenre
import com.twentyfouri.tvlauncher.data.DateTimeRepository
import com.twentyfouri.tvlauncher.data.RecommendationsRepository
import com.twentyfouri.tvlauncher.images.SearchImageStorage
import kotlinx.coroutines.runBlocking
import org.koin.core.component.KoinComponent
import org.koin.core.component.get
import timber.log.Timber
import java.io.Serializable
import java.time.LocalDate
import java.time.format.DateTimeFormatter

class SuggestionContentProvider : ContentProvider(), KoinComponent {

    companion object {
        const val CATCH_UP_SEARCH_CODE = 0
        const val LIVE_TV_SEARCH_CODE = 1
        const val RECORDING_SEARCH_CODE = 2
        const val CHANNEL_SEARCH_CODE = 3
        const val KEY_NAME = SearchManager.SUGGEST_COLUMN_TEXT_1
        const val KEY_DESCRIPTION = SearchManager.SUGGEST_COLUMN_TEXT_2
        const val KEY_IMAGE = SearchManager.SUGGEST_COLUMN_RESULT_CARD_IMAGE
        const val KEY_PRODUCTION_YEAR = SearchManager.SUGGEST_COLUMN_PRODUCTION_YEAR
        const val KEY_COLUMN_DURATION = SearchManager.SUGGEST_COLUMN_DURATION
    }

    private lateinit var dateTimeRepository: DateTimeRepository
    private lateinit var uriMatcher: UriMatcher
    private lateinit var searchImageStorage: SearchImageStorage

    override fun query(
        uri: Uri,
        projection: Array<String>?,
        selection: String?,
        selectionArgs: Array<String>?,
        sortOrder: String?
    ): Cursor? {
        context?.let {
            val catchUpSearchPath: String = it.resources.getString(R.string.catch_up_search_path)
            val liveTVSearchPath: String = it.resources.getString(R.string.live_tv_search_path)
            val recordingSearchPath: String = it.resources.getString(R.string.recording_search_path)
            val channelSearchPath: String = it.resources.getString(R.string.channel_search_path)
            val authority: String = it.packageName

            uriMatcher = UriMatcher(UriMatcher.NO_MATCH).apply {
                addURI(authority, catchUpSearchPath, CATCH_UP_SEARCH_CODE)
                addURI(authority, "$catchUpSearchPath/*", CATCH_UP_SEARCH_CODE)
                addURI(authority, liveTVSearchPath, LIVE_TV_SEARCH_CODE)
                addURI(authority, "$liveTVSearchPath/*", LIVE_TV_SEARCH_CODE)
                addURI(authority, recordingSearchPath, RECORDING_SEARCH_CODE)
                addURI(authority, "$recordingSearchPath/*", RECORDING_SEARCH_CODE)
                addURI(authority, channelSearchPath, CHANNEL_SEARCH_CODE)
                addURI(authority, "$channelSearchPath/*", CHANNEL_SEARCH_CODE)
            }
        }

        return if (::uriMatcher.isInitialized) {
            when (uriMatcher.match(uri)) {
                CATCH_UP_SEARCH_CODE -> this.getResultsForSearchType(
                    selectionArgs,
                    uri,
                    SearchGenre.CATCHUP
                )
                LIVE_TV_SEARCH_CODE -> this.getResultsForSearchType(
                    selectionArgs,
                    uri,
                    SearchGenre.LIVE_TV
                )
                RECORDING_SEARCH_CODE -> this.getResultsForSearchType(
                    selectionArgs,
                    uri,
                    SearchGenre.RECORDINGS
                )
                CHANNEL_SEARCH_CODE -> this.getResultsForSearchType(
                    selectionArgs,
                    uri,
                    SearchGenre.CHANNELS
                )
                else -> null
            }
        } else null
    }

    private fun getResultsForSearchType(
        selectionArgs: Array<String>?,
        uri: Uri,
        genre: SearchGenre?
    ): MatrixCursor {
        return if (selectionArgs == null) {
            throw IllegalArgumentException("selectionArgs must be provided for the uri $uri")
        } else {
            val asyncCursor = MatrixCursor(queryProjection)
            if (selectionArgs[0].isNotEmpty()) {
                createRecommendationsRepository()?.apply {
                    genre?.let {
                        searchByGenre(selectionArgs[0], it).map { item ->
                            asyncCursor.addRow(item.toRow())
                        }
                    }
                }
            }
            asyncCursor
        }
    }

    override fun getType(uri: Uri): String? = null

    override fun insert(uri: Uri, values: ContentValues?): Uri? = null

    override fun delete(uri: Uri, selection: String?, selectionArgs: Array<String>?): Int = 0

    override fun update(uri: Uri, values: ContentValues?, selection: String?, selectionArgs: Array<String>?): Int = 0

    override fun onCreate(): Boolean {
        dateTimeRepository = DateTimeRepository(context!!)
        searchImageStorage = SearchImageStorage(context!!.applicationContext)
        return true
    }

    private val queryProjection = arrayOf(
        BaseColumns._ID,
        KEY_NAME,
        KEY_DESCRIPTION,
        KEY_IMAGE,
        KEY_PRODUCTION_YEAR,
        KEY_COLUMN_DURATION,
        SearchManager.SUGGEST_COLUMN_INTENT_DATA_ID
    )

    private fun createRecommendationsRepository(): RecommendationsRepository? {
        val context = this.context?.applicationContext
        if (context == null) {
            Timber.w("Context missing for recommendations repository")
            return null
        }

        return try {
            get<RecommendationsRepository>()
        } catch (e: Exception) {
            Timber.w(e,"Context missing for recommendations repository")
            null
        }
    }

    private fun SmartMediaItem.toRow(): Array<Serializable?> {
        val image = getImage(this)
        val uri = Flavor().getSmartMediaUriParamsFromReference(reference)
        val formatter = DateTimeFormatter.ofPattern("dd-MM-yyyy")
        var releaseYear = LocalDate.now().year
        val description = getDescriptionLine(this)

        if (this.releaseDate.isNotEmpty()) {
            releaseYear = LocalDate.parse(this.releaseDate, formatter).year
        }

        return arrayOf(
                this.reference,
                this.title,
                description,
                image,
                releaseYear,
                this.durationInSeconds * 1000,
                uri.toString()
        )
    }

    private fun getImage(item: SmartMediaItem): String {
        return if (item.type == SmartMediaType.LIVE_CHANNEL) {
            runBlocking {
                searchImageStorage.getOrCreateChannelImage(item).toString()
            }
        } else {
            // 548 x 308 is size in pixels of google katnis row item image
            Flavor().pickBasedOnFlavor(item.images, 548, 308, SmartImages.UNRESTRICTED) ?: ""
        }
    }

    private fun getDescriptionLine(item: SmartMediaItem): String {
        val line = StringBuilder(item.channelName ?: "")
        val startDate = item.startDate
        val endDate = item.endDate

        if (startDate != null && endDate != null) {
            if (line.isNotEmpty()) line.append(" - ")

            val localContext = context
            val date = when {
                dateTimeRepository.isLive(startDate, endDate) && localContext != null -> {
                    localContext.getString(R.string.search_result_time_now)
                }
                dateTimeRepository.isToday(startDate) -> {
                    dateTimeRepository.formatClockTime(startDate.millis)
                }
                else -> dateTimeRepository.formatCatchupDate(item.startDate)
            }
            line.append(date)
        }

        return line.toString()
    }
}