/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */

package com.twentyfouri.tvlauncher.notifications

import android.database.Cursor
import android.os.Bundle
import android.view.View
import androidx.loader.app.LoaderManager
import androidx.loader.content.CursorLoader
import androidx.loader.content.Loader
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.databinding.NotificationsPanelViewBinding
import com.twentyfouri.tvlauncher.ui.BaseActivity
import org.koin.android.ext.android.inject

/**
 * Displays a side panel containing a list of notifications.
 */

class NotificationsSidePanelActivity : BaseActivity(), LoaderManager.LoaderCallbacks<Cursor> {
    private lateinit var mPanelAdapter: NotificationsPanelAdapter
    private val mReadNotifications: ReadNotifications by inject()

    private var isViewing = false
    private var isLoaded = false

    private var nullableBinding: NotificationsPanelViewBinding? = null
    private val binding: NotificationsPanelViewBinding get() = nullableBinding ?: throw IllegalStateException("trying to access uninitialized binding")

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        mPanelAdapter = NotificationsPanelAdapter(
            cursor = null
        )

        setContentView(R.layout.notifications_panel_view)
        nullableBinding = NotificationsPanelViewBinding.bind(findViewById(R.id.notifications_panel_view))

        binding.notificationsList.adapter = mPanelAdapter
        binding.notificationsList.isFocusable = true

        LoaderManager.getInstance(this).initLoader(
            0, null, this@NotificationsSidePanelActivity
        )
    }

    override fun onDestroy() {
        nullableBinding = null
        super.onDestroy()
    }

    override fun onResumeFragments() {
        super.onResumeFragments()
        isViewing = true
        markNotificationsAsReadIfNeeded()
    }

    override fun onPause() {
        super.onPause()
        isViewing = false
    }

    private fun showNoNotificationsMessage() {
        binding.notificationsList.visibility = View.GONE
        binding.noNotificationsMessage.visibility = View.VISIBLE
    }

    private fun showNotifications() {
        binding.noNotificationsMessage.visibility = View.GONE
        binding.notificationsList.visibility = View.VISIBLE
    }

    private fun markNotificationsAsReadIfNeeded() {
        if (isLoaded && isViewing) {
            mReadNotifications.markAsRead(mPanelAdapter.cursor)
        }
    }

    override fun onCreateLoader(id: Int, args: Bundle?): Loader<Cursor> {
        return CursorLoader(
            this, NotificationsContract.CONTENT_URI,
            TvNotification.PROJECTION, null, null, null
        )
    }

    override fun onLoadFinished(loader: Loader<Cursor>, data: Cursor?) {
        isLoaded = true

        mPanelAdapter.changeCursor(data)
        if (data != null && data.count > 0) {
            showNotifications()
        } else {
            showNoNotificationsMessage()
        }

        markNotificationsAsReadIfNeeded()
    }

    override fun onLoaderReset(loader: Loader<Cursor>) {
        mPanelAdapter.changeCursor(null)
        isLoaded = false
    }
}
