/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */

package com.twentyfouri.tvlauncher.notifications

import android.database.Cursor
import android.database.DatabaseUtils
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.RecyclerView
import com.twentyfouri.tvlauncher.R
import timber.log.Timber


/**
 * Adapter for the [RecyclerView] in the notifications side panel which displayed
 * both dismissible and non-dismissible notifications.
 */
class NotificationsPanelAdapter(cursor: Cursor?) :
    RecyclerView.Adapter<NotificationsPanelAdapter.NotificationPanelViewHolder>() {

    var cursor: Cursor? = null
        private set

    override fun getItemCount(): Int {
        return cursor?.count ?: 0
    }

    init {
        this.cursor = cursor
        setHasStableIds(true)
    }

    override fun onCreateViewHolder(
        parent: ViewGroup, viewType: Int
    ): NotificationPanelViewHolder {
        val inflater = LayoutInflater.from(parent.context)

        val trayItem: View
        if (viewType == TYPE_DISMISSIBLE) {
            trayItem = inflater.inflate(
                R.layout.notification_panel_item_dismissible,
                parent, false
            )
        } else {
            trayItem = inflater.inflate(
                R.layout.notification_panel_item,
                parent, false
            )
        }

        return NotificationPanelViewHolder(trayItem)
    }

    override fun onBindViewHolder(
        holder: NotificationPanelViewHolder,
        position: Int
    ) {
        if (cursor?.moveToPosition(position) == false) {
            throw IllegalStateException("Can't move cursor to position $position")
        }
        onBindViewHolder(holder, cursor!!)
    }

    override fun getItemViewType(position: Int): Int {
        if (cursor?.moveToPosition(position) == false) {
            throw IllegalStateException("Can't move cursor to position $position")
        }

        val dismissible = cursor?.getInt(TvNotification.COLUMN_INDEX_DISMISSIBLE) != 0
        val ongoing = cursor?.getInt(TvNotification.COLUMN_INDEX_ONGOING) != 0
        return if (ongoing || !dismissible) {
            TYPE_PERSISTENT
        } else {
            TYPE_DISMISSIBLE
        }
    }

    override fun getItemId(position: Int): Long {
        if (cursor?.moveToPosition(position) == false) {
            Timber.wtf("Can't move cursor to position $position")
            return View.NO_ID.toLong()
        }

        val key = cursor!!.getString(TvNotification.COLUMN_INDEX_KEY)
        return key.hashCode().toLong()
    }

    private fun onBindViewHolder(viewHolder: NotificationPanelViewHolder, cursor: Cursor) {
        viewHolder.setNotification(TvNotification.fromCursor(cursor))
    }

    class NotificationPanelViewHolder(itemView: View) : RecyclerView.ViewHolder(itemView) {

        fun setNotification(notification: TvNotification) {
            (itemView as NotificationPanelItemView).setNotification(notification)
        }
    }

    /**
     * Swap in a new Cursor, and close the old Cursor.
     *
     * @param newCursor The new cursor to be used.
     */
    fun changeCursor(newCursor: Cursor?) {
        if (DEBUG) {
            Timber.d("changeCursor() called with: " + "newCursor = [" + DatabaseUtils.dumpCursorToString(
                newCursor) + "]")
        }

        cursor = newCursor
        notifyDataSetChanged()
    }

    companion object {
        private const val TAG = "NotifsPanelAdapter"
        private const val DEBUG = false

        private const val TYPE_DISMISSIBLE = 0
        private const val TYPE_PERSISTENT = 1
    }
}