/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */

package com.twentyfouri.tvlauncher.notifications

import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.animation.AnimatorSet
import android.animation.ObjectAnimator
import android.content.Context
import android.util.AttributeSet
import android.view.View
import android.view.View.OnFocusChangeListener
import android.widget.TextView
import com.twentyfouri.tvlauncher.R


/**
 * View for a dismissible notification displayed in the notifications side panel.
 * Handles swapping of background in RTL layouts.
 * Handles dismiss button focus animation.
 */
class NotificationPanelDismissibleItemView : NotificationPanelItemView {
    private lateinit var mDismissButton: View
    private lateinit var mDismissText: TextView
    private var mViewFocusTranslationX: Int = 0
    private var mDismissTranslationX: Int = 0
    private var mIsRtl: Boolean = false

    constructor(context: Context) : super(context) {
        initializeTranslationValues()
    }

    constructor(context: Context, attrs: AttributeSet) : super(context, attrs) {
        initializeTranslationValues()
    }

    private fun initializeTranslationValues() {
        mIsRtl = resources.configuration.layoutDirection == View.LAYOUT_DIRECTION_RTL

        mViewFocusTranslationX = resources.getDimensionPixelSize(
            R.dimen.notification_panel_item_show_button_translate_x
        )
        mDismissTranslationX = resources.getDimensionPixelSize(
            R.dimen.notification_panel_item_dismiss_translate_x
        )

        if (mIsRtl) {
            mViewFocusTranslationX *= -1
            mDismissTranslationX *= -1
        }
    }

    override fun onFinishInflate() {
        super.onFinishInflate()

        // This is to set the focus search to the default behavior, so that on dismissible views
        // the user can navigate left/right to the dismiss button.
        mMainContentText.nextFocusLeftId = View.NO_ID
        mMainContentText.nextFocusRightId = View.NO_ID

        mDismissButton = findViewById(R.id.dismiss_button)
        mDismissText = findViewById(R.id.dismiss_text)

        if (mIsRtl) {
            mMainContentText.setBackgroundResource(R.drawable.notification_background_left)
            mDismissButton.setBackgroundResource(R.drawable.notification_background_right)
        } else {
            mMainContentText.setBackgroundResource(R.drawable.notification_background_right)
            mDismissButton.setBackgroundResource(R.drawable.notification_background_left)
        }

        val dismissAnimator = AnimatorSet()
        val containerSlide = ObjectAnimator.ofFloat(
            mMainContentText, View.TRANSLATION_X,
            mViewFocusTranslationX.toFloat(), mDismissTranslationX.toFloat()
        )
        val dismissButtonSlide = ObjectAnimator.ofFloat(
            mDismissButton,
            View.TRANSLATION_X, mViewFocusTranslationX.toFloat(), mDismissTranslationX.toFloat()
        )

        dismissAnimator.playTogether(dismissButtonSlide, containerSlide)

        dismissAnimator.addListener(object : AnimatorListenerAdapter() {
            override fun onAnimationStart(animation: Animator) {
                collapseText()
                mDismissText.visibility = View.GONE
            }

            override fun onAnimationEnd(animation: Animator) {
                mDismissButton.visibility = View.INVISIBLE
                mMainContentText.visibility = View.INVISIBLE
                setBackgroundColor(context.getColor(R.color.notification_selection_color))
                NotificationsUtils.dismissNotification(context, mNotificationKey!!)
            }
        })

        val gainFocus = AnimatorSet()
        val containerSlideOut = ObjectAnimator.ofFloat(
            mMainContentText,
            View.TRANSLATION_X.name, 0f, mViewFocusTranslationX.toFloat()
        )
        val dismissButtonFocusGain = ObjectAnimator.ofFloat(
            mDismissButton,
            View.TRANSLATION_X, 0f, mViewFocusTranslationX.toFloat()
        )
        gainFocus.playTogether(dismissButtonFocusGain, containerSlideOut)

        val loseFocus = AnimatorSet()
        val containerSlideIn = ObjectAnimator.ofFloat(
            mMainContentText,
            View.TRANSLATION_X, mViewFocusTranslationX.toFloat(), 0f
        )
        val dismissButtonFocusLost = ObjectAnimator.ofFloat(
            mDismissButton,
            View.TRANSLATION_X, mViewFocusTranslationX.toFloat(), 0f
        )
        loseFocus.playTogether(dismissButtonFocusLost, containerSlideIn)
        loseFocus.addListener(object : AnimatorListenerAdapter() {
            override fun onAnimationEnd(animation: Animator) {
                mDismissText.visibility = View.GONE
            }
        })

        mDismissButton.setOnClickListener {
            if (mNotificationKey != null) {
                dismissAnimator.start()
            }
        }
        mDismissButton.onFocusChangeListener = OnFocusChangeListener { view, focused ->
            if (focused) {
                // Slide the views to the side and show the dismiss text
                mDismissText.visibility = View.VISIBLE
                gainFocus.start()
            } else {
                // Slide the views back to their original positions and hide the dismiss text
                loseFocus.start()
            }
        }
    }

    override fun setNotification(notif: TvNotification) {
        super.setNotification(notif)
        mDismissText.text = notif.dismissButtonLabel
        mDismissButton.visibility = View.VISIBLE
    }
}

