package com.twentyfouri.tvlauncher.images

import android.content.Context
import android.graphics.*
import com.bumptech.glide.Glide
import com.bumptech.glide.signature.ObjectKey
import com.twentyfouri.androidcore.utils.Log
import com.twentyfouri.smartmodel.model.dashboard.SmartImages
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.ImageType
import com.twentyfouri.tvlauncher.R
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.util.concurrent.TimeUnit
import kotlin.math.min

class SearchImageRenderer(
    private val context: Context
) {
    private val resources = context.resources

    suspend fun generateSearchChannelImage(channel: SmartMediaItem): Bitmap? {
        val icon = loadImage(channel) ?: return null

        val bitmap = Bitmap.createBitmap(
            resources.getDimensionPixelSize(R.dimen.search_image_channel_width),
            resources.getDimensionPixelSize(R.dimen.search_image_channel_height),
            Bitmap.Config.ARGB_8888)

        val canvas = Canvas(bitmap)

        canvas.drawColor(context.getColor(R.color.search_channel_image_background_color))

        val iconMargin = resources.getDimensionPixelSize(R.dimen.search_image_channel_icon_margin)
        drawScaledAndCentered(canvas, icon, iconMargin)

        return bitmap
    }

    private suspend fun loadImage(channel: SmartMediaItem): Bitmap? {
        require(channel.type == SmartMediaType.LIVE_CHANNEL) { "Media item is not of type LIVE_CHANNEL" }

        val url = Flavor().pickBasedOnFlavor(Flavor().getImageOfType(channel, ImageType.LIGHT),1, 1, SmartImages.UNRESTRICTED)

        if (url.isNullOrEmpty()) return null

        return try {
            withContext(Dispatchers.IO) {
                Glide.with(context)
                        .asBitmap()
                        .signature(ObjectKey(Flavor().glideCacheExpirationInterval))
                        .load(url)
                        .submit()
                        .get(30, TimeUnit.SECONDS)
            }
        } catch (e: Exception) {
            Log.d(javaClass.simpleName, "Failed to fetch $url", e)
            null
        }
    }

    private fun drawScaledAndCentered(canvas: Canvas, icon: Bitmap, edgeMargins: Int) {
        val drawableWidth = canvas.width - ( edgeMargins * 2 )
        val drawableHeight = canvas.height - ( edgeMargins * 2 )

        val scale = min(
            drawableWidth / icon.width.toFloat(),
            drawableHeight / icon.height.toFloat()
        )

        val scaledWidth = icon.width * scale
        val scaledHeight = icon.height * scale

        val drawBounds = RectF(
            (canvas.width - scaledWidth) / 2,
            (canvas.height - scaledHeight) / 2,
            (canvas.width + scaledWidth) / 2,
            (canvas.height + scaledHeight) / 2
        )

        canvas.drawBitmap(icon, null, drawBounds, null)
    }

}