package com.twentyfouri.tvlauncher.homepagechannels

import android.app.job.JobParameters
import android.app.job.JobService
import android.content.Context
import android.util.Log
import androidx.tvprovider.media.tv.TvContractCompat
import com.twentyfouri.tvlauncher.homepagechannels.room.HomepageChannel
import kotlinx.coroutines.*
import org.koin.core.component.KoinComponent
import org.koin.core.component.get
import timber.log.Timber

/**
 * A service that will populate the TV provider with channels that every user should have. Once a
 * channel is created, it trigger another service to add programs.
 */
class SyncChannelJobService : JobService(), KoinComponent {

    private var syncJob: Job? = null

    override fun onStartJob(jobParameters: JobParameters): Boolean {
        Timber.d("Starting channel creation job")
        syncJob = CoroutineScope(Dispatchers.Main).launch {
            syncHomepageChannels(applicationContext)
            //Daisy chain listening for the next change to the channel.
            jobFinished(jobParameters, false)
        }
        return true
    }

    override fun onStopJob(jobParameters: JobParameters): Boolean {
        try {
            syncJob?.cancel()
        } catch ( e: CancellationException ) { }
        return true
    }

    private suspend fun syncHomepageChannels(mContext: Context) = withContext(Dispatchers.Default) {
        var homepageChannels: List<HomepageChannel> = HomepageChannelDatabaseRepository.getHomepageChannels(mContext)
        val numOfChannelsInTVProvider: Int = TvUtil.getNumberOfChannels(mContext)
        // Checks if the default channels are added. Since a user can add more channels from
        // your app later, the number of channels in the provider can be greater than the number
        // of default channels.
        if (numOfChannelsInTVProvider >= homepageChannels.size && homepageChannels.isNotEmpty()) {
            Timber.d("Already loaded default channels into the provider")
        } else { // Create subscriptions from mocked source.
            homepageChannels = get<HomepageChannelNetworkRepository>().createHomepageChannels(mContext)
            for (subscription in homepageChannels) {
                val channelId = TvUtil.createChannel(mContext, subscription)
                Timber.d("Added HomepageChannel with id: $channelId")
                subscription.channelId = channelId
                TvContractCompat.requestChannelBrowsable(mContext, channelId)
            }
            HomepageChannelDatabaseRepository.saveHomepageChannels(mContext, homepageChannels)
        }
        // Kick off a job to update default programs.
        // The program job should verify if the channel is visible before updating programs.
        for (channel in homepageChannels) {
            TvUtil.scheduleSyncingProgramsForChannel(mContext, channel.channelId)
        }
    }

    companion object {
        private const val TAG = "SyncChannelJobService"
    }
}