package com.twentyfouri.tvlauncher.homepagechannels

import android.content.Context
import android.content.SharedPreferences
import com.twentyfouri.tvlauncher.homepagechannels.room.HomepageChannel
import com.twentyfouri.tvlauncher.homepagechannels.room.HomepageChannelDatabase
import com.twentyfouri.tvlauncher.homepagechannels.room.HomepageChannelItem
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

object HomepageChannelDatabaseRepository {

    suspend fun saveHomepageChannels(context: Context, homepageChannels: List<HomepageChannel>) = withContext(Dispatchers.IO) {
         HomepageChannelDatabase
             .getInstance(context)
             .homepageChannelDAO
             .insertChannels(*homepageChannels.toTypedArray())
    }

    /**
     * Returns homepageChannels
     *
     * @param context
     * @return a list of homepageChannels or empty list if none exist.
     */
    suspend fun getHomepageChannels(context: Context): List<HomepageChannel> = withContext(Dispatchers.IO) {
        HomepageChannelDatabase
            .getInstance(context)
            .homepageChannelDAO
            .getAllChannels()
    }

    /**
     * Finds a homepageChannel given a channel id that the homepageChannel is associated with.
     *
     * @param context
     * @param channelId of the channel that the homepageChannel is associated with.
     * @return a homepageChannel or null if none exist.
     */
    suspend fun findHomepageChannelByChannelId(context: Context, channelId: Long): HomepageChannel? = withContext(Dispatchers.IO) {
        HomepageChannelDatabase
            .getInstance(context)
            .homepageChannelDAO
            .find(channelId)
    }

    /**
     * Finds a homepageChannel with the given name.
     *
     * @param context
     * @param name of the homepageChannel.
     * @return a homepageChannel or null if none exist.
     */
    suspend fun findHomepageChannelByName(context: Context, name: String?): HomepageChannel? = withContext(Dispatchers.IO){
        name ?: return@withContext null
        HomepageChannelDatabase
            .getInstance(context)
            .homepageChannelDAO
            .find(name)
    }

    /**
     * Overrides the movies stored for a given homepageChannel.
     *
     * @param context
     * @param channelId of the channel that the movies are associated with.
     * @param items to be stored.
     */
    suspend fun saveChannelItems(context: Context, channelId: Long, items: List<HomepageChannelItem>) = withContext(Dispatchers.IO) {
        HomepageChannelDatabase
            .getInstance(context)
            .homepageChannelDAO
            .insertChannelItems(*(items.onEach { it.channelId = channelId }.toTypedArray()))
    }

    /**
     * Removes the list of movies associated with a channel. Overrides the current list with an
     * empty list
     *
     * @param context
     * @param channelId of the channel that the movies are associated with.
     */
    suspend fun removeChannelItems(context: Context, channelId: Long) = withContext(Dispatchers.IO) {
        HomepageChannelDatabase
            .getInstance(context)
            .homepageChannelDAO
            .deleteChannelItems(channelId)
    }

    /**
     * Finds movie in homepageChannels with channel id and updates it. Otherwise will add the new movie
     * to the homepageChannel.
     *
     * @param context
     * @param channelId of the homepageChannel that the movie is associated with.
     * @param channelItem to be persisted or updated.
     */
    suspend fun saveChannelItem(context: Context, channelId: Long, channelItem: HomepageChannelItem) = withContext(Dispatchers.IO) {
        HomepageChannelDatabase
            .getInstance(context)
            .homepageChannelDAO
            .insertChannelItems(channelItem.also { it.channelId = channelId })
    }

    /**
     * Returns movies stored for a given homepageChannel.
     *
     * @param context
     * @param channelId of the homepageChannel that the movie is associated with.
     * @return a list of items for a homepageChannel
     */
    suspend fun getChannelItems(context: Context, channelId: Long): List<HomepageChannelItem> = withContext(Dispatchers.IO) {
        HomepageChannelDatabase
            .getInstance(context)
            .homepageChannelDAO
            .getChannelItems(channelId)
    }

    /**
     * Finds a movie in a homepageChannel by its id.
     *
     * @param context
     * @param channelId of the homepageChannel that the movie is associated with.
     * @param channelItemId of the movie.
     * @return a channel item or null if none exist.
     */
    suspend fun findChannelItemById(context: Context, channelId: Long, channelItemId: Long): HomepageChannelItem? = withContext(Dispatchers.IO) {
        HomepageChannelDatabase
            .getInstance(context)
            .homepageChannelDAO
            .getChannelItem(channelId, channelItemId)
    }
}