package com.twentyfouri.tvlauncher.data

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Observer
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.dashboard.SmartPlaylistOptions
import com.twentyfouri.smartmodel.model.error.RecordingStorageException
import com.twentyfouri.smartmodel.model.media.SmartMediaDetail
import com.twentyfouri.smartmodel.model.menu.SmartNavigationAction
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.smartmodel.model.recording.SmartRecordingStatus
import com.twentyfouri.smartmodel.model.recording.SmartRecordingStorage
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PlaylistType
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.data.apihandler.ApiHandler
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.utils.Navigator
import com.twentyfouri.tvlauncher.utils.RecordingsDialogsHelper
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.last
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.withContext
import org.joda.time.DateTime

class RecordingsRepository(
    private val smartApi: FlowSmartApi,
    private val apiHandler: ApiHandler,
    private val recordingsSettingsRepository: RecordingSettingsRepository,
    private val epgRepository: EpgRepository
) {

    private val recordingsEnabled = MutableLiveData<Boolean>()
    private val recordingsRefreshInProgress = MutableLiveData<Boolean>(false)
    private val recordingsLD = MutableLiveData<List<SmartMediaItem>>()
    private val recordingsJustIdsLD = MutableLiveData<List<String>>()

    private suspend fun startRecording(mediaDetail: SmartMediaDetail, autoRecording: Boolean): SmartMediaItem {
        val smartRecording = Flavor().convertDetailToRecording(mediaDetail, autoRecording)
        smartRecording.recordingData?.apply {
            beginOffsetInSeconds = recordingsSettingsRepository.getStartBefore() * SECONDS_IN_MINUTE
            endOffsetInSeconds = recordingsSettingsRepository.getStartAfter() * SECONDS_IN_MINUTE
        }
        return smartApi.createRecording(smartRecording).last()
    }

    private suspend fun stopRecording(mediaDetail: SmartMediaDetail, autoRecording: Boolean, removeRecorded: Boolean): SmartMediaItem {
        val smartRecording = Flavor().convertDetailToRecording(mediaDetail, autoRecording)
        if (autoRecording && removeRecorded) { // removing only past, recorded items (batchDeleteRecordings())
            val listOfRecordings = getRecordingsInternal().filter {
                it.seriesReference == mediaDetail.seriesReference && it.type == SmartMediaType.RECORDING && it.isInPast
            }
//            Log.d("recordings", "stopRecording delete recorded episodes - recording ids: ${listOfRecordings
//                    .joinToString(separator = " ") { (it.reference as? SmartVideoMediaReference)?.id.toString() }}")
            smartApi.deleteRecordings(listOfRecordings).collect()
        } else if (autoRecording) { // removing only future, not yet recorded items (deleteAutoRecording())
            smartApi.deleteRecordings(listOf(
                getRecordingsInternal().first {
                    it.seriesReference == mediaDetail.seriesReference && it.type == SmartMediaType.AUTO_RECORDING
                }
            )).collect()
            getRecordingsInternal().firstOrNull {
                it.seriesReference == mediaDetail.seriesReference && it.type == SmartMediaType.RECORDING && it.isNow
            }?.let {
                smartApi.deleteRecordings(listOf(it)).collect()
            }
        } else { // removing single recording (deleteSingleRecording())
            smartApi.deleteRecordings(listOf(smartRecording)).collect()
        }
        return smartRecording.apply {
            recordingData?.apply {
                recordingStatus = SmartRecordingStatus.OTHER
            }
        }
    }

    fun startRecordingLD(mediaDetail: SmartMediaDetail?, autoRecording: Boolean, catchBlock : suspend CoroutineScope.(e: Exception) -> Boolean): LiveData<SmartMediaItem> {
        val recordingLD = MutableLiveData<SmartMediaItem>()
        getRecordingsStorageLD().observeOnce { storage ->
            apiHandler.launchNew(
                block = {
                    if (autoRecording) {
                        val remainingPercentageInStorage = 1 - (storage.usedSize.toFloat() / storage.totalSize.toFloat())
                        when {
                            remainingPercentageInStorage < 0.1 -> catchBlock(RecordingStorageException(c = STORAGE_ALMOST_FULL_10))
                            remainingPercentageInStorage < 0.25 -> {
                                mediaDetail?.also { recordingLD.postValue(startRecording(it, autoRecording)) }
                                catchBlock(RecordingStorageException(c = STORAGE_ALMOST_FULL_25))
                            }
                            else -> mediaDetail?.also { recordingLD.postValue(startRecording(it, autoRecording)) }
                        }
                    }
                    else
                        mediaDetail?.also { recordingLD.postValue(startRecording(it, autoRecording)) }
                },
                catchBlock = catchBlock
            )
        }
        return recordingLD
    }

    private fun <T> LiveData<T>.observeOnce(observer: (T) -> Unit) {
        observeForever(object: Observer<T> {
            override fun onChanged(value: T) {
                observer(value)
                removeObserver(this)
            }
        })
    }

    fun getStorageFullCatchBlock(resourceRepository: ResourceRepository) : suspend CoroutineScope.(e: Exception) -> Boolean = {
        withContext(Dispatchers.Main) {
            if (it is RecordingStorageException) {
                when (it.code) {
                    STORAGE_ALMOST_FULL_25 ->
                        RecordingsDialogsHelper.showStorageAlmostFullDialog25(resourceRepository) {
                            Navigator.getInstance()
                                .navigate(SmartNavigationTarget.to(SmartNavigationAction.USER_RECORDINGS))
                        }
                    STORAGE_ALMOST_FULL_10 ->
                        RecordingsDialogsHelper.showStorageAlmostFullDialog10(resourceRepository) {
                            Navigator.getInstance()
                                .navigate(SmartNavigationTarget.to(SmartNavigationAction.USER_RECORDINGS))
                        }
                    else -> RecordingsDialogsHelper.showStorageFullDialog(resourceRepository) {
                        Navigator.getInstance()
                            .navigate(SmartNavigationTarget.to(SmartNavigationAction.USER_RECORDINGS))
                    }
                }
                true
            } else {
                false
            }
        }
    }

    fun stopRecordingLD(mediaDetail: SmartMediaDetail?, autoRecording: Boolean, removeRecorded: Boolean): LiveData<SmartMediaItem> {
//        Log.d("recordings", "stopRecordingLd autoRecording: $autoRecording, removeRecorded: $removeRecorded, mediaDetail: $mediaDetail")
        val recordingLD = MutableLiveData<SmartMediaItem>()
        apiHandler.launchNew {
            mediaDetail?.also { recordingLD.postValue(stopRecording(it, autoRecording, removeRecorded)) }
        }
        return recordingLD
    }

    private fun List<SmartMediaItem>.enhanceChannelInfo(channels: List<SmartMediaItem>): List<SmartMediaItem> {
        forEach { recording ->
            channels
                .find { channel -> channel.reference == recording.channelReference }
                ?.also { channelOfTheRecording -> recording.channelReference = channelOfTheRecording.reference }
        }
        return this
    }

    private suspend fun getRecordings(): List<SmartMediaItem> {
        return getRecordingsInternal().filter {
            it.recordingData?.recordingStatus == SmartRecordingStatus.SUCCESS ||
                    it.recordingData?.recordingStatus == SmartRecordingStatus.WAIT_RECORD ||
                    it.recordingData?.recordingStatus == SmartRecordingStatus.RECORDING ||
                    it.type == SmartMediaType.AUTO_RECORDING
        }.toMutableList()
    }

    private suspend fun getRecordingsInternal(): List<SmartMediaItem> {
        val playlistReference = Flavor().getPlaylistReference(PlaylistType.RECORDINGS)
        val optionsSelected = smartApi.getPlaylistOptions(playlistReference).last().createSelection()
        return smartApi.getRecordings(optionsSelected).last().items
    }

    private suspend fun isSubscribedToRecordings(): Boolean {
        // TODO: add subscriptions - missing api services and documentation
        return true
//        val recordingsPackageReference = Flavor().getRecordingsPackageReference() ?: return false
//        return smartApi.getSubscriptions().any {
//            it.reference == recordingsPackageReference
//                    && it.isSubscribed
//                    && it.startDate != null
//                    && it.startDate!!.millis < TimeProvider.nowMs()
//                    && it.endDate != null
//                    && it.endDate!!.millis > TimeProvider.nowMs()
//        }
    }

    private fun updateRecordingsAndRecordingsEnabled() {
        apiHandler.joinPreviousOrLaunchNew(
            synchronizedJobName = "updateRecordingsAndRecordingsEnabled",
            block = {
                recordingsRefreshInProgress.postValue(true)
                this.coroutineContext[Job]?.invokeOnCompletion {
                    recordingsRefreshInProgress.postValue(false)
                }
                val isSubscribedToRecordings = isSubscribedToRecordings()
                if (isSubscribedToRecordings) {
                    //TODO use recordings as a Flow not result
                    val recordings = getRecordings()
                    epgRepository.getAllChannelsFlow()
                            .map { recordings.toList().enhanceChannelInfo(it) } //toList() makes a clone
                            //.collect { recordingsLD.postValue(it) }
                            .last().also { recordingsLD.postValue(it) }
                } else {
                    recordingsLD.postValue(emptyList())
                }
                //if the getRecordings() has crashed, next line is not called
                recordingsEnabled.postValue(isSubscribedToRecordings)
            },
            catchBlock = {
                recordingsLD.postValue(emptyList())
                recordingsEnabled.postValue(false)
                true
            }
        )
    }

    private val SmartMediaItem.isInPast: Boolean
        get() = (endDate ?: DateTime(0)) < TimeProvider.now()

    private val SmartMediaItem.isNow: Boolean
        get() {
            val now = TimeProvider.now()
            return (startDate ?: DateTime(Long.MAX_VALUE)) <= now && (endDate ?: DateTime(0)) >= now
        }

    fun getRefreshStatus(): LiveData<Boolean> = recordingsRefreshInProgress

    fun getRecordingsLD(): LiveData<List<SmartMediaItem>?> {
        when (recordingsEnabled.value) {
            null -> updateRecordingsAndRecordingsEnabled() //TODO just temporarily disabled
            true -> {
                apiHandler.joinPreviousOrLaunchNew(
                    synchronizedJobName = "getRecordingsLD",
                    block = {
                        recordingsRefreshInProgress.postValue(true)
                        this.coroutineContext[Job]?.invokeOnCompletion {
                            recordingsRefreshInProgress.postValue(false)
                        }
                        //TODO use recordings as a Flow not result
                        val recordings = getRecordings()
                        epgRepository.getAllChannelsFlow()
                                .map { recordings.toList().enhanceChannelInfo(it) } //toList() makes a clone
//                                .collect { recordingsLD.postValue(it) }
                                .last().also { recordingsLD.postValue(it) }
                    }
                )
            }
            else -> { /*recordingsEnabled.postValue(false)*/ } //TODO just temporarily disabled
        }
        return recordingsLD
    }

    fun getRecordingsJustIdsLD(): LiveData<List<String>?> {
        if (recordingsEnabled.value == true) {
            apiHandler.joinPreviousOrLaunchNew(
                    synchronizedJobName = "getRecordingsJustIdsLD",
                    block = {
                        smartApi.getRecordings(
                                options = SmartPlaylistOptions().createSelection()
                                    //TODO if the method getRecordingsJustIdsLD is ever needed uncomment and put it in Flavor
                                    //.apply { extraFields.add(EXTRA_KEY_FOR_JUST_RECORDED_EVENTS_IDS) }
                        ).last().items.map { Flavor().getChannelId(it) }.also { recordingsJustIdsLD.postValue(it) }
                    }
            )
        }
        return recordingsJustIdsLD
    }

    fun refreshRecordings() {
        getRecordingsLD()
    }

    fun getRecordingsEnabledLD(): LiveData<Boolean> {
        updateRecordingsAndRecordingsEnabled()
        return recordingsEnabled
    }

    fun getRecordingsStorageLD(): LiveData<SmartRecordingStorage> {
        val recordingsLD = MutableLiveData<SmartRecordingStorage>()
        apiHandler.launchNew { smartApi.getRecordingsStorage().first().also { recordingsLD.postValue(it) } }
        return recordingsLD
    }

    fun isAutoRecordingSingleEpisode(seriesReference: SmartMediaReference?) : Boolean {
        seriesReference?.let {
            return getRecordingsLD().value?.filter { it.seriesReference == seriesReference && it.isInPast }?.size == 1
        } ?: return true
    }

    private companion object {
        private const val SECONDS_IN_MINUTE = 60
        private const val STORAGE_ALMOST_FULL_10 = "10"
        private const val STORAGE_ALMOST_FULL_25 = "25"
        private const val TAG = "RecordingsRepository"
    }
}