package com.twentyfouri.tvlauncher.data

import android.content.Context
import android.os.Bundle
import android.util.Log
import androidx.work.*
import com.google.firebase.analytics.FirebaseAnalytics
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.persistence.SSTResult
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import kotlinx.coroutines.flow.*
import org.koin.core.component.KoinComponent
import org.koin.core.component.get
import timber.log.Timber

//BEWARE if you want to use another constructor for the Worker (not just the inherited one), you have to implement
//the WorkerFactory and assign it via custom Configuration.Provider
@Suppress("unused")
class PopulateDatabaseWorker(
    private val appContext: Context,
    workerParams: WorkerParameters
): CoroutineWorker(appContext, workerParams), KoinComponent {

    companion object {
        const val Progress = "Progress"
    }

    override suspend fun doWork(): Result {
        try {
            FirebaseAnalytics.getInstance(appContext).logEvent(
                "populate_db_start",
                Bundle().apply {
                    putString("populate_db_start", currentTimeFormatted())
                }
            )
            Timber.tag("Flow").d("populateDatabase doWork")
            Flavor().populateDatabase(get<FlowSmartApi>())
                ?.onStart {
                    Timber.tag("Flow").d("populateDatabase onStart")
                    setProgress(workDataOf(Progress to 0))
                }
                ?.onCompletion {
                    Timber.tag("Flow").d("populateDatabase onCompletion")
                    setProgress(workDataOf(Progress to 100))
                    PopulateDatabaseWorkManager.enqueueAgainAfterSuccess(appContext)
                }
                ?.onEach {
                    when (it.status) {
                        SSTResult.Status.LOADING -> setProgress(workDataOf(Progress to it.data))
                        else -> {}
                    }
                }
                ?.collect()
            FirebaseAnalytics.getInstance(appContext).logEvent(
                "populate_db_success",
                Bundle().apply {
                    putString("populate_db_end", currentTimeFormatted())
                }
            )
            Timber.tag("Flow").d("populateDatabase success")
            return Result.success()
        } catch (e: Exception) {
            FirebaseAnalytics.getInstance(appContext).logEvent(
                "populate_db_failed",
                Bundle().apply {
                    putString("populate_db_end", currentTimeFormatted())
                    putString("exception", e.message)
                }
            )
            Timber.tag("Flow")
                .d("populateDatabase exception -> retry tomorrow. error: ${e.message} ")
            //we need to return success. Failure would stop the WorkManager and Retry would try to retry it too soon
            //we just want to try it again tomorrow which is done by the onCompletion above which happens always even when
            //there is an exception coming from populateDatabase
            return Result.success()
        }
    }

    private fun currentTimeFormatted() =
        "${TimeProvider.now().hourOfDay().asString}:${TimeProvider.now().minuteOfHour().asString}"

}