package com.twentyfouri.tvlauncher.data

import android.content.Context
import android.util.Log
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.work.ExistingWorkPolicy
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.WorkInfo
import androidx.work.WorkManager
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import org.joda.time.DateTime
import timber.log.Timber
import java.util.concurrent.TimeUnit
import kotlin.random.Random

object PopulateDatabaseWorkManager {

    // Defines interval when update should be run ... 1 and 3 means interval from 1 o'clock to 4 o'clock
    private const val enqueuePopulateDatabaseWindowStart = 1
    private const val enqueuePopulateDatabaseWindowLength = 3

    private const val UNIQUE_WORK_NAME = "PopulateDatabaseWork"
    private const val WORK_REQUEST_TAG = "PopulateDatabaseWorkRequest"

    private var lastTimeOfPopulateDatabaseSuccess = 0L

    fun enqueue(context: Context) {
        Timber.tag("Flow").d("populateDatabase enqueue - do not override")
        val now = TimeProvider.now()
        enqueue(context, now, ExistingWorkPolicy.KEEP)
    }

    fun enqueueAgainAfterSuccess(context: Context) {
        Timber.tag("Flow").d("populateDatabase enqueue - override")
        val now = TimeProvider.now()
        lastTimeOfPopulateDatabaseSuccess = now.millis
        enqueue(context, now, ExistingWorkPolicy.REPLACE)
    }

    private fun enqueue(context: Context, now: DateTime, policy: ExistingWorkPolicy) {
        val workRequest = getWorkRequest(now)
        val workManager = WorkManager.getInstance(context)
        workManager.enqueueUniqueWork(UNIQUE_WORK_NAME, policy, workRequest)
    }

    public fun getWorkInfosLD(context: Context) = WorkManager.getInstance(context).getWorkInfosForUniqueWorkLiveData(UNIQUE_WORK_NAME)

    private fun getWorkRequest(now: DateTime) = OneTimeWorkRequestBuilder<PopulateDatabaseWorker>()
        .setInitialDelay(getRandomStartTimeDiff(now), TimeUnit.MILLISECONDS)
        .addTag(WORK_REQUEST_TAG)
        .build()

    private fun getRandomStartTimeDiff(now: DateTime): Long {
        //randomly pick some minute in the interval length
        val rand = Random.nextInt(0, enqueuePopulateDatabaseWindowLength * 60)
        //count target time based on interval start and random minute
        var target = now.withHourOfDay(enqueuePopulateDatabaseWindowStart).withMinuteOfHour(0)
            .plusMinutes(rand)
        //in case target time already passed or the populate already happened today move it to next day
        if (target < now || DateTime(lastTimeOfPopulateDatabaseSuccess).dayOfMonth == now.dayOfMonth)
            target = target.plusDays(1)
        //debug
//        val target = now.plusMinutes(1)
        val millis = target.millis - now.millis
        Timber.tag("Flow").d("populateDatabase target time: $target")
        return millis
    }
}