package com.twentyfouri.tvlauncher.data

import android.util.Log
import com.twentyfouri.androidcore.epg.model.EpgChannel
import com.twentyfouri.smartmodel.model.dashboard.SmartImages
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.ImageType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.Flavor
import org.joda.time.DateTime
import java.util.concurrent.ConcurrentLinkedQueue
import kotlin.math.max
import kotlin.math.min

class EpgChannelExt(
    val smartMediaItem: SmartMediaItem,
    resourceRepository: ResourceRepository
): EpgChannel(
    Flavor().pickBasedOnFlavor(Flavor().getImageOfType(smartMediaItem, ImageType.DARK),
        resourceRepository.getDimensionPixelSize(R.dimen.epg_view_channel_image_width),
        resourceRepository.getDimensionPixelSize(R.dimen.epg_view_channel_image_height),
        SmartImages.UNRESTRICTED),
    smartMediaItem.channelName ?: "",
    Flavor().getChannelId(smartMediaItem),
    null
) {

    private val loadedRanges = ConcurrentLinkedQueue<EpgRange>()

    fun clearLoadedRanges() {
        //Log.d("maxiEpg", "channel ${smartMediaItem.channelNumber} clearLoadedRanges")
        loadedRanges.clear()
    }

    fun getLoadedRanges(from: Long, to: Long) : List<EpgRange> {
        val tmpRange = EpgRange(from, to).apply { status = Status.EMPTY }
        return loadedRanges.filter { isTimeOverlapping(it, tmpRange) }
    }

    fun getRangeLeftToBeLoaded(from: Long, to: Long) : EpgRange {
        // if only one page can be loaded for testing purposes -> uncomment this line
//        if (loadedRanges.any { it.status == Status.LOADED }) return EpgRange(0, 0)
        val tmpRange = EpgRange(from, to).apply { status = Status.EMPTY }
        val rltbl = loadedRanges.firstOrNull { isTimeOverlappingNoTouch(it, tmpRange) }
            ?.let { cutLoadedPartOfRange(tmpRange, it) }
            ?: tmpRange
        //Log.d("maxiEpg", "${print2(from, to)} leftToBeLoaded ${rltbl.getWidth() > 0}")
        return rltbl
    }

    private fun print2(from: Long, to: Long): String =
        "channel:${smartMediaItem.channelNumber}, ${print1(from, to)}"

    override fun isLoading(fromTime: Long, toTime: Long): Boolean {
//        return false
        //this return false just make it disappear always
        val visibleRange = EpgRange(fromTime, toTime)
        val isLoading = loadedRanges.any { it.status == Status.LOADING && isTimeOverlapping(it, visibleRange)}
        //Log.d("maxiEpg", "${print2(fromTime, toTime)} isLoading:$isLoading")
        return isLoading
    }

    fun markLoadingRange(from: Long, to: Long) {
        //Log.d("maxiEpg", "${print2(from, to)} markLoadingRange")
        val newRange = EpgRange(from, to).apply { status = Status.LOADING }
        loadedRanges.add(newRange)
    }

    fun markLoadedRange(from: Long, to: Long) {
        //Log.d("maxiEpg", "${print2(from, to)} markLoadedRange")
        val loadedRange = EpgRange(from, to).apply { status = Status.LOADED }
        val loadingRanges = loadedRanges.filter { it.status == Status.LOADING }
        loadingRanges.forEach {
            if (loadedRange != it && isTimeOverlappingNoTouch(it, loadedRange)) {
                it.status = Status.LOADED
                mergeRanges(it)
            }
        }
    }

    private fun mergeRanges(modifiedRange: EpgRange) {
        loadedRanges.forEach {
            if (modifiedRange != it
                && isTimeOverlapping(it, modifiedRange)
                && modifiedRange.status == it.status) {
                mergeOverlapping(it, modifiedRange)
                loadedRanges.remove(modifiedRange)
                // this merge could trigger further merges
                mergeRanges(it)
                return
            }
        }
    }

    private fun mergeOverlapping(epgRange: EpgRange, epgRangeToMerge: EpgRange) {
        epgRange.from = min(epgRange.from, epgRangeToMerge.from)
        epgRange.to = max(epgRange.to, epgRangeToMerge.to)
    }

    class EpgRange(
        var from: Long = 0,
        var to: Long = 0) {
        var status: Status = Status.EMPTY
        fun getWidth(): Long = to - from
    }

    enum class Status {
        EMPTY,
        LOADING,
        LOADED
    }

    companion object {
        internal fun print1(from: Long, to: Long): String =
            "from:${DateTime(from).let{"${it.hourOfDay.to2()}:${it.minuteOfHour.to2()}"}}, to:${DateTime(to).let{"${it.hourOfDay.to2()}:${it.minuteOfHour.to2()}"}}"

        private fun Int.to2() =
            toString().let { if (it.length > 1) it else "0$it" }

        fun isTimeOverlapping(epgRange1: EpgRange, epgRange2: EpgRange): Boolean {
            return (epgRange1.from <= epgRange2.from && epgRange1.to >= epgRange2.from)
                    || (epgRange2.from <= epgRange1.from && epgRange2.to >= epgRange1.from)
        }
        fun isTimeOverlappingNoTouch(epgRange1: EpgRange, epgRange2: EpgRange): Boolean {
            return (epgRange1.from <= epgRange2.from && epgRange1.to > epgRange2.from)
                    || (epgRange2.from <= epgRange1.from && epgRange2.to > epgRange1.from)
        }

        /**
         * Warning, this function assumes, that ranges are overlapping
         */
        fun cutLoadedPartOfRange(newRange: EpgRange, loadedRange: EpgRange): EpgRange {
            if (newRange.from >= loadedRange.from && newRange.to <= loadedRange.to) {
                return EpgRange(0, 0)
            }
            if (newRange.from <= loadedRange.from) {
                newRange.to =  loadedRange.from
            }
            if (newRange.to >= loadedRange.to) {
                newRange.from = loadedRange.to
            }
            return newRange
        }
    }
}