package com.twentyfouri.tvlauncher.data

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.util.Log
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Observer
import com.twentyfouri.smartmodel.model.dashboard.SmartPageSection
import com.twentyfouri.tvlauncher.common.extensions.ifNotNull
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PlaylistType
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import timber.log.Timber

class AppListRepository (
    private val appListDao: AppListDao
): BroadcastReceiver() {

    private var serverListOfAppsNamesToBeShownAsRecommended : List<String>? = null

    private val playStoreLD = MutableLiveData<List<AppListItem>>()
    private val favoriteLD = MutableLiveData<List<AppListItem>>()
    private val appsLD = MutableLiveData<List<AppListItem>>()
    private val gamesLD = MutableLiveData<List<AppListItem>>()
    private val recommendedLD = MutableLiveData<List<AppListItem>>()

    private fun getAppList() : List<AppListItem> = appListDao.getAppList()

    private fun getFavorite(recommendedApps: List<String>?) : List<AppListItem> = appListDao.getFavoriteAppList(recommendedApps)

    private fun getGameList() : List<AppListItem> = appListDao.getGameList()

    private fun getRecommended(recommendedApps: List<String>?, reloadNeeded: Boolean = false) : List<AppListItem> = appListDao.getRecommendedAppList(recommendedApps, refreshNeeded = reloadNeeded)

//    fun getSettingsList(signalStrength: SignalStrength?, notificationCount: Int): ArraySet<AppListItem> = appListDao.getSettingsList(signalStrength, notificationCount)

    private fun getPlayStoreList() : List<AppListItem> = appListDao.getPlayStoreList()

    fun getFavoriteAppsSelectionList(): List<AppListItem> = appListDao.getFavoriteAppSelectionList(favorites = favoriteLD.value)

    fun registerSectionContentLDForUpdates(section: SmartPageSection, sectionDataUpdater: Observer<List<Any>>, serverListOfApps: List<String>? = null) {
        Timber.tag("Rows2").d("AppListRepository registerSectionContentLDForUpdates ... ${section.id}")
        serverListOfApps.ifNotNull {
            serverListOfAppsNamesToBeShownAsRecommended = serverListOfApps
        }
        GlobalScope.launch(Dispatchers.Main) {
            when (Flavor().getPlaylistType(section)) {
                PlaylistType.RECOMMENDED_APPS -> recommendedLD.observeForever(sectionDataUpdater)
                PlaylistType.FAVORITE_APPS -> favoriteLD.observeForever(sectionDataUpdater)
                PlaylistType.PLAY_STORE -> playStoreLD.observeForever(sectionDataUpdater)
                PlaylistType.ALL_APPS -> appsLD.observeForever(sectionDataUpdater)
                PlaylistType.ALL_GAMES -> gamesLD.observeForever(sectionDataUpdater)
                else -> { /*do nothing*/ }
            }
        }
        //trigger the initial call for data
        onReceive(null, null)
    }

    private fun MutableLiveData<List<AppListItem>>.postValueIfChanged(newApps: List<AppListItem>) {
        if (value == null || value!!.size != newApps.size) {
            postValue(newApps)
            return
        }
        newApps.forEachIndexed { i, newApp ->
            if (newApp.label != value?.get(i)?.label) {
                postValue(newApps)
                return
            }
        }
    }

    override fun onReceive(context: Context?, intent: Intent?) {
        Timber.tag("Rows2").d("AppListRepository onReceive ... ${intent?.action}")
        recommendedLD.postValueIfChanged(getRecommended(serverListOfAppsNamesToBeShownAsRecommended, true))
        favoriteLD.postValueIfChanged(getFavorite(serverListOfAppsNamesToBeShownAsRecommended))
        playStoreLD.postValueIfChanged(getPlayStoreList())
        appsLD.postValueIfChanged(getAppList())
        gamesLD.postValueIfChanged(getGameList())
    }

    fun clearSection(section: SmartPageSection, sectionDataUpdater: Observer<List<Any>>) {
        GlobalScope.launch(Dispatchers.Main) {
            when (Flavor().getPlaylistType(section)) {
                PlaylistType.RECOMMENDED_APPS -> recommendedLD.removeObserver(sectionDataUpdater)
                PlaylistType.FAVORITE_APPS -> favoriteLD.removeObserver(sectionDataUpdater)
                PlaylistType.PLAY_STORE -> playStoreLD.removeObserver(sectionDataUpdater)
                PlaylistType.ALL_APPS -> appsLD.removeObserver(sectionDataUpdater)
                PlaylistType.ALL_GAMES -> gamesLD.removeObserver(sectionDataUpdater)
                else -> { /*do nothing*/ }
            }
        }
    }

    fun registerAsReceiverOn(context: Context) {
        Timber.d("registerAsReceiverOn ... $context")
        context.registerReceiver(this, IntentFilter().apply {
            addAction(Intent.ACTION_PACKAGE_ADDED)
            addAction(Intent.ACTION_PACKAGE_REMOVED)
            addDataScheme("package")
        })
    }

    fun unregisterAsReceiverOn(context: Context) {
        Timber.d("UNregisterAsReceiverOn ... $context")
        try {
            context.unregisterReceiver(this)
        } catch (e: IllegalArgumentException) {

        }
    }

    fun storeFavoriteApp(appItem: AppListItem, favoritePriority: Int) {
        appListDao.storeFavoriteApp(appItem, favoritePriority)
    }

    fun getFavoriteAppsFromDatabase():List<AppUsage> = appListDao.getFavoriteAppsFromDatabase()

    fun insertFavoriteApp(favoriteApps: List<AppUsage>){
        if (favoriteApps.isNotEmpty()){
                appListDao.insertFavoriteApp(favoriteApps)
            }

    }

    fun updateFavoriteAppsPriority(appList: List<Any>) {
        appListDao.updateFavoriteAppsPriority(appList)
    }
}