package com.twentyfouri.tvlauncher.adapters

import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.ListAdapter
import androidx.recyclerview.widget.RecyclerView
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.data.HardcodedPlaylistReference
import com.twentyfouri.tvlauncher.viewmodels.RowViewModel
import com.twentyfouri.tvlauncher.widgets.OfflineRow
import com.twentyfouri.tvlauncher.widgets.RowView

class RowAdapter(
    private val layoutStyle: Int
) : ListAdapter<RowViewModel, RowAdapterViewHolder>(RowViewModelDiffCallback()) {

    override fun getItemViewType(position: Int): Int = //MOCK_ROW_TYPE
        when (getItem(position).sectionStyle) {
            HardcodedPlaylistReference.Type.OFFLINE.name -> OFFLINE_ROW_TYPE
            else -> ROW_VIEW_TYPE
        }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RowAdapterViewHolder = when (viewType) {
        OFFLINE_ROW_TYPE -> onCreateOfflineViewHolder(parent)
        MOCK_ROW_TYPE -> onCreateMockViewHolder(parent)
        else -> onCreateRowViewViewHolder(parent)
    }

    private fun onCreateRowViewViewHolder(parent: ViewGroup): RowAdapterViewHolder {
        val rowView = LayoutInflater.from(parent.context).inflate(
            R.layout.row_view_for_viewholder,
            parent,
            false
        ) as RowView
        rowView.setLayoutStyle(layoutStyle)
        return RowViewViewHolder(rowView)
    }

    private fun onCreateOfflineViewHolder(parent: ViewGroup): RowAdapterViewHolder {
        val rowView = LayoutInflater.from(parent.context).inflate(
            R.layout.offline_row_for_viewholder,
            parent,
            false
        ) as OfflineRow
        return OfflineViewHolder(rowView)
    }

    private fun onCreateMockViewHolder(parent: ViewGroup): RowAdapterViewHolder {
        val rowView = LayoutInflater.from(parent.context).inflate(
                R.layout.row_mock_viewholder,
                parent,
                false
        )
        return MockRowViewHolder(rowView)
    }

    override fun onBindViewHolder(holder: RowAdapterViewHolder, position: Int) {
//        Log.d("Rows3", "bind ${(getItem(position).sectionViewModel.pageSection.label)}")
        holder.bind(getItem(position), position)
    }

    override fun onViewAttachedToWindow(holder: RowAdapterViewHolder) {
        super.onViewAttachedToWindow(holder)
//        Log.d("Rows3", "attach ${(holder.itemView as? RowView)?.binding?.sectionViewModel?.pageSection?.label}")
        holder.onAttached()
    }

    override fun onViewDetachedFromWindow(holder: RowAdapterViewHolder) {
//        Log.d("Rows3", "detach ${(holder.itemView as? RowView)?.binding?.sectionViewModel?.pageSection?.label}")
        super.onViewDetachedFromWindow(holder)
        holder.onDetached()
    }

    override fun onDetachedFromRecyclerView(recyclerView: RecyclerView) {
//        Log.d("Rows3", "detach RowAdapter")
        super.onDetachedFromRecyclerView(recyclerView)
        (0 until recyclerView.childCount).forEach { childInd ->
            (recyclerView.getChildAt(childInd) as? RowView)?.onRowAdapterDetached()
        }
    }

    fun isRowFirstVisible(position: Int): Boolean {
        return getFirstVisibleRowPosition() == position
    }

    fun getFirstVisibleRowPosition(): Int {
        for (pos in 0 until itemCount) {
            if (isRowVisible(pos)) return pos
        }
        return -1
    }

    fun isUsingFilters(): Boolean {
        if(itemCount <= 0) return false
        return getItem(0).isUsingFilters()
    }

    fun tryFocusFilters() {
        if(isUsingFilters()) {
            getItem(0).filterViewHolders.value?.getOrNull(0)?.view?.requestFocus()
        }
    }

    fun getFirstNotForceHiddenRowPosition(): Int {
        for (pos in 0 until itemCount) {
            val item = getItem(pos)
            if (item.forceHide.value == false && item.rowIsEmpty.value != true) return pos
        }
        return -1
    }

    fun customSubmitList(list: MutableList<RowViewModel>?) {
        list ?: return
        var start = NO_INDEX
        var count = 0

        list.forEachIndexed { index, item ->
            cachedRowIds.contains( item.index ).ifFalse {
                (start == -1).ifTrue { start = index }
                 count++
            }
        }
        super.submitList(list)
        if (start != NO_INDEX) notifyItemRangeInserted(start, count)
        cachedRowIds = list.map { it.index }
    }

    private var cachedRowIds: List<Int> = emptyList()

    override fun submitList(list: MutableList<RowViewModel>?) {
        throw IllegalStateException("customSubmitList shall be used")
    }

    /**
     * Returns if the given row is visible, based on the state of the
     * corresponding [RowViewModel.rowVisibility].
     */
    fun isRowVisible(position: Int): Boolean {
        if(position < 0 || position >= itemCount) return false
        return getItem(position).rowVisibility.value == View.VISIBLE
    }

    companion object {
        const val ROW_VIEW_TYPE = 0
        const val OFFLINE_ROW_TYPE = 1
        const val MOCK_ROW_TYPE = 2
        private const val NO_INDEX = -1
    }
}