package com.twentyfouri.tvlauncher

import android.content.Context
import android.graphics.drawable.Drawable
import android.net.Uri
import android.view.KeyEvent
import com.google.android.exoplayer2.C
import com.google.android.exoplayer2.DefaultLoadControl
import com.google.android.exoplayer2.drm.MediaDrmCallback
import com.google.android.exoplayer2.source.dash.DashMediaSource
import com.google.android.exoplayer2.text.CaptionStyleCompat
import com.google.android.exoplayer2.upstream.DefaultBandwidthMeter
import com.twentyfouri.androidcore.epg.model.EpgEvent
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.dashboard.*
import com.twentyfouri.smartmodel.model.media.*
import com.twentyfouri.smartmodel.model.menu.SmartMenu
import com.twentyfouri.smartmodel.model.menu.SmartMenuItem
import com.twentyfouri.smartmodel.model.payment.SmartPurchase
import com.twentyfouri.smartmodel.model.payment.SmartSubscription
import com.twentyfouri.smartmodel.model.payment.SmartSubscriptionReference
import com.twentyfouri.smartmodel.persistence.SSTResult
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.data.ExternalRowPlaylistReference
import com.twentyfouri.tvlauncher.data.HardcodedPageReference
import com.twentyfouri.tvlauncher.data.HardcodedPlaylistReference
import com.twentyfouri.tvlauncher.data.channel.ExternalRow
import com.twentyfouri.tvlauncher.delegates.AppChannelsDelegate
import com.twentyfouri.tvlauncher.delegates.AppChannelsDelegateImpl
import com.twentyfouri.tvlauncher.delegates.SearchDelegateImpl
import com.twentyfouri.tvlauncher.delegates.SearchableDelegate
import com.twentyfouri.tvlauncher.ui.BaseFragment
import com.twentyfouri.tvlauncher.ui.EpgFragment
import com.twentyfouri.tvlauncher.utils.KeyEventAction
import com.twentyfouri.tvlauncher.viewmodels.MediaRestrictionIconsViewModel
import com.twentyfouri.tvlauncher.ui.SubscriptionsFragment
import com.twentyfouri.tvlauncher.utils.StringsMapper
import com.twentyfouri.tvlauncher.viewmodels.FilterViewModel
import com.twentyfouri.tvlauncher.viewmodels.MetadataViewModel
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.first
import org.joda.time.DateTime
import org.joda.time.DateTimeConstants
import org.joda.time.Seconds
import java.util.*

abstract class FlavorBase {
    // TODO replace this in future with custom navigation flow diagram or something
    open val navigationEpgLiveSkipDetail: Boolean = false
    open val navigationHomeLiveSkipDetail: Boolean = false

    open fun getRecordingsPackageReference(): SmartSubscriptionReference? = null
    open fun showFavorites(): Boolean = false
    open fun isFavoriteRow(section: SmartPageSection): Boolean = false
    open fun isRecordingRow(section: SmartPageSection): Boolean = false
    open fun getSearchFilter(genre: SearchGenre): SmartPlaylistFilterReference? = null
    open fun getSearchSorting(genre: SearchGenre): SmartPlaylistSorting? = null
    open fun getChannelSorting(): SmartPlaylistSorting? = null
    abstract fun getPageReference(type: PageType, menus: List<SmartMenu>?): SmartPageReference
    abstract fun getPlaylistReference(type: PlaylistType): SmartPlaylistReference
    abstract fun getPlaylistReference(type: PlaylistType, externalReference: SmartMediaReference?): SmartPlaylistReference
    abstract fun getGenreFilterReference(title: String, type: PlaylistType? = null): SmartPlaylistFilterReference?
    abstract fun convertMediaReferenceToDetail(reference: SmartMediaReference): SmartMediaDetail
    open fun convertDetailToRecording(mediaDetail: SmartMediaDetail, autoRecording: Boolean): SmartMediaItem = mediaDetail
    open fun isAutoRecording(mediaDetail: SmartMediaDetail?) = false
    abstract fun getChannelId(mediaItem: SmartMediaItem): String
    abstract fun getImageOfType(mediaItem: SmartMediaItem, imageType: ImageType): SmartImages
    abstract fun getMenuType(menuType: String): MenuType
    open fun getPlaylistTypeForLocalization(sectionStyle: String): PlaylistType = when (sectionStyle) {
        HardcodedPlaylistReference.Type.RECOMMENDED_APPS.name -> PlaylistType.RECOMMENDED_APPS
        HardcodedPlaylistReference.Type.FAVORITE_APPS.name -> PlaylistType.FAVORITE_APPS
        HardcodedPlaylistReference.Type.ALL_APPS.name -> PlaylistType.ALL_APPS
        HardcodedPlaylistReference.Type.ALL_GAMES.name -> PlaylistType.ALL_GAMES
        HardcodedPlaylistReference.Type.PLAY_STORE.name -> PlaylistType.PLAY_STORE
        HardcodedPlaylistReference.Type.OFFLINE.name -> PlaylistType.OFFLINE
        else -> PlaylistType.UNKNOWN
    }
    open fun getPageSize(section: SmartPageSection) = defaultPageSize
    open val defaultPageSize = 15
    //if you want to e.g. artificially limit total amount of items or pages, here is the correct method to override
    open fun hasAnotherPage(section: SmartPageSection, previousPageTotalCount: Int = 0): Boolean = section.selectedOptions?.let { it.pagingOffset < it.totalCount } ?: true
    open fun getPlaylistTotalSizeMax(section: SmartPageSection): Int = Int.MAX_VALUE
    open fun getPlaylistType(section: SmartPageSection?): PlaylistType {
        return when (section?.playlistReference) {
            is ExternalRowPlaylistReference -> PlaylistType.EXTERNAL_CHANNEL
            else -> getPlaylistTypeForLocalization(section?.sectionStyle ?: "")
        }
    }
    open fun getDrmScheme(): UUID = C.WIDEVINE_UUID
    open fun getDrmProperties(license: SmartMediaStreamLicense): MutableMap<String, String> = mutableMapOf()
    open fun getDrmCallback(drmCallback: MediaDrmCallback?, drmToken: String, streamUrl: String): MediaDrmCallback? = drmCallback
    open fun isTunnelingEnabled(): Boolean = false
    open val getBufferForPlaybackMs: Int = DefaultLoadControl.DEFAULT_BUFFER_FOR_PLAYBACK_MS
    open val getReBufferForPlaybackMs: Int = DefaultLoadControl.DEFAULT_BUFFER_FOR_PLAYBACK_AFTER_REBUFFER_MS
    open val getLivePresentationDelayMs: Long = DashMediaSource.DEFAULT_LIVE_PRESENTATION_DELAY_MS
    open val getLivePresentationDelayOverrideManifest: Boolean = false
    open fun getLiveStreamUrl(stream: SmartMediaStream): String = stream.primaryUrl
    open fun getStartOverStreamUrl(stream: SmartMediaStream, eventStart: Long): String = stream.primaryUrl
    open val useDifferentUrlForStartOver = false
    open val startOverOffsetSize = 0 //in seconds (used in Entel as DVR window size)

    open fun getCatchupStreamUrl(stream: SmartMediaStream): String = stream.primaryUrl
    abstract fun getCatchupMediaStreamReference(detail: SmartMediaDetail, channel: SmartMediaItem?): SmartMediaStreamReference?
    open fun isChannelAdult(mediaItem: SmartMediaItem?): Boolean {
        return mediaItem?.restrictions?.find { it.type == SmartRestrictionType.PARENTAL_PIN } != null
    }
    abstract fun getSmartMediaUriParamsFromReference(reference: SmartMediaReference): Uri?
    abstract fun getSmartMediaReferenceFromUri(uri: Uri): SmartMediaReference?
    abstract fun getSmartMediaReferenceForId(id: String): SmartMediaReference
    open fun getMenuItemTitle(menuItem: SmartMenuItem): String = menuItem.id

    // function to override potential server values (or when server does not provide those data)
    open suspend fun getLauncherMenu(smartApi: FlowSmartApi?): List<SmartMenu>? = smartApi?.getMenus()?.first()
    open suspend fun getLauncherPage(smartApi: FlowSmartApi?, pageReference: SmartPageReference): SmartPage? = smartApi?.getPage(pageReference)?.first()
    open fun isAppsPage(pageReference: SmartPageReference?)
            = (pageReference as? HardcodedPageReference)?.type == HardcodedPageReference.Type.APPS || isAppsPageFromServer(pageReference)
    open fun isAppsPageFromServer(pageReference: SmartPageReference?) = false
    open fun isHomePage(pageReference: SmartPageReference?): Boolean = false

    // TODO this function should be part of generic Smart API
    abstract suspend fun getSubscriptions(smartApi: FlowSmartApi, reference: SmartMediaReference): List<SmartSubscription>
    abstract suspend fun subscribeProduct(smartApi: FlowSmartApi, productReference: SmartSubscriptionReference?, isAutoRenew: Boolean?, subscriptionProduct: SmartSubscription?) : SmartPurchase
    open fun getMediaRestrictionViewModel(): MediaRestrictionIconsViewModel = MediaRestrictionIconsViewModel()
    open fun canInfoButtonBeVisible() = true
    abstract suspend fun isLoggedIn(smartApi: FlowSmartApi): Boolean
    open fun allowOfflineIfLoginFails() = true
    open fun shouldHandleLongPress() = false
    open fun shouldHandleOkOnPlayerControls() = false
    open val shouldDisableHorizontalZapp = false
    open val shouldDisableVerticalZapp = false
    open val minimalDelayForHorizontalRepeat = 0L
    open val minimalDelayForVerticalRepeat = 0L
    open val isRecordingAllowed = false
    open val isLiveRecordingAllowed = false
    open val shouldLookForPlayableChannel = false
    open val shouldHomeButtonFocusHomeTab = false
    open val shouldEpgHandleNumberButtons = false
    open val shouldEpgCloseEpg = false
    open val shouldStopGoToLiveInPlayer = false
    open val shouldShowSubtitlesInfoInPlayer = false
    open val shouldLiveFocusLive = false
    open val shouldLiveOpenPlayer = false
    open val shouldPlayerHandleLiveButton = false
    open val shouldBackLongPressRestartLauncher = false
    open val backLongPressTimeInterval = 5000 .. 10000
    open val shouldEpgStoreFocusOnDetailOpen = false
    open val glideCacheExpirationInterval = 0L //if static value is set cache never expires
    open fun shouldHandleBackAsLast() = false
    open fun shouldHandleGuideButton() = false
    open suspend fun checkDeviceLocation(smartModelApi: FlowSmartApi) = Unit
    open fun getWatchLiveIcon(resourceRepository: ResourceRepository): Drawable? = null

    open fun getAppChannelsDelegate(): AppChannelsDelegate? {
        return AppChannelsDelegateImpl()
    }
    open val disableAddBannerToAppChannelRow: Boolean = true

    //In case some recommended app have new version with different package name but both of them are installed
    //then only NEW app should be part of recommended apps list, the OLD one should be moved to common apps list
    //Define replacements as Pair(OLD, NEW) app package name
    //Both package names still need to be part of "recommended_apps_list" array to cover cases when only one of the apps is installed
    open fun getReplacedRecommendedApps(context: Context): List<Pair<String, String>> = emptyList()

    open fun getSearchableDelegate(): SearchableDelegate? {
        return SearchDelegateImpl()
    }

    open fun getTimezoneID(): String? = null

    open suspend fun getSubscriptionChannels(smartApi: FlowSmartApi, subscription: SmartSubscription): List<SmartMediaItem> = emptyList()

    open val getEpgConfigValues: EpgFragment.EPGConfigValues = EPGConfigValuesDefault()

    //both pairs are NAME and LANGUAGE(isoCode) of track
    open var defaultAudioSelection: Pair<String?, String?> = Pair(null, null)
    open var defaultSubtitleSelection: Pair<String?, String?> = Pair("none", "off") // none and off means subtitles disabled
    open fun getSubtitlesStyle():CaptionStyleCompat = CaptionStyleCompat.DEFAULT

    //TODO this is temporary solution - will be removed once we start using EPG database caching
    open val defaultWindowStartTime: DateTime = TimeProvider.now()
    open val defaultControlsWindows: Int = com.twentyfouri.tvlauncher.data.EpgRepository.PLAYER_CONTROLS_EPG_WINDOW
    open fun getEpgDataFilteredSorted(epgData: MutableList<SmartMediaItem>, channelReference: SmartMediaReference): List<SmartMediaItem> = epgData

    open fun allowDisplayOfExternalRows() = false

    open val bookmarkForConcurrencyHeartbeatTick: Long  = -1 //no value
    open val bookmarkForConcurrencyHeartbeatInitDelay: Long = 0
    open val useBookmarksForConcurrency = false //bookmarks are used as heartbeat to detect concurrent streaming on Entel backend

    open val useBookmarks: Boolean = false
    open val bookmarkResetThresholdMillis: Int = -1

    open val startOverIconDisplayRule = StartOverIconDisplayRule.NEVER
    open val displayPlayerBehindEPG = false
    open fun getPlayerControlsChannelTitle(number: String, name: String): String = "$number $name"

    open val displayNotificationAccessWarning = BuildConfig.IS_APP.not()

    open val displayDiagnosticsMenuIcon = false

    @Suppress("FunctionName")
    open fun getHomepageChannel_SmartPageSection_AllChannels(): SmartPageSection? = null
    @Suppress("FunctionName")
    open fun getHomepageChannel_SmartPageSection_OnNow(): SmartPageSection? = null

    open fun getAllChannelsSectionStyle(): String? = null

    open val shouldStartStreamTimeout = false

    open fun compareRecordingWithEvent(recording: SmartMediaItem?, event: SmartMediaItem?) =
            recording?.liveEventReference == event?.reference

    open fun populateDatabase(smartApi: FlowSmartApi?): Flow<SSTResult<Int>>? = null

    open fun isCatchupStreamAvailable(item: SmartMediaItem): Boolean = true

    open val shouldPreventKeyRepeatOnLongPress = false
    abstract fun getActionForKeyEvent(event: KeyEvent): KeyEventAction

    open fun pickBasedOnFlavor(smartImages: SmartImages?, width: Int, height: Int, differenceProperty: SmartImages.DifferenceProperty): String? {
        return smartImages?.pick(width, height, differenceProperty)
    }

    open fun getSubscriptionFragment(channelItem: SmartMediaItem): BaseFragment? = SubscriptionsFragment.newInstance(channelItem)
    
    open val handleUpDownChannelInCatchup = false

    open val handleInPlayerMessageCatchupNotAvailableYet = false
    //when catchup is not available during this time we expect that it is still encoding on server side
    open val maxExpectedCatchupEncodingTimeMs = 30 * DateTimeConstants.MILLIS_PER_MINUTE

    open val restartAfterStandby = false

    open val stopPlayerInStandby = true

    open val resetEpgFragmentOnButtonClick = true

    open val adsDurationBeforeRecordingMillis = 0
    open val adsDurationAfterRecordingMillis = 0

    open fun getIsRequestedInChannelRow(whereMetadataAreDisplayed: MetadataViewModel.WhereMetadataAreDisplayed?): Boolean = false

    open fun getBookmarkProgress(positionInSeconds: Int, startDate: DateTime?, endDate: DateTime?, durationInSeconds: Int?): Int {
        if (startDate == null || endDate == null) return 0
        return (100 * positionInSeconds) / (Seconds.secondsBetween(startDate, endDate).seconds)
    }

    open fun getBookmarkReference(detail: SmartMediaDetail): SmartMediaReference? = null

    open fun shouldPostBookmark(event: SmartPlayerEvent, startBookmarkSent: Boolean, isLive: Boolean?): Boolean {
        return event.position > adsDurationBeforeRecordingMillis
    }

    open fun cleanupUpcomingPrograms(programs: List<Any>): List<Any> {
        return programs
    }

    open fun setupSmartApiAnalyticsListener(smartModelApi: FlowSmartApi, youboraAnalytics: YouboraAnalytics) = Unit

    open val shouldCheckNetworkConnectionQuality = false
    open val shouldCheckIfUpdateAvailable = false
    open fun checkNetworkConnectionQuality(context: Context) {}
    open fun isAppAllowedToAddRow(externalRow: ExternalRow, context: Context): Boolean = false
    open fun externalRowsHardcodedPosition(sections: MutableList<SmartPageSection>): Int = sections.size
    open fun getBannedMoveIntervalForSection(pageSection: SmartPageSection): Pair<Int, Int> = Pair(0,0)
    open fun getCustomBandwidthMeter(context: Context): DefaultBandwidthMeter? = null

    open val showPauseIconOnlyIfPausedByUser = false

    open val shouldDisplayMessageOnConnectionError = false

    open val playerNoDataString = StringsMapper.translate(StringsMapper.PLAYER_NO_DATA)

    open fun getFilterType(filterReference: SmartPlaylistFilterReference): FilterViewModel.Type = FilterViewModel.Type.NONE
    open fun getFilterDate(filterReference: SmartPlaylistFilterReference?): Int? = null
    open fun getFilterContent(filterReference: SmartPlaylistFilterReference?): SmartMediaType = SmartMediaType.UNKNOWN
    open fun getFilterContentTranslation(filterReference: SmartPlaylistFilterReference?, resourceRepository: ResourceRepository): String? = null
    open fun applyAdditionalFilters(selectedFilterRefs: List<SmartPlaylistFilterReference>): List<SmartPlaylistFilterReference> = selectedFilterRefs
    open fun getFormattedDateForGrid(originalDate: String): String = originalDate
    open val allowChannelCacheReset = false

    open fun getLogInfoFromSmartMediaItem(mediaItem: SmartMediaItem?): String = "not implemented"
    open fun getLogInfoFromEPGEvent(epgEvent: EpgEvent): String = "not implemented"
    open fun getLogInfoFromSmartMediaReference(ref: SmartMediaReference?): String = "not implemented"
}

enum class ImageType {
    LIGHT,
    DARK,
    OVERLAY
}

enum class PageType {
    MAIN,
    EPG,
    GRID,
    APPS
}

enum class MenuType {
    HOME,
    EPG,
    GRID,
    PERSONAL,
    APPS,
    NOTIFICATIONS,
    VOICE,
    SEARCH,
    SETTINGS,
    DIAGNOSTICS,
    UNKNOWN
}

enum class PlaylistType {
    MOVIES,
    FAVORITE_CHANNELS,
    ALL_CHANNELS,
    ALL_CHANNELS_PAGED,
    ON_NOW,
    RECOMMENDED_APPS,
    FAVORITE_APPS,
    RECENT_RECORDINGS,
    OFFLINE,
    ALL_APPS,
    ALL_GAMES,
    PLAY_STORE,
    UNKNOWN,
    UPCOMING_PROGRAMS,
    EXTERNAL_CHANNEL,
    DO_NOT_MISS_CHANNELS,
    CATCHUP_PROGRAMS,
    LAST_WATCHED_CHANNELS,
    MOST_WATCHED,
    LAST_WATCHED,
    RECORDINGS,
    GRID
}

enum class SearchGenre {
    LIVE_TV,
    CATCHUP,
    RECORDINGS,
    CHANNELS
}

enum class StartOverIconDisplayRule {
    NEVER,
    ANYTIME_BASED_ON_SEEKING_RULE,
    ONLY_PAST_BASED_ON_SEEKING_RULE,
    CATCHUP_BASED
}

class EPGConfigValuesDefault: EpgFragment.EPGConfigValues() {
    override val SCROLL_PAGE_SIZE = 5
    override val INACTIVITY_REFRESH_MS = 1000 * 60 * 30
    override val HORIZONTAL_PAGE_SIZE_MS = 1000 * 60 * 60 * 2
    override val VERTICAL_PAGE_SIZE_CHANNEL_COUNT = 8
    override val EPG_DAYS_INTO_PAST = 7
    override val EPG_DAYS_INTO_FUTURE = 6
    override val MILLISECONDS_IN_DAY = 24 * 60 * 60 * 1000
    override val REQUEST_FOCUS = "REQUEST_FOCUS"
}

