package com.twentyfouri.tvlauncher.delegates

import android.content.ComponentName
import android.content.Intent
import android.content.pm.ResolveInfo
import android.net.Uri
import android.os.Build
import android.util.Log
import com.twentyfouri.androidcore.utils.ImageSpecification
import com.twentyfouri.androidcore.utils.ResourceImageSpecification
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.media.SmartMediaDetail
import com.twentyfouri.smartmodel.phoenix.referencemodels.PhoenixMediaReference
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.utils.androidOsVersions
import com.twentyfouri.tvlauncher.common.utils.deviceOsVersion
import com.twentyfouri.tvlauncher.data.getPackageName
import timber.log.Timber
import java.net.URLEncoder

class AppChannelsDelegateImpl : AppChannelsDelegate() {

    companion object {
        val TAG: String = AppChannelsDelegateImpl::class.java.simpleName
        const val NETFLIX_PACKAGE_NAME = "com.netflix.ninja"
        const val NETFLIX_DEBUG_CHANNEL_ID = "1714541"
        const val NETFLIX_RELEASE_CHANNEL_ID = "330376"

        // For source types information @see: https://jira.24i.com/browse/PRJ1210ENT-131
        const val NETFLIX_SOURCE_TYPE_FROM_RECOMMENDED_APPS_ROW = "2"
        const val NETFLIX_SOURCE_TYPE_FROM_APPS_ROW = "3"
        const val NETFLIX_SOURCE_TYPE_FROM_EPG = "17"
        const val NETFLIX_SOURCE_TYPE_FROM_SOFT_ZAPPING = "18"

        const val NETFLIX_IID_FROM_RECOMMENDED_APPS_ROW = "908b263f"
        const val NETFLIX_IID_FROM_APPS_ROW = "bdb1dccb"
        const val NETFLIX_IID_FROM_EPG = "0eace196"
        const val NETFLIX_IID_FROM_SOFT_ZAPPING = "2319e22f"
        const val NETFLIX_IID_FROM_NUMBER_ZAPPING = "d27c39f9"
    }

    private val printLogs: Boolean = false

    /**
     * @return List of flavor specific app channels package names.
     */
    override fun getAppChannelIDs(): List<String> = listOf(NETFLIX_DEBUG_CHANNEL_ID, NETFLIX_RELEASE_CHANNEL_ID)

    /**
     * @param channelId: Name of the app channel app package name.
     * @param prevChID: is a string representing the channel number preceding the app channel.
     * @param nextChID: is a string representing the channel number following the app channel.
     * @return Intent: The intent of the corresponding app from recommended apps row.
     */
    override fun getAppChannelIntent(
        channelId: String,
        prevChID: String?,
        nextChID: String?,
        isNumberZapp: Boolean
    ): Intent? {
        if (printLogs) Timber.d("Getting app channel intent.")
        return when (channelId) {
            NETFLIX_DEBUG_CHANNEL_ID, NETFLIX_RELEASE_CHANNEL_ID -> buildNetflixIntentWithSourceTypePayload(
                sourceType = NETFLIX_SOURCE_TYPE_FROM_EPG,
                interactionId = if (isNumberZapp) NETFLIX_IID_FROM_NUMBER_ZAPPING else NETFLIX_IID_FROM_EPG,
                prevChID = prevChID,
                nextChID = nextChID
            )
            else -> null
        }
    }

    /**
     * @param reference: Reference of the app channel app package name.
     * @param prevChID: is a string representing the channel number preceding the app channel.
     * @param nextChID: is a string representing the channel number following the app channel.
     * @return Intent: The intent of the corresponding app from soft zapping.
     */
    override fun getSoftZappingAppChannelIntent(reference: SmartMediaReference, prevChID: String?, nextChID: String?): Intent? {
        if (printLogs) Timber.d("Getting soft zapping app channel intent.")
        return when (reference.toString()) {
            NETFLIX_DEBUG_CHANNEL_ID, NETFLIX_RELEASE_CHANNEL_ID -> buildNetflixIntentWithSourceTypePayload(
                sourceType = NETFLIX_SOURCE_TYPE_FROM_SOFT_ZAPPING,
                interactionId = NETFLIX_IID_FROM_SOFT_ZAPPING,
                prevChID = prevChID,
                nextChID = nextChID
            )
            else -> null
        }
    }

    /**
     * @param info: Param of class ResolveInfo needed to build the intent.
     * @return Intent: The intent of the corresponding app from apps row.
     */
    override fun getAppIntent(info: ResolveInfo): Intent {
        if (printLogs) Timber.d("Getting app intent.")
        return when (info.getPackageName()) {
            NETFLIX_PACKAGE_NAME -> buildNetflixIntent(
                sourceType = NETFLIX_SOURCE_TYPE_FROM_APPS_ROW,
                interactionId = NETFLIX_IID_FROM_APPS_ROW)
            else -> buildDefaultIntent(info)
        }
    }

    /**
     * @param info: Param of class ResolveInfo needed to build the intent.
     * @return Intent: The intent of the corresponding app from recommended apps row.
     */
    override fun getRecommendedAppIntent(info: ResolveInfo): Intent {
        if (printLogs) Timber.d("Getting recommended app channel intent.")
        return when (info.getPackageName()) {
            NETFLIX_PACKAGE_NAME -> buildNetflixIntent(
                sourceType = NETFLIX_SOURCE_TYPE_FROM_RECOMMENDED_APPS_ROW,
                interactionId = NETFLIX_IID_FROM_RECOMMENDED_APPS_ROW)
            else -> buildDefaultIntent(info)
        }
    }

    /**
     * @param info: Param of class ResolveInfo needed to build the intent.
     * @return Intent: The intent of the corresponding app from favorite apps row.
     */
    override fun getFavoriteAppIntent(info: ResolveInfo): Intent {
        if (printLogs) Timber.d("Getting favorite app channel intent.")
        return when (info.getPackageName()) {
            NETFLIX_PACKAGE_NAME -> buildNetflixIntent(
                sourceType = NETFLIX_SOURCE_TYPE_FROM_RECOMMENDED_APPS_ROW,
                interactionId = NETFLIX_IID_FROM_RECOMMENDED_APPS_ROW)
            else -> buildDefaultIntent(info)
        }
    }

    /**
     * @param sourceType: is a string representing the source type that we are going to send with the intent.
     * @param interactionId: used instead of sourceType in version Ninja 7 and later to more accurately reflect specific launch points in partner integrations
     * @param prevChID: is a string representing the channel number preceding the Netflix channel number. Must set to -1 if the Netflix channel is the first channel in the EPG.
     * @param nextChID: is a string representing the channel number following the Netflix channel number. Must set to -1 if the Netflix channel is the last channel in the EPG.
     * @return Intent: The intent of the Netflix EPG Channel.
     * @see "https://jira.24i.com/browse/PRJ1210ENT-131"
     */
    private fun buildNetflixIntentWithSourceTypePayload(sourceType: String?, interactionId: String?, prevChID: String?, nextChID: String?): Intent {
        if (printLogs) Timber.d("Building netflix epg channel intent.")
        if (deviceOsVersion.api > Build.VERSION_CODES.P) return buildNetflixIntent(sourceType,
            interactionId)
        require(!(sourceType.isNullOrEmpty() || prevChID.isNullOrEmpty() || nextChID.isNullOrEmpty())) {
            "To build a netflix channel intent a valid " +
                    "preceding and following channel number is necessary, also a valid source type."
        }
        // It is important to see that the content inside sourceTypePayload is url encoded. @see https://jira.24i.com/browse/PRJ1210ENT-379.
        val sourceTypePayload =
            URLEncoder.encode("chID=999&prevChID=$prevChID&nextChID=$nextChID&category=OTHER",
                "utf-8")
        val uri =
            Uri.parse("http://www.netflix.com/browse?source_type=$sourceType&source_type_payload=$sourceTypePayload")
        val intent = Intent(Intent.ACTION_VIEW, uri)
        intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
        intent.setPackage(NETFLIX_PACKAGE_NAME)
        return intent
    }

    /**
     * @param sourceType: is a string representing the source type that we are going to send with the intent.
     * @param interactionId: used instead of sourceType in version Ninja 7 and later to more accurately reflect specific launch points in partner integrations
     * @return Intent: The corresponding intent of the Netflix app icon in the apps row.
     * @see "https://jira.24i.com/browse/PRJ1210ENT-131"
     */
    private fun buildNetflixIntent(sourceType: String?, interactionId: String?): Intent {
        if (printLogs) Timber.d("Building netflix app channel intent.")
        val launchPoint = if (deviceOsVersion.api > Build.VERSION_CODES.P) {
            "iid=${interactionId}"
        } else {
            "source_type=$sourceType"
        }
        val uri = Uri.parse("http://www.netflix.com/browse?${launchPoint}")
        val intent = Intent(Intent.ACTION_VIEW, uri)
        intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
        intent.setPackage(NETFLIX_PACKAGE_NAME)
        return intent
    }

    /**
     * @return Intent: The corresponding intent that we will use to launch another app.
     * @param info: Param of class ResolveInfo needed to build the intent.
     */
    private fun buildDefaultIntent(info: ResolveInfo): Intent {
        if (printLogs) Timber.d("Building default app intent.")
        val intent = Intent.makeMainActivity(
            ComponentName(
                info.getPackageName(),
                info.activityInfo.name
            )
        )
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        return intent
    }

    /**
     * @param channel: The app channel (app channels don't have events) from which we want to create a SmartMediaReference.
     * @return SmartMediaReference: SmartMediaReference from the specified channel.
     */
    override fun getAppChannelMediaReference(channel: com.twentyfouri.androidcore.epg.model.EpgChannel): SmartMediaReference? {
        if (printLogs) Timber.d("Getting app channel media reference.")
        return PhoenixMediaReference.forEpg(channel.id)
    }

    /**
     * @param item: The corresponding app channel smart media item from which we want to get a smart media detail.
     * @param resourceRepository: Resource repository necessary to set values from resources to the params in the smart detail that we will return. For
     * example translated strings or images.
     * @return SmartMediaDetail: Smart media detail with information from the app channel parsed.
     */
    override fun getAppChannelMediaDetail(item: SmartMediaItem, resourceRepository: ResourceRepository): SmartMediaDetail? {
        if (printLogs) Timber.d("Getting app channel media detail.")
        return if (item.reference.toString() == NETFLIX_DEBUG_CHANNEL_ID || item.reference.toString() == NETFLIX_RELEASE_CHANNEL_ID) {
            // TODO remove specific case for Netflix. Description should came from backend.
            SmartMediaDetail(item.reference, SmartMediaType.UNKNOWN).apply {
                title = item.title
                description = resourceRepository.getString(R.string.netflix_channel_description)
                images = item.images
            }
        } else {
            SmartMediaDetail(item.reference, SmartMediaType.UNKNOWN).apply {
                title = item.title
                description = item.description
                images = item.images
            }
        }
    }

    /**
     * In some cases we want set a custom image for an app channel.
     * This method is a getter that returns an ImageSpecification of the app channel.
     * @param reference: Reference of the app channel.
     * @param resourceRepository: Resource repository to access the image that we want to return.
     * @return ImageSpecification with the image that we want to set for the app channel.
     */
    override fun getAppChannelSpecificImage(reference: SmartMediaReference, resourceRepository: ResourceRepository): ImageSpecification? {
        if (printLogs) Timber.d("Getting app channel specific image.")
        return when (reference.toString()) {
            NETFLIX_DEBUG_CHANNEL_ID, NETFLIX_RELEASE_CHANNEL_ID -> ResourceImageSpecification(R.drawable.netflix_logo)
            else -> null
        }
    }
}