package com.twentyfouri.tvlauncher

import android.app.Service
import android.content.Context
import android.graphics.Color
import android.graphics.drawable.Drawable
import android.net.Uri
import android.net.wifi.WifiManager
import android.view.KeyEvent
import android.view.KeyEvent.*
import com.google.android.exoplayer2.C
import com.google.android.exoplayer2.text.CaptionStyleCompat
import com.google.android.exoplayer2.upstream.DefaultBandwidthMeter
import com.twentyfouri.androidcore.epg.model.EpgEvent
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.dashboard.*
import com.twentyfouri.smartmodel.model.error.AppCannotRunException
import com.twentyfouri.smartmodel.model.media.SmartMediaDetail
import com.twentyfouri.smartmodel.model.media.SmartMediaStream
import com.twentyfouri.smartmodel.model.media.SmartMediaStreamReference
import com.twentyfouri.smartmodel.model.media.SmartPlayerEvent
import com.twentyfouri.smartmodel.model.menu.SmartMenu
import com.twentyfouri.smartmodel.model.menu.SmartMenuItem
import com.twentyfouri.smartmodel.model.payment.SmartSubscription
import com.twentyfouri.smartmodel.model.payment.SmartSubscriptionReference
import com.twentyfouri.smartmodel.model.user.SmartSessionState
import com.twentyfouri.smartmodel.persistence.SSTResult
import com.twentyfouri.smartmodel.phoenix.PhoenixFlowSmartApi
import com.twentyfouri.smartmodel.phoenix.PhoenixSavedSession
import com.twentyfouri.smartmodel.phoenix.mapper.SmartMediaMapper
import com.twentyfouri.smartmodel.phoenix.model.location.PhoenixDeviceLocationOptions
import com.twentyfouri.smartmodel.phoenix.referencemodels.*
import com.twentyfouri.smartmodel.phoenix.util.populateDatabase
import com.twentyfouri.smartmodel.util.last
import com.twentyfouri.tvlauncher.common.*
import com.twentyfouri.tvlauncher.common.Flavor.Companion.activeProfile
import com.twentyfouri.tvlauncher.common.Flavor.Companion.onnet
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.extensions.ifEqualTo
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.ui.SemaphoreState
import com.twentyfouri.tvlauncher.common.ui.messagedialog.*
import com.twentyfouri.tvlauncher.common.utils.*
import com.twentyfouri.tvlauncher.data.EpgEventExt
import com.twentyfouri.tvlauncher.data.HardcodedMenuMapper
import com.twentyfouri.tvlauncher.data.HardcodedPlaylistReference
import com.twentyfouri.tvlauncher.data.SpecialAppLists
import com.twentyfouri.tvlauncher.data.channel.ExternalRow
import com.twentyfouri.tvlauncher.ui.BaseFragment
import com.twentyfouri.tvlauncher.ui.EpgFragment
import com.twentyfouri.tvlauncher.ui.MainActivity
import com.twentyfouri.tvlauncher.ui.SubscriptionsGridFragment
import com.twentyfouri.tvlauncher.utils.KeyEventAction
import com.twentyfouri.tvlauncher.viewmodels.EntelRestrictionIconsViewModel
import com.twentyfouri.tvlauncher.viewmodels.MediaRestrictionIconsViewModel
import com.twentyfouri.tvlauncher.viewmodels.MetadataViewModel
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.first
import java.util.*
import timber.log.Timber

class Flavor : FlavorBase() {

    companion object {
        //TODO if there will be more filters ever available in the filtered playlist, this should be part of Flavor
        private const val GenreToFilterFilteredChannelsRow = "Basispakket"
        private const val GenreToFilterAdultChannels = "adulto"
    }

    override suspend fun isLoggedIn(smartApi: FlowSmartApi): Boolean {
        val loginResult = smartApi.restoreSession().last()
        activeProfile = (loginResult.savedSession as PhoenixSavedSession).activeProfile
        return when (loginResult.sessionState) {
            SmartSessionState.LOGGED_IN -> {
                Timber.tag("isLoggedIn").d("SessionState: LOGGED_IN")
                //check if profile type changed from last time
                val profileType = ProfileType.fromString(activeProfile?.customerProfileType)
                if (ProfileType.fromString(SharedPreferencesUtils.getLastUserType()) != profileType
                    || SharedPreferencesUtils.getLastUserType() == null
                ) {
                    if (Flavor().shouldCheckNetworkConnectionQuality && SharedPreferencesUtils.isFirstTimeDeviceUsage()) {
                        SharedPreferencesUtils.putShouldCheckNetworkQuality(true)
                        SharedPreferencesUtils.putNetworkQualityCheckReason(
                            NetworkConnectionState.NetworkQualityCheckReason.NEW_USER.ordinal
                        )
                    }
                    SharedPreferencesUtils.putLastUserType(profileType.value)
                    //clear EpgDatabase cache
                    com.twentyfouri.tvlauncher.common.Flavor().clearEpgData(ReflectionUtils.getApplicationContext(), smartApi )
                }

                true
            }
            else -> {
                Timber.tag("isLoggedIn").d("SessionState: ELSE")
                false
            }
        }
    }

    override fun populateDatabase(smartApi: FlowSmartApi?): Flow<SSTResult<Int>>? {
        return (smartApi as? PhoenixFlowSmartApi)?.ktSmartApi?.populateDatabase()
    }

    override val allowChannelCacheReset: Boolean = true

    override val handleUpDownChannelInCatchup = true

    override fun getMediaRestrictionViewModel(): MediaRestrictionIconsViewModel {
        return EntelRestrictionIconsViewModel()
    }

    override fun allowOfflineIfLoginFails() = true

    override fun getChannelSorting(): SmartPlaylistSorting? = SmartPlaylistSorting(
        PhoenixSortingReference("Channel number"),
        "Channel number"
    )

    override fun getPageReference(type: PageType, menus: List<SmartMenu>?) = EntelMenuMapper.findSmartPageReferenceInMenuById(
        menus = menus,
        menuItemId = when (type) {
            PageType.MAIN -> getHomePageId()
            PageType.EPG -> getEpgPageId()
            PageType.APPS -> getAppsPageId()
            PageType.GRID -> "0" //TODO later
        }
    ) ?: PhoenixPageReference(type.toString())

    override suspend fun getLauncherPage(
        smartApi: FlowSmartApi?,
        pageReference: SmartPageReference
    ): SmartPage? {
        val page = smartApi?.getPage(pageReference)?.first()
        if (pageReference is PhoenixPageReference) {
            when (pageReference.id) {
                getEpgPageId() -> { }
                getAppsPageId() -> { }
                getHomePageId() -> {
                    val sections = page?.sections?.toMutableList() ?: emptyList<SmartPageSection>().toMutableList()
                    sections.find { (it.playlistReference as? PhoenixPlaylistReference)?.kind == PhoenixPlaylistReference.KIND_RECOMMENDED_APPS  }
                        ?.apply { sectionStyle = HardcodedPlaylistReference.Type.FAVORITE_APPS.name }
                    page?.sections = sections//.take(2).drop(1)
                }
            }
        }
        return page
    }

    @Suppress("ConstantConditionIf")
    private fun getHomePageId() = if (BuildConfig.IS_STAGING) STG_HOME_PAGE_ID else PROD_HOME_PAGE_ID

    @Suppress("ConstantConditionIf")
    private fun getEpgPageId() = if (BuildConfig.IS_STAGING) STG_EPG_PAGE_ID else PROD_EPG_PAGE_ID

    @Suppress("ConstantConditionIf")
    private fun getAppsPageId() = if (BuildConfig.IS_STAGING) STG_APPS_PAGE_ID else PROD_APPS_PAGE_ID

    override fun getPlaylistReference(type: PlaylistType, externalReference: SmartMediaReference?) = when (type) {
        PlaylistType.RECOMMENDED_APPS -> HardcodedPlaylistReference(HardcodedPlaylistReference.Type.RECOMMENDED_APPS)
        PlaylistType.UPCOMING_PROGRAMS -> (externalReference as? PhoenixMediaReference)?.let { PhoenixPlaylistReference.fromUpNext(externalReference) }
            ?: throw IllegalArgumentException("Missing external reference for upcoming")
//        PlaylistType.LAST_WATCHED_CHANNELS -> PhoenixPlaylistReference.fromLastWatched()
        else -> throw IllegalArgumentException("Incorrect playlist type")
    }

    override fun getPlaylistReference(type: PlaylistType) = getPlaylistReference(type, null)

    override fun getGenreFilterReference(title: String, type: PlaylistType?) =
        when(type) {
            PlaylistType.LAST_WATCHED_CHANNELS -> PhoenixGenreFilter("false", GenreToFilterAdultChannels)
            else -> PhoenixGenreFilter("TODO-genreId", GenreToFilterFilteredChannelsRow)
        }

    override fun convertMediaReferenceToDetail(reference: SmartMediaReference) = SmartMediaDetail(reference, SmartMediaType.MOVIE)

    override fun getChannelId(mediaItem: SmartMediaItem) = (mediaItem.reference as? PhoenixMediaReference)?.id ?: "-1"

    override fun getImageOfType(mediaItem: SmartMediaItem, imageType: ImageType): SmartImages = when (imageType) {
        ImageType.LIGHT,
        ImageType.DARK -> mediaItem.images
        ImageType.OVERLAY -> mediaItem.overlayImages?.getOrElse(SmartMediaMapper.KEY_CHANNEL_ICON_DARK) { SmartImages() } ?: SmartImages()
    }

    override fun getMenuType(menuType: String) = when (menuType) {
        HardcodedMenuMapper.Type.SEARCH.name -> MenuType.SEARCH
        HardcodedMenuMapper.Type.NOTIFICATIONS.name -> MenuType.NOTIFICATIONS
        HardcodedMenuMapper.Type.VOICE.name -> MenuType.VOICE
        HardcodedMenuMapper.Type.SETTINGS.name -> MenuType.SETTINGS
        HardcodedMenuMapper.Type.APPS.name -> MenuType.APPS
        HardcodedMenuMapper.Type.PERSONAL.name -> MenuType.PERSONAL
        HardcodedMenuMapper.Type.HOME.name -> MenuType.HOME
        HardcodedMenuMapper.Type.EPG.name -> MenuType.EPG
        HardcodedMenuMapper.Type.DIAGNOSTICS.name -> MenuType.DIAGNOSTICS
        else -> MenuType.UNKNOWN
    }

    override fun getPageSize(section: SmartPageSection): Int {
        return when(getPlaylistType(section)) {
            PlaylistType.LAST_WATCHED_CHANNELS -> 5
            else -> super.getPageSize(section)
        }
    }

    override fun getPlaylistType(section: SmartPageSection?): PlaylistType {
        when (val sectionStyleFromBase = super.getPlaylistType(section)) {
            PlaylistType.UNKNOWN -> { //super implementation was NOT able to identify the section style
                return when ((section?.playlistReference as? PhoenixPlaylistReference)?.kind) {
                    PhoenixPlaylistReference.KIND_ON_NOW_CHANNEL -> PlaylistType.ON_NOW
                    PhoenixPlaylistReference.KIND_EPG_CHANNELS_CHANNEL -> PlaylistType.ALL_CHANNELS_PAGED
                    PhoenixPlaylistReference.KIND_RECOMMENDED_APPS -> PlaylistType.FAVORITE_APPS
                    PhoenixPlaylistReference.KIND_UPNEXT -> PlaylistType.UPCOMING_PROGRAMS
                    PhoenixPlaylistReference.KIND_LAST_WATCHED -> PlaylistType.LAST_WATCHED_CHANNELS
                    PhoenixPlaylistReference.KIND_MOST_WATCHED -> PlaylistType.MOST_WATCHED
                    else -> PlaylistType.UNKNOWN
                }
            }
            else -> { //super implementation WAS able to identify the section style
                return sectionStyleFromBase
            }
        }
    }

    override fun getBannedMoveIntervalForSection(pageSection: SmartPageSection): Pair<Int, Int> {
        return when (getPlaylistType(pageSection)) {
            PlaylistType.FAVORITE_APPS -> Pair(1,1)
            else -> super.getBannedMoveIntervalForSection(pageSection)
        }
    }

    override fun getDrmScheme(): UUID {
        return C.WIDEVINE_UUID
    }

    override fun getCustomBandwidthMeter(context: Context): DefaultBandwidthMeter {
        return DefaultBandwidthMeter.Builder(context)
                .setInitialBitrateEstimate(Int.MAX_VALUE.toLong())
                .build()
    }

    override val getBufferForPlaybackMs = 800
// intentionally left commented, for those properties we should use default values for now (hotfix 7.20)
//    override val getReBufferForPlaybackMs = 3000
//    override val getLivePresentationDelayMs = 4000L
//    override val getLivePresentationDelayOverrideManifest = true

    override fun getLiveStreamUrl(stream: SmartMediaStream): String {
        //proper URL is selected by choosing correct xxxxx or xxxx_Offnet mediaFile from asset list response
        //!! there is cache so switch onnet/offnet will have no effect, clear data is needed
        return stream.primaryUrl
    }

    override val useDifferentUrlForStartOver = true
    override val startOverOffsetSize = 6600 //defined by Entel/Velocix, in fact it is DVR window size
    override fun getStartOverStreamUrl(stream: SmartMediaStream, eventStart: Long): String {
        val useOnnetStream = when (onnet) {
            OnNetState.ON_NET -> true
            OnNetState.OFF_NET -> false
            OnNetState.USER_TYPE_BASED -> {
                when (ProfileType.fromString(SharedPreferencesUtils.getLastUserType())) {
                    ProfileType.FTTH -> true
                    ProfileType.OTT -> false
                }
            }
        }

        return (
            if (useOnnetStream) {
                if (BuildConfig.IS_STAGING) stream.primaryUrl.replace("live1.lab.tvdevryd.com", "startover.lab.tvdevryd.com")
                else stream.primaryUrl.replace("live.entel.cl", "startover.entelchile.net")
            } else {
                if (BuildConfig.IS_STAGING) stream.primaryUrl.replace("ottlive.lab.tvdevryd.com", "ottstartover.lab.tvdevryd.com")
                else stream.primaryUrl.replace("ottlive.entelchile.net", "ottstartover.entelchile.net")
            }
        ) + "&offset=" + startOverOffsetSize
    }

    override fun getCatchupStreamUrl(stream: SmartMediaStream): String {
        //correct URL should be selected from getPlaybackContext
        //based on onnet type=DASH_STB_WV_FULL / type=DASH_STB_WV_LIGHT or type=DASH_STB_WV_FULL_Offnet / type=DASH_STB_WV_LIGHT_Offnet
        return stream.primaryUrl
    }

    override fun getCatchupMediaStreamReference(
        detail: SmartMediaDetail,
        channel: SmartMediaItem?
    ): SmartMediaStreamReference? {
        //we expect there are stream references either only from FILE_ID or only NOT from FILE_ID. Check method SmartMediaDetailMapper#convertEditions
        val streamReferenceWithSupposedFileIdOrigin =
            detail.editions.firstOrNull()?.streams?.firstOrNull() as? PhoenixMediaStreamReference
        streamReferenceWithSupposedFileIdOrigin?.fileIdOrigin.ifEqualTo(
            PhoenixMediaStreamReferenceFileIdOrigin.FILE_ID
        ) { return streamReferenceWithSupposedFileIdOrigin }
        detail.getSmartMediaStreamReference(PhoenixMediaStreamReferenceFileIdOrigin.MEDIA_ID)
            ?.also { return it }
        detail.getSmartMediaStreamReference(PhoenixMediaStreamReferenceFileIdOrigin.LINEAR_ASSET_ID)
            ?.also { return it }
        //PRJ1210ENT-5167 - This edition does not seem to be necessary. Uncomment if it is for some reason.
//        detail.getSmartMediaStreamReference(PhoenixMediaStreamReferenceFileIdOrigin.EPG_CHANNEL_ID)
//            ?.also { return it }
        return null
    }

    private fun SmartMediaDetail.getSmartMediaStreamReference(fileIdOrigin: PhoenixMediaStreamReferenceFileIdOrigin): SmartMediaStreamReference? {
        editions.forEach { ed ->
            ed.streams.forEach { st ->
                (st as? PhoenixMediaStreamReference)?.fileIdOrigin.ifEqualTo(fileIdOrigin) { return st }
            }
        }
        return null
    }

    override fun getMenuItemTitle(menuItem: SmartMenuItem): String = menuItem.label

    override fun getSmartMediaUriParamsFromReference(reference: SmartMediaReference): Uri? = null

    override fun getSmartMediaReferenceFromUri(uri: Uri): SmartMediaReference? = null

    override fun getSmartMediaReferenceForId(id: String): SmartMediaReference = PhoenixMediaReference.forEpg(id)

    //TODO use proper SmartModel api
    override suspend fun getSubscriptions(
        smartApi: FlowSmartApi,
        reference: SmartMediaReference
    ) = (smartApi as PhoenixFlowSmartApi).ktSmartApi.getAvailableSubscriptions(reference)

    //TODO use proper SmartModel api
    override suspend fun subscribeProduct(
        smartApi: FlowSmartApi,
        productReference: SmartSubscriptionReference?,
        isAutoRenew: Boolean?,
        subscriptionProduct: SmartSubscription?
    ) = (smartApi as PhoenixFlowSmartApi).ktSmartApi.subscribeProduct(subscriptionProduct)

    //TODO use proper SmartModel api
    override suspend fun getSubscriptionChannels(
        smartApi: FlowSmartApi,
        subscription: SmartSubscription): List<SmartMediaItem> = (smartApi as PhoenixFlowSmartApi).ktSmartApi.loadSubscriptionChannels(subscription)

    override fun canInfoButtonBeVisible() = false

    override fun shouldHandleLongPress() = true

    override fun shouldHandleOkOnPlayerControls() = true

    override fun shouldHandleBackAsLast() = BuildConfig.IS_APP.not()

    override fun shouldHandleGuideButton() = true

    override suspend fun checkDeviceLocation(smartModelApi: FlowSmartApi) {
        val options = PhoenixDeviceLocationOptions()
        options.gatewayMacAddress = getGatewayMacAddress()
        options.isInHomeNetwork = when (onnet) {
            OnNetState.ON_NET -> true
            OnNetState.OFF_NET -> false
            OnNetState.USER_TYPE_BASED -> {
                when (ProfileType.fromString(activeProfile?.customerProfileType)) {
                    ProfileType.FTTH -> true
                    ProfileType.OTT -> false
                }
            }
        }
        options.arpRecords = getArpRecords()
        smartModelApi.checkDeviceLocation(options).collect()
    }

    private fun getArpRecords(): List<Pair<String, String>> {
        val result: ArrayList<Pair<String, String>> = arrayListOf()
        Runtime.getRuntime().exec("ip neighbor").inputStream.bufferedReader().readText().lines().forEach { line ->
            line.split(" ").let { words ->
                val ip = words.find { word -> word.count { char -> char == '.' } == 3 }
                val mac = words.find { word -> word.count { char -> char == ':' } == 5 }
                if (!ip.isNullOrEmpty() && !mac.isNullOrEmpty()) result.add(Pair(ip, mac))
            }
        }
        return result
    }

    private fun getGatewayMacAddress(): String {
        try {
            //try to get gateway (default) IP from route list table
            val routeTable = Runtime.getRuntime().exec("ip route list match 0 table all scope global").inputStream.bufferedReader().readText()
            val ip = routeTable.split(" ").find { word -> word.count { char -> char == '.' } == 3 } //IP have three dots
                ?: throw AppCannotRunException()

            //try to find IP and read MAC in arp file
            Runtime.getRuntime().exec("ip neighbor").inputStream.bufferedReader().readText().lines().forEach { line ->
                line.split(" ").let { words ->
                    if (words.contains(ip)) {
                        words.find { word -> word.count { char -> char == ':' } == 5 }?.let { mac -> //MAC have five colons
                            //MAC found so return it
                            return mac
                        }
                    }
                }
            }
            //no mac found
            throw AppCannotRunException()
        } catch (e: Exception) {
            e.printStackTrace()
            throw AppCannotRunException()
        }
    }

    override fun getWatchLiveIcon(resourceRepository: ResourceRepository): Drawable? {
        return resourceRepository.getDrawable(R.drawable.ic_watch_live)
    }

    override fun isAppsPage(pageReference: SmartPageReference?) = super.isAppsPage(pageReference)
            || (pageReference as? PhoenixPageReference)?.id == PageType.APPS.name

    override fun isAppsPageFromServer(pageReference: SmartPageReference?) = (pageReference as? PhoenixPageReference)?.id == getAppsPageId()

    override fun isHomePage(pageReference: SmartPageReference?): Boolean {
        if (pageReference !is PhoenixPageReference) return false
        return getHomePageId() == pageReference.id
    }

    override val getEpgConfigValues: EpgFragment.EPGConfigValues = EPGConfigValuesEntel()

    override var defaultAudioSelection: Pair<String?, String?> = Pair("español", "es")
    override var defaultSubtitleSelection: Pair<String?, String?> = Pair("none", "off")
    override fun getSubtitlesStyle(): CaptionStyleCompat {
        return CaptionStyleCompat(
            Color.WHITE, //foreground color
            Color.TRANSPARENT, //background color
            Color.TRANSPARENT, //window color
            CaptionStyleCompat.EDGE_TYPE_OUTLINE, //edge type
            Color.BLACK, //edge color
            null //typeface
        )
    }

    override val bookmarkForConcurrencyHeartbeatTick: Long = 30000 //30 seconds in ms
    override val bookmarkForConcurrencyHeartbeatInitDelay: Long = 5000 //5 seconds in ms
    override val useBookmarksForConcurrency = true //bookmarks are used as heartbeat to detect concurrent streaming on Entel backend

    override fun shouldPostBookmark(event: SmartPlayerEvent, startBookmarkSent: Boolean, isLive: Boolean?): Boolean {
        return event.position > adsDurationBeforeRecordingMillis && startBookmarkSent
    }

    override fun allowDisplayOfExternalRows(): Boolean = true

    override fun isAppAllowedToAddRow(externalRow: ExternalRow, context: Context): Boolean {
        try {
            val allowedAppRowResources = SpecialAppLists.getAllowedAppRows(context)
            for (pos in 0 until allowedAppRowResources.length()) {
                val resId = allowedAppRowResources.getResourceId(pos, -1)
                if (resId < 0) continue
                val allowedAppEntryValues = context.resources.obtainTypedArray(allowedAppRowResources.getResourceId(pos, -1))
                if (allowedAppEntryValues.getString(0) == externalRow.packageName) {
                    for (allowedAppEntryValueIndex in 1 until allowedAppEntryValues.length()) {
                        if (allowedAppEntryValues.getString(allowedAppEntryValueIndex) == externalRow.displayName) {
                            allowedAppEntryValues.recycle()
                            return true
                        }
                    }
                }
            }
        } catch (e: Exception){
            e.printStackTrace()
        }
        return false
    }

    override val restartAfterStandby = true

    override val disableAddBannerToAppChannelRow: Boolean = false

    override fun externalRowsHardcodedPosition(sections: MutableList<SmartPageSection>): Int = sections.size - 1

    override fun getPlayerControlsChannelTitle(number: String, name: String): String = number

    override val navigationHomeLiveSkipDetail: Boolean = true

    override val displayDiagnosticsMenuIcon = true

    override val displayPlayerBehindEPG = true

    override val startOverIconDisplayRule = StartOverIconDisplayRule.ONLY_PAST_BASED_ON_SEEKING_RULE

    override val shouldLookForPlayableChannel = true

    override val shouldHomeButtonFocusHomeTab = true

    override val shouldEpgHandleNumberButtons = true

    override val shouldEpgCloseEpg = true

    override val shouldEpgStoreFocusOnDetailOpen: Boolean = true

    override val shouldLiveFocusLive = true

    override val shouldLiveOpenPlayer = true

    override val shouldPlayerHandleLiveButton = true

    override val shouldBackLongPressRestartLauncher = true

    override val resetEpgFragmentOnButtonClick = true

    override fun compareRecordingWithEvent(
        recording: SmartMediaItem?,
        event: SmartMediaItem?
    ): Boolean {
        return false
    }

    @Suppress("RedundantNullableReturnType")
    override fun getSearchSorting(genre: SearchGenre): SmartPlaylistSorting? {
        return SmartPlaylistSorting(
            reference = PhoenixSortingReference(""),
            label = "none"
        )
    }

    @Suppress("RedundantNullableReturnType")
    override fun getSubscriptionFragment(channelItem: SmartMediaItem): BaseFragment? {
        return SubscriptionsGridFragment.newInstance(channelItem)
    }

    override val shouldPreventKeyRepeatOnLongPress = true

    override fun getActionForKeyEvent(event: KeyEvent): KeyEventAction {
        return when (event.keyCode){
            KEYCODE_BACK,
            KEYCODE_BUTTON_B -> KeyEventAction.BACK
            KEYCODE_MEDIA_STOP -> KeyEventAction.STOP
            KEYCODE_MEDIA_PLAY_PAUSE -> KeyEventAction.PLAY_PAUSE
            KEYCODE_MEDIA_REWIND -> KeyEventAction.RW
            KEYCODE_MEDIA_FAST_FORWARD -> KeyEventAction.FF
            KEYCODE_CHANNEL_DOWN -> KeyEventAction.CHANNEL_DOWN
            KEYCODE_CHANNEL_UP -> KeyEventAction.CHANNEL_UP
            KEYCODE_DPAD_LEFT -> KeyEventAction.DPAD_LEFT
            KEYCODE_DPAD_RIGHT -> KeyEventAction.DPAD_RIGHT
            KEYCODE_DPAD_DOWN -> KeyEventAction.DPAD_DOWN
            KEYCODE_DPAD_UP -> KeyEventAction.DPAD_UP
            KEYCODE_DPAD_CENTER -> KeyEventAction.DPAD_CENTER
            in KEYCODE_NUMPAD_0..KEYCODE_NUMPAD_9,
            in KEYCODE_0..KEYCODE_9 -> KeyEventAction.NUMBER
            KEYCODE_LAST_CHANNEL -> KeyEventAction.LAST
            KEYCODE_F4 -> KeyEventAction.STARTOVER
            KEYCODE_INFO -> KeyEventAction.INFO
            KEYCODE_MEDIA_RECORD -> KeyEventAction.RECORD
            KEYCODE_TV -> KeyEventAction.LIVE
            KEYCODE_PROG_RED -> KeyEventAction.APPS
            KEYCODE_PROG_GREEN -> KeyEventAction.AUDIO_SUBTITLES
            KEYCODE_PROG_BLUE -> KeyEventAction.DIAGNOSIS

            else -> return KeyEventAction.NO_ACTION
        }
    }

    override fun getIsRequestedInChannelRow(whereMetadataAreDisplayed: MetadataViewModel.WhereMetadataAreDisplayed?): Boolean {
        return whereMetadataAreDisplayed == MetadataViewModel.WhereMetadataAreDisplayed.CHANNEL_ROW
    }

    override fun cleanupUpcomingPrograms(programs: List<Any>): List<Any> {
        return programs.drop(1) //first item needs to be deleted because it is current live event
    }

    override fun setupSmartApiAnalyticsListener(smartModelApi: FlowSmartApi, youboraAnalytics: YouboraAnalytics) {
        (smartModelApi as? PhoenixFlowSmartApi)?.ktSmartApi?.analyticsListener = EntelApiAnalyticsListener(youboraAnalytics)
    }

    override val shouldCheckNetworkConnectionQuality = true

    override val shouldCheckIfUpdateAvailable: Boolean = true

    override fun checkNetworkConnectionQuality(context: Context) {
        // We are validating network quality in case of:
        // - First installation.
        // - Profile change.
        // - Change in active network.
        if(!SharedPreferencesUtils.getShouldCheckNetworkQuality()) return
        val userType = ProfileType.fromString(SharedPreferencesUtils.getLastUserType())
        if (userType == ProfileType.OTT) {
            SharedPreferencesUtils.putShouldCheckNetworkQuality(false)
            return // OTT user is not restricted at all.
        }
        val networkInfo = NetworkInfo.getActiveConnectionInfo(context)
        if (networkInfo.type == InterfaceType.LAN) {
            if (SharedPreferencesUtils.getApprovedNetworks().isNullOrEmpty()) {
                SharedPreferencesUtils.putApprovedNetworks(setOf("LAN"))
            }
            SharedPreferencesUtils.putShouldCheckNetworkQuality(false)
            return // LAN connection is always fine.
        }
        if (NetworkInfo.isConnectionOK(networkInfo)) {
            SharedPreferencesUtils.putShouldCheckNetworkQuality(false)
            NetworkInfo.saveApprovedNetwork(context) // Add approved network to whitelist
            YouboraAnalytics.getInstance(context).reportYouboraNetworkConnection(context, networkInfo)
        } else {
            when (SharedPreferencesUtils.getNetworkQualityCheckReason()){
                NetworkConnectionState.NetworkQualityCheckReason.NEW_USER.ordinal -> {
                    resolveBadQualityForNewUser(networkInfo, context)
                }
                NetworkConnectionState.NetworkQualityCheckReason.NETWORK_CHANGE.ordinal -> {
                    SharedPreferencesUtils.getApprovedNetworks().isNullOrEmpty().ifTrue {
                        //still no approved network, react as the user is new.
                        resolveBadQualityForNewUser(networkInfo, context)
                        return
                    }
                    resolveBadQualityForNetworkChange(networkInfo, context)
                }
            }
        }
    }

    private fun resolveBadQualityForNewUser(networkInfo: ActiveConnectionInfo, context: Context) {
        showQualityBadDialog(
            context = context,
            is5GHz = (networkInfo as? ActiveConnectionInfoWifi)?.is5GHz == true,
            isNewUser = true
        )
    }

    private fun resolveBadQualityForNetworkChange(networkInfo: ActiveConnectionInfo, context: Context) {
        //currently any network quality is dismissible in case of network change.
        SharedPreferencesUtils.putShouldCheckNetworkQuality(false)
        //find out if active network is in whitelist
        val wifiInfo =
            (context.getSystemService(Service.WIFI_SERVICE) as? WifiManager)?.connectionInfo
        val networkWasApproved = SharedPreferencesUtils.getApprovedNetworks()
            ?.any { it.toIntOrNull() == wifiInfo?.networkId } ?: false
        if (networkWasApproved) {
            showQualityInfoDialog(SemaphoreState.ORANGE_WIFI, context)
        } else {
            showQualityBadDialog(
                context = context,
                is5GHz = (networkInfo as? ActiveConnectionInfoWifi)?.is5GHz == true,
                isNewUser = false
            )
        }
    }

    private fun showQualityInfoDialog(state: SemaphoreState, context: Context) {
        val messageDialogModel = MessageDialogModel(
            context.getString(R.string.quality_info_title),
            context.getString(R.string.quality_bad_message_approved_current_user),
            context.getString(R.string.quality_info_ok),
            MessageDialogCodes.connectionQualityInfo
        )
        messageDialogModel.semaphoreState = state
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        dialogFragment.mListener = object : MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean =
                answer is MessageDialogAction.Event && answer.keyEvent.keyCode in KEYCODE_0..KEYCODE_9
        }
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    private fun showQualityBadDialog(context: Context, is5GHz: Boolean, isNewUser: Boolean) {
        val message = if (isNewUser) {
            if (is5GHz) {
                context.getString(R.string.quality_bad_message_new_user_5GHz)
            } else {
                context.getString(R.string.quality_bad_message_new_user_2_4GHz)
            }
        } else {
            context.getString(R.string.quality_bad_message_banned_current_user)
        }
        val messageDialogModel = MessageDialogModel(
                context.getString(R.string.quality_bad_title),
                message,
                if (isNewUser) arrayOf(context.getString(R.string.quality_bad_retry)) else arrayOf(),
                if (isNewUser) null else context.getString(R.string.quality_info_ok),
                MessageDialogCodes.connectionQualityBad
        )
        messageDialogModel.semaphoreState = SemaphoreState.RED_WIFI
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        dialogFragment.mListener = object : MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean {
                if (answer is MessageDialogAction.Event && answer.keyEvent.keyCode == KEYCODE_BACK) {
                    //just to make dialog cancelable in DEBUG builds
                    return !BuildConfig.IS_DEBUG
                }
                if (answer is MessageDialogAction.Event) {
                    //consume key events to ban dialog dismiss
                    return true
                }
                if (answer !is MessageDialogAction.Result) {
                    return false
                }
                when(answer.type) {
                    OPTION_A -> {
                        //Requirement is to restart directly
                        (context as MainActivity).restartActivity()
                        // Retry only without restart - maybe used after Entel validation
                        //dialogFragment.dismiss()
                        //checkNetworkConnectionQuality(context)
                    }
                    else -> {}
                }
                return false
            }
        }
        dialogFragment.isCancelable = BuildConfig.IS_DEBUG
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    override val showPauseIconOnlyIfPausedByUser = true

    override val shouldDisplayMessageOnConnectionError = true
    override val playerNoDataString = ""

    override fun getLogInfoFromSmartMediaItem(mediaItem: SmartMediaItem?): String {
        return when (mediaItem?.type) {
            SmartMediaType.LIVE_EVENT -> {
                val ref = (mediaItem.reference as? PhoenixMediaReference)?.id ?: ""
                val time = "${mediaItem.startDate?.millis}-${mediaItem.endDate?.millis}"
                val ch = if(mediaItem.channelNumber <= 0) "x" else mediaItem.channelNumber
                "(T:$time ID:$ref) (CH:$ch)"
            }
            SmartMediaType.LIVE_CHANNEL -> {
                val ref = (mediaItem.reference as? PhoenixMediaReference)?.externalId ?: ""
                val ch = if(mediaItem.channelNumber <= 0) "x" else mediaItem.channelNumber
                "(ID:$ref) (CH:$ch)"
            }
            else -> ""
        }
    }

    override fun getLogInfoFromEPGEvent(epgEvent: EpgEvent): String {
        val ref = ((epgEvent as? EpgEventExt)?.smartMediaItem?.reference as? PhoenixMediaReference)?.id ?: ""
        val time = "${epgEvent.startTime}-${epgEvent.endTime}"
        return "T:$time ID:$ref"
    }

    override fun getLogInfoFromSmartMediaReference(ref: SmartMediaReference?): String {
        return (ref as? PhoenixMediaReference)?.id ?: ""
    }
}

class EPGConfigValuesEntel: EpgFragment.EPGConfigValues() {
    override val SCROLL_PAGE_SIZE = 6
    override val INACTIVITY_REFRESH_MS = 1000 * 60 * 30
    override val HORIZONTAL_PAGE_SIZE_MS = 1000 * 60 * 60 * 2
    override val VERTICAL_PAGE_SIZE_CHANNEL_COUNT = 8
    override val EPG_DAYS_INTO_PAST = 1
    override val EPG_DAYS_INTO_FUTURE = 5
    override val MILLISECONDS_IN_DAY = 24 * 60 * 60 * 1000
    override val REQUEST_FOCUS = "REQUEST_FOCUS"
}

